/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * oss.c
 * OSS audio output
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/soundcard.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/* Used to set buffer num and size */
#define FRAG_SPEC	0x0010000D

/*
 * Global data
 */

			/* Sound device */
static char		SoundDev[] = "/dev/dsp";
static char		MixerDev[] = "/dev/mixer";

			/* sound device file descriptor */
static int		fd_dsp = -2;

			/* Default fragment size and number */
static int		fsize = 16384;
static int		fcount = -1;

/*
 * Types
 */

struct priv_t
{
	long			bufferBPS;
	XMM_AudioFormat		daf;
};

/*
 * Filter info
 */

#define	OSS_AUDIO_FORMATS	6

static XMM_AudioFormat oss_af[OSS_AUDIO_FORMATS] =
{
    { XMM_AUDIO_FMT_S16LE, XMM_AUDIO_ANY_SRATE, 1, 2, XMM_AUDIO_ANY_BRATE, 0, 0, "16bit, signed, LE, mono" },
    { XMM_AUDIO_FMT_S16LE, XMM_AUDIO_ANY_SRATE, 2, 4, XMM_AUDIO_ANY_BRATE, 0, 0, "16bit, signed, LE, stereo" },
//    { XMM_AUDIO_FMT_S16LE, 44100, 1, 2, 88200, 0, 0, "16bit, signed, LE, mono" },
//    { XMM_AUDIO_FMT_S16LE, 44100, 2, 4, 44100 * 4, 0, 0, "16bit, signed, LE, stereo" },

//    { XMM_AUDIO_FMT_U8, XMM_AUDIO_ANY_SRATE, 1, 1, XMM_AUDIO_ANY_BRATE, 0, 0, "8bit, unsigned, mono" },
//    { XMM_AUDIO_FMT_U8, XMM_AUDIO_ANY_SRATE, 2, 2, XMM_AUDIO_ANY_BRATE, 0, 0, "8bit, unsigned, stereo" },
    { XMM_AUDIO_FMT_U8, 44100, 1, 1, 44100, 0, 0, "8bit, unsigned, mono" },
    { XMM_AUDIO_FMT_U8, 44100, 2, 2, 88200, 0, 0, "8bit, unsigned, stereo" },

    { XMM_AUDIO_FMT_ULAW, XMM_AUDIO_ANY_SRATE, 1, 1, XMM_AUDIO_ANY_BRATE, 0, 0, "u-law, mono" },
    { XMM_AUDIO_FMT_ULAW, XMM_AUDIO_ANY_SRATE, 2, 1, XMM_AUDIO_ANY_BRATE, 0, 0, "u-law, stereo" },
};

static XMM_FilterAudioInfo	oss_fai =
{
    XMM_FILTER_ACF_AUTOSELECT,
    "",					/* Filename. Will be initialized later */
    "OSS",				/* Name */
    "",					/* Description */
    "Copyright (c) 1999 Arthur Kleer",	/* Copyright */
    OSS_AUDIO_FORMATS,			/* Number of supported formats */
    oss_af				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Prototypes
 */

static int oss_GetVolume( void *xmm, uint32_t *volume );
static int oss_SetVolume( void *xmm, uint32_t volume );
static int oss_init( void *xmm, XMM_AudioFormat *aformat, int query );

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *oss_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  int			tmp, ret = XMM_RET_OK;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  if( oss_init( xmm, saf, flags & XMM_FILTER_AOF_QUERY ) != XMM_RET_OK )
	return (void *)ret;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(OSS) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* get block size */
  if( ioctl( fd_dsp, SNDCTL_DSP_GETBLKSIZE, &tmp ) == -1 )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Optaining DSP's block size: %s", sys_errlist[errno] );
	return NULL;
  }

  xmm_logging( 2, "OSS! Started ( %i Hz, %i channel(s), format %x, bs = %i )\n",
				saf->samprate, saf->channels, saf->format, tmp );

  /* Initialize data */
  priv->bufferBPS = (long)( saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8 );

  /* Set initialized format */
  priv->daf.format = saf->format;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE );
  priv->daf.blockSize = saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8;
  priv->daf.extraSize = 0;
  priv->daf.extraType = 0;
  priv->daf.desc[0] = '\0';
  
  /* return filter address */
  return pFilter;
}

/*
 * Free filter
 */
static int oss_Close( XMM_PluginFilterAudio *filter )
{
  filter->Control( filter, XMM_CTLSET_FLUSH, 0, NULL );

  if( fd_dsp >= 0 )
  {
	close( fd_dsp );
	fd_dsp = -2;
  }

  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int oss_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;
  audio_buf_info	abi;

  switch( cmd )
  {
	case XMM_CTLQUERY_AFORMAT:
		if( oss_init( filter->sys.xmm, (XMM_AudioFormat *)data, 1 ) == XMM_RET_OK )
			return XMM_CTLRET_TRUE;
		else	return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_DELAY:
		ioctl( fd_dsp, SNDCTL_DSP_GETOSPACE, &abi );
		*((double *)data) = ((double)abi.fragstotal * abi.fragsize - abi.bytes ) / priv->bufferBPS;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_MAX_DELAY:
		ioctl( fd_dsp, SNDCTL_DSP_GETOSPACE, &abi );
		*((double *)data) = (double) abi.fragstotal * abi.fragsize / priv->bufferBPS;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_TOTAL:
		ioctl( fd_dsp, SNDCTL_DSP_GETOSPACE, &abi );
		*((uint32_t *)data) = (uint32_t) abi.fragstotal * abi.fragsize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_FREE:
		ioctl( fd_dsp, SNDCTL_DSP_GETOSPACE, &abi );
		*((uint32_t *)data) = (uint32_t) abi.bytes;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_SIZE:
		ioctl( fd_dsp, SNDCTL_DSP_GETOSPACE, &abi );
		*((uint32_t *)data) = (uint32_t) abi.fragsize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_VOLUME:
		oss_GetVolume( filter->sys.xmm, (uint32_t *) data );
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data) = param;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &oss_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_VOLUME:
		oss_SetVolume( filter->sys.xmm, param );
		return XMM_CTLRET_TRUE;

	case XMM_CTLSET_FLUSH:
		ioctl( fd_dsp, SNDCTL_DSP_RESET, NULL );
		return XMM_CTLRET_TRUE;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(OSS) cmd = 0x%x", cmd );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(OSS) cmd ( 0x%x )", cmd );
}

/*
 * Process data
 */
static int oss_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  int			ret;

  ret = write( fd_dsp, src, isize );
  if( ret < 0 )	return xmm_SetError( filter->sys.xmm, XMM_RET_ERROR, "(OSS) Error on write ( %i: %s )\n", errno, strerror(errno));

  if( osize )	*osize = 0;	/* Nothing in dest */
  return ret;
}

/*
 * Process data (BQ)
 */
static int oss_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin info
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_AOUTPUT,
				XMM_VERSION_NUM,
				"",
				"OSS",
				"Sound: OSS",
				"Copyright (c) 1999 Arthur Kleer",
				NULL, NULL },
				oss_Open, oss_Close, oss_Control,
				oss_Process, oss_ProcessBQ };

/*
 * Internal code
 */

/* XMMP sound format --> OSS sound format */
struct format_conv
{
	int	xmm;
	int	oss;
};

static struct format_conv format_table_init[] =
{
	{ XMM_AUDIO_FMT_ULAW		, AFMT_MU_LAW },
	{ XMM_AUDIO_FMT_ALAW		, AFMT_A_LAW },
	{ XMM_AUDIO_CODEC_ADPCM_IMA	, AFMT_IMA_ADPCM },
	{ XMM_AUDIO_FMT_U8		, AFMT_U8 },
	{ XMM_AUDIO_FMT_S16LE		, AFMT_S16_LE },
	{ XMM_AUDIO_FMT_S16BE		, AFMT_S16_BE },
	{ XMM_AUDIO_FMT_S8		, AFMT_S8 },
	{ XMM_AUDIO_FMT_U16LE		, AFMT_U16_LE },
	{ XMM_AUDIO_FMT_U16BE		, AFMT_U16_BE },
	{ XMM_AUDIO_FMT_MPEG		, AFMT_MPEG },
	{ XMM_AUDIO_FMT_AC3		, AFMT_AC3 },
	{ 0, 0 }
};

/*
 * Initialize OSS
 */
static int oss_init( void *xmm, XMM_AudioFormat *saf, int query )
{
  int	tmp, xmm_fmt_idx, fmt, bits, oss_formats, fd;

  /* Check for multiple use */
  if( fd_dsp != -2 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) OSS plugin already in use." );

  /* Check device access */
  if( access( SoundDev, W_OK ) != 0 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Error accessing audio device <%s>: %s", SoundDev, sys_errlist[errno] );

  /* Try to open device */
  if(( fd = open( SoundDev, O_WRONLY | O_NONBLOCK )) < 0 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Opening audio device <%s>: %s", SoundDev, sys_errlist[errno] );

  close( fd );

  /* Open device */
  if(( fd_dsp = open( SoundDev, O_WRONLY )) < 0 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Opening audio device <%s>: %s", SoundDev, sys_errlist[errno] );

  /* get buffer size */
  tmp = FRAG_SPEC;

  if( fsize > 0 )
  {
	for( bits = -1; fsize; fsize >>= 1, bits++ );
	tmp = ( tmp & 0xFFFF0000 ) | bits;
  }

  if( fcount != -1 )
  {
	tmp = ( fcount << 16 ) | ( tmp & 0x0000FFFF );
  }

  /* set fragment size and count */
  if( ioctl( fd_dsp, SNDCTL_DSP_SETFRAGMENT, &tmp ) < 0 )
  {
	close( fd_dsp );
	fd_dsp = -2;
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Setting FRAGMENT to %i: %s", FRAG_SPEC, sys_errlist[errno] );
  }

  ioctl( fd_dsp, SNDCTL_DSP_RESET, NULL );

  /* get supported formats */
  if( ioctl( fd_dsp, SNDCTL_DSP_GETFMTS, &oss_formats ) < 0 )
  {
	close( fd_dsp );
	fd_dsp = -2;
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Getting supported formats" );
  }

#ifdef VERBOSE
  xmm_logging( 1, "OSS! INFO: formats (0x%x):", oss_formats );
  if( oss_formats & AFMT_MU_LAW	)	xmm_logging( 1, " u-law" );
  if( oss_formats & AFMT_A_LAW )	xmm_logging( 1, " a-law" );
#ifdef AFMT_IMA_ADPCM
  if( oss_formats & AFMT_IMA_ADPCM )	xmm_logging( 1, " IMA ADPCM" );
#endif
  if( oss_formats & AFMT_U8 )		xmm_logging( 1, " U8" );
  if( oss_formats & AFMT_S16_LE )	xmm_logging( 1, " S16LE" );
  if( oss_formats & AFMT_S16_BE )	xmm_logging( 1, " S16BE" );
  if( oss_formats & AFMT_S8 )		xmm_logging( 1, " S8" );
  if( oss_formats & AFMT_U16_LE )	xmm_logging( 1, " U16LE" );
  if( oss_formats & AFMT_U16_BE )	xmm_logging( 1, " U16BE" );
#ifdef AFMT_MPEG
  if( oss_formats & AFMT_MPEG )		xmm_logging( 1, " MPEG" );
#endif
#ifdef AFMT_AC3
  if( oss_formats & AFMT_AC3 )		xmm_logging( 1, " AC3" );
#endif
  xmm_logging( 1, "\n" );
#endif

  /* Find XMM format */
  for( xmm_fmt_idx = 0; format_table_init[xmm_fmt_idx].xmm; xmm_fmt_idx++ )
	if( saf->format == format_table_init[xmm_fmt_idx].xmm )
		break;

  if( format_table_init[xmm_fmt_idx].xmm == 0 )
  {
	close( fd_dsp );
	fd_dsp = -2;
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Unknown/Unsupported XMM format 0x%x", saf->format );
  }

  /* Check if format supported by OSS */
  if(( oss_formats & format_table_init[xmm_fmt_idx].oss ) == 0 )
  {
	close( fd_dsp );
	fd_dsp = -2;
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) OSS format ( 0x%x ) not supported [ needed for XMM format 0x%x ]", format_table_init[xmm_fmt_idx].oss, format_table_init[xmm_fmt_idx].xmm );
  }

  /* Set format */
  tmp = fmt = format_table_init[xmm_fmt_idx].oss;
  if( ioctl( fd_dsp, SNDCTL_DSP_SETFMT, &tmp ) < 0 )
  {
	close( fd_dsp );
	fd_dsp = -2;
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Setting DSP to format %u: %s", (unsigned) tmp, sys_errlist[errno] );
  }

  if( tmp != fmt )
  {
	close( fd_dsp );
	fd_dsp = -2;
	return xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, "(OSS) OSS driver changed format ( 0x%x --> 0x%x )", fmt, tmp );
  }

  /* Channels */
  tmp = saf->channels;
  if( ioctl( fd_dsp, SNDCTL_DSP_CHANNELS, &tmp ) < 0 )
  {
	close( fd_dsp );
	fd_dsp = -2;
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Unable to set DSP to %s mode: %s", saf->channels == 2 ? "Stereo" : "Mono", sys_errlist[errno] );
  }

  if( tmp != saf->channels )
  {
	close( fd_dsp );
	fd_dsp = -2;
	return xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, "(OSS) OSS driver changed channels number ( 0x%x --> 0x%x )", saf->channels, tmp );
  }

  /* Sample rate */
  tmp = saf->samprate;
  if( ioctl( fd_dsp, SNDCTL_DSP_SPEED, &tmp ) < 0 )
  {
	close( fd_dsp );
	fd_dsp = -2;
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Unable to set audio sampling rate to %i: %s", saf->samprate, sys_errlist[errno] );
  }

  if( tmp != saf->samprate )
  {
	close( fd_dsp );
	fd_dsp = -2;
	return xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, "(OSS) OSS driver changed sample rate number ( 0x%x --> 0x%x )", saf->samprate, tmp );
  }

  /* close device - only in query mode */
  if( query )
  {
	close( fd_dsp );
	fd_dsp = -2;
  }

  /* return */
  return XMM_RET_OK;
}


/* * * * * * * * *
 * * * Mixer * * *
 * * * * * * * * */

/*
 * Get volume
 */
static int oss_GetVolume( void *xmm, uint32_t *volume )
{
  int fd, value, supported;

  if(( fd = open( MixerDev , O_RDWR )) == -1 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Mixer: Opening mixer for read: %s", sys_errlist[errno] );

  if( ioctl( fd, SOUND_MIXER_READ_DEVMASK, &supported ) < 0 )
	supported = 0xffff;	/* All supported ??? */

  if(( supported & SOUND_MASK_PCM ) == 0 )
  {
	close( fd );
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Mixer: Volume read not supported" );
  }

  if( ioctl( fd, SOUND_MIXER_READ_PCM, &value ) < 0 )
  {
	close( fd );
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Mixer: Unable to read volume" );
  }

  *volume = ( value & 0x00FF ) | ((( value & 0xFF00 ) << 8 ));

  close( fd );
  return 0;
}

/*
 * Set volume
 */
static int oss_SetVolume( void *xmm, uint32_t volume )
{
  int fd, value, supported;

  if(( fd = open( MixerDev , O_RDWR )) == -1 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Mixer: Opening mixer for write: %s", sys_errlist[errno] );

  if( ioctl( fd, SOUND_MIXER_READ_DEVMASK, &supported ) < 0 )
	supported = 0xffff;	/* All supported ??? */

  if(( supported & SOUND_MASK_PCM ) == 0 )
  {
	close( fd );
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Mixer: Volume write not supported" );
  }

  value = (( volume & 0xFFFF0000 ) >> 8 ) | ( volume & 0x0000FFFF );
  if( ioctl( fd, SOUND_MIXER_WRITE_PCM, &value ) < 0 )
  {
	close( fd );
	return xmm_SetError( xmm, XMM_RET_ERROR, "(OSS) Mixer: Unable to write volume" );
  }

  close( fd );
  return 0;
}
