/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * vorbis.c
 * Ogg/Vorbis audio output ( writer )
 */

#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>
#include <vorbis/vorbisenc.h>

/*
 * Definitions
 */

#define	PCM_BUFFER_SIZE		16384

/*
 * Global data
 */

			/* Default fragment size and number */
static int		fsize = 16384;

/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_AudioFormat		daf;

    /* Ogg/Vorbis encoder stuff */
    ogg_stream_state		os;
    ogg_page			og;
    ogg_packet			op;
    vorbis_info			vi;
    vorbis_comment		vc;
    vorbis_dsp_state		vd;
    vorbis_block		vb;

    /* Buffer stuff */
    double			bufferDelay;
    long			bufferSize;
    long			bufferSSize;
};

/*
 * Filter info
 */

#define	VORBIS_AUDIO_FORMATS	50

static XMM_AudioFormat vorbis_af[VORBIS_AUDIO_FORMATS] =
{
    /*
     * Stereo
     */

    { XMM_AUDIO_CODEC_VORBIS, 48000, 2, 1, 320 * 1000, 0, 0, "320 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 48000, 2, 1, 256 * 1000, 0, 0, "256 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 48000, 2, 1, 224 * 1000, 0, 0, "224 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 48000, 2, 1, 192 * 1000, 0, 0, "192 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 48000, 2, 1, 160 * 1000, 0, 0, "160 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 48000, 2, 1, 128 * 1000, 0, 0, "128 kBit/s, 48000Hz, Stereo" },

    { XMM_AUDIO_CODEC_VORBIS, 44100, 2, 1, 320 * 1000, 0, 0, "320 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 44100, 2, 1, 256 * 1000, 0, 0, "256 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 44100, 2, 1, 224 * 1000, 0, 0, "224 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 44100, 2, 1, 192 * 1000, 0, 0, "192 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 44100, 2, 1, 160 * 1000, 0, 0, "160 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 44100, 2, 1, 128 * 1000, 0, 0, "128 kBit/s, 44100Hz, Stereo" },

    { XMM_AUDIO_CODEC_VORBIS, 32000, 2, 1, 224 * 1000, 0, 0, "224 kBit/s, 32000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 32000, 2, 1, 192 * 1000, 0, 0, "192 kBit/s, 32000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 32000, 2, 1, 160 * 1000, 0, 0, "160 kBit/s, 32000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 32000, 2, 1, 128 * 1000, 0, 0, "128 kBit/s, 32000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 32000, 2, 1, 112 * 1000, 0, 0, "112 kBit/s, 32000Hz, Stereo" },

    { XMM_AUDIO_CODEC_VORBIS, 24000, 2, 1, 160 * 1000, 0, 0, "160 kBit/s, 24000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 24000, 2, 1, 128 * 1000, 0, 0, "128 kBit/s, 24000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 24000, 2, 1, 112 * 1000, 0, 0, "112 kBit/s, 24000Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 24000, 2, 1,  96 * 1000, 0, 0, " 96 kBit/s, 24000Hz, Stereo" },

    { XMM_AUDIO_CODEC_VORBIS, 22050, 2, 1, 160 * 1000, 0, 0, "160 kBit/s, 22050Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 22050, 2, 1, 128 * 1000, 0, 0, "128 kBit/s, 22050Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 22050, 2, 1, 112 * 1000, 0, 0, "112 kBit/s, 22050Hz, Stereo" },
    { XMM_AUDIO_CODEC_VORBIS, 22050, 2, 1,  96 * 1000, 0, 0, " 96 kBit/s, 22050Hz, Stereo" },

    /*
     * Mono
     */

    { XMM_AUDIO_CODEC_VORBIS, 48000, 1, 1, 160 * 1000, 0, 0, "160 kBit/s, 48000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 48000, 1, 1, 128 * 1000, 0, 0, "128 kBit/s, 48000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 48000, 1, 1, 112 * 1000, 0, 0, "112 kBit/s, 48000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 48000, 1, 1,  96 * 1000, 0, 0, " 96 kBit/s, 48000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 48000, 1, 1,  80 * 1000, 0, 0, " 80 kBit/s, 48000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 48000, 1, 1,  64 * 1000, 0, 0, " 64 kBit/s, 48000Hz, Mono" },

    { XMM_AUDIO_CODEC_VORBIS, 44100, 1, 1, 160 * 1000, 0, 0, "160 kBit/s, 44100Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 44100, 1, 1, 128 * 1000, 0, 0, "128 kBit/s, 44100Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 44100, 1, 1, 112 * 1000, 0, 0, "112 kBit/s, 44100Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 44100, 1, 1,  96 * 1000, 0, 0, " 96 kBit/s, 44100Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 44100, 1, 1,  80 * 1000, 0, 0, " 80 kBit/s, 44100Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 44100, 1, 1,  64 * 1000, 0, 0, " 64 kBit/s, 44100Hz, Mono" },

    { XMM_AUDIO_CODEC_VORBIS, 32000, 1, 1, 112 * 1000, 0, 0, "112 kBit/s, 32000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 32000, 1, 1,  96 * 1000, 0, 0, " 96 kBit/s, 32000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 32000, 1, 1,  80 * 1000, 0, 0, " 80 kBit/s, 32000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 32000, 1, 1,  64 * 1000, 0, 0, " 64 kBit/s, 32000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 32000, 1, 1,  56 * 1000, 0, 0, " 56 kBit/s, 32000Hz, Mono" },

    { XMM_AUDIO_CODEC_VORBIS, 24000, 1, 1,  80 * 1000, 0, 0, " 80 kBit/s, 24000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 24000, 1, 1,  64 * 1000, 0, 0, " 64 kBit/s, 24000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 24000, 1, 1,  56 * 1000, 0, 0, " 56 kBit/s, 24000Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 24000, 1, 1,  48 * 1000, 0, 0, " 48 kBit/s, 24000Hz, Mono" },

    { XMM_AUDIO_CODEC_VORBIS, 22050, 1, 1,  80 * 1000, 0, 0, " 80 kBit/s, 22050Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 22050, 1, 1,  64 * 1000, 0, 0, " 64 kBit/s, 22050Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 22050, 1, 1,  56 * 1000, 0, 0, " 56 kBit/s, 22050Hz, Mono" },
    { XMM_AUDIO_CODEC_VORBIS, 22050, 1, 1,  48 * 1000, 0, 0, " 48 kBit/s, 22050Hz, Mono" },

};

static XMM_FilterAudioInfo	vorbis_fai =
{
    0,
    "",					/* Filename. Will be initialized later */
    "Ogg/Vorbis",			/* Name */
    "",					/* Description */
    "Copyright (c) 2000 Arthur Kleer",	/* Copyright */
    VORBIS_AUDIO_FORMATS,		/* Number of supported formats */
    vorbis_af				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Prototypes
 */

static int vorbis_check( void *xmm, XMM_AudioFormat *aformat, int query );

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *vorbis_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  int			ret = XMM_RET_OK;
  const char		*ofn;
  ogg_packet		header, header_comm, header_code;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  if( vorbis_check( xmm, saf, flags & XMM_FILTER_AOF_QUERY ) != XMM_RET_OK )
	return (void *)ret;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(Ogg/Vorbis) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Open output */
  if(( ofn = xmm_AcOutputFilename( xmm, NULL, ".ogg" )) == NULL )
	return NULL;

  if(( priv->pIO = xmm_IOOpen( xmm, ofn, XMM_IO_WRITE )) == NULL )
	return NULL;

  /*
   * Init Ogg/Vorbis
   */

  /* Initialize encoding */
  vorbis_info_init( &priv->vi );
  vorbis_encode_init( &priv->vi, saf->channels, saf->samprate, -1, saf->bitrate, -1 );

  /* Add encoder as comment line */
  vorbis_comment_init( &priv->vc );
  vorbis_comment_add( &priv->vc, "Encoded by "XMM_PROJECT_NAME" Ogg/Vorbis Plugin v"XMM_VERSION_STRING );

  /* Init analyser */
  vorbis_analysis_init( &priv->vd, &priv->vi );
  vorbis_block_init( &priv->vd, &priv->vb );
  
  /* Init packet->stream encoder ( serial number is random ) */
  srand( time( NULL ));
  ogg_stream_init( &priv->os, rand());

  /* Write the three Ogg/Vorbis headers */
  vorbis_analysis_headerout( &priv->vd, &priv->vc, &header, &header_comm, &header_code );
  ogg_stream_packetin( &priv->os, &header);
  ogg_stream_packetin( &priv->os, &header_comm);
  ogg_stream_packetin( &priv->os, &header_code);

  while( 1 )
  {
	int result = ogg_stream_flush( &priv->os, &priv->og );
	if( result == 0 )	break;
	priv->pIO->Write( priv->pIO, priv->og.header, 1, priv->og.header_len );
	priv->pIO->Write( priv->pIO, priv->og.body, 1, priv->og.body_len );
  }

  /* correct fsize */
  if( fsize <= 0 )	fsize = PCM_BUFFER_SIZE;

  xmm_logging( 2, "Ogg/Vorbis! Started ( %i Hz, %i channel(s), format = %x, bs = %i )\n",
			saf->samprate, saf->channels, saf->format, fsize );

  /* Initialize data */
  priv->bufferDelay = fsize / (double)( saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ));
  priv->bufferSSize = ( saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8 );
  priv->bufferSize = fsize;

  /* Set initialized format */
  priv->daf.format = saf->format;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE );
  priv->daf.blockSize = saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8;
  priv->daf.extraSize = 0;
  priv->daf.extraType = 0;
  priv->daf.desc[0] = '\0';
  
  /* return filter address */
  return pFilter;
}

/*
 * Free filter
 */
static int vorbis_Close( XMM_PluginFilterAudio *filter )
{
  struct priv_t *priv = filter->sys.priv;
  int		ret;

  /* Signalize end of stream */
  vorbis_analysis_wrote( &priv->vd, 0 );

  /* Write blocks */
  while( vorbis_analysis_blockout( &priv->vd, &priv->vb ) == 1 )
  {
	vorbis_analysis( &priv->vb, &priv->op );
	/* packet into bitstream */
	ogg_stream_packetin( &priv->os, &priv->op );

	do 
	{
	    ret = ogg_stream_pageout( &priv->os, &priv->og );
	    if( ret == 0 )	break;

	    priv->pIO->Write( priv->pIO, priv->og.header, 1, priv->og.header_len );
	    priv->pIO->Write( priv->pIO, priv->og.body, 1, priv->og.body_len );
	} while( !ogg_page_eos( &priv->og ));
  }

  /* Close encoder stuff */
  ogg_stream_clear( &priv->os );
  vorbis_block_clear( &priv->vb );
  vorbis_dsp_clear( &priv->vd );
  vorbis_info_clear( &priv->vi );

  priv->pIO->Close( priv->pIO );

  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int vorbis_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLQUERY_AFORMAT:
		if( vorbis_check( filter->sys.xmm, (XMM_AudioFormat *)data, 1 ) == XMM_RET_OK )
			return XMM_CTLRET_TRUE;
		else	return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_MAX_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_TOTAL:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_FREE:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data) = param;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &vorbis_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_FLUSH:
		return XMM_CTLRET_TRUE;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(Ogg/Vorbis) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(Ogg/Vorbis) cmd ( 0x%x )" );
}


/*
 * Process data
 */
static int vorbis_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t	*priv = filter->sys.priv;
  float		**buffer;
  int		written = 0, done, i, ret, ss, ch, samples;
  signed char	*ptr = src;

  ss = priv->vi.channels * 2;	/* We need the sample size in the inner loop */
  samples = isize / priv->bufferSSize;

  /* get buffer */
  buffer = vorbis_analysis_buffer( &priv->vd, isize );

  /* Convert to uninterleaved float */
  for( i = 0; i < samples; i++ )
  {
	for( ch = 0; ch < priv->vi.channels; ch++ )
		buffer[ch][i] = (( ptr[i * ss + ch * 2 + 1] << 8 ) | ( 0x00ff & (int)ptr[i * ss + ch * 2] )) / 32768.00;
  }

  /* Write data */
  vorbis_analysis_wrote( &priv->vd, i );

  /* Write blocks */
  while( vorbis_analysis_blockout( &priv->vd, &priv->vb ) == 1 )
  {
	vorbis_analysis( &priv->vb, &priv->op );
	/* packet into bitstream */
	ogg_stream_packetin( &priv->os, &priv->op );

	do 
	{
		ret = ogg_stream_pageout( &priv->os, &priv->og );
		if( ret == 0 )	break;

		done = priv->pIO->Write( priv->pIO, priv->og.header, 1, priv->og.header_len );
		done += priv->pIO->Write( priv->pIO, priv->og.body, 1, priv->og.body_len );
		written += done;
	} while( !ogg_page_eos( &priv->og ));
  }

  if( written == 0 )	return XMM_RET_ERROR;

  if( osize )	*osize = 0;	/* Nothing in dest */
  return written;
}

/*
 * Process data (BQ)
 */
static int vorbis_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin info
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_AOUTPUT,
				XMM_VERSION_NUM,
				"",
				"Ogg/Vorbis",
				"Sound: Ogg/Vorbis",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				vorbis_Open, vorbis_Close, vorbis_Control,
				vorbis_Process, vorbis_ProcessBQ };

/*
 * Internal code
 */

/*
 * Initialize Ogg/Vorbis
 */
static int vorbis_check( void *xmm, XMM_AudioFormat *saf, int query )
{

  if(( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(Ogg/Vorbis) Format 0x%x not supported ( 16bit PCM data needed )", saf->format );

  if(( saf->format & XMM_AUDIO_MASK_SIZE ) != 16 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(Ogg/Vorbis) MPEG Audio supports only 16 bit PCM data" );

  /* return */
  return XMM_RET_OK;
}
