/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wave.c
 * WAVE audio output ( writer )
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>
#include "wave.h"

/*
 * Definitions
 */

#define	PCM_BUFFER_SIZE		16384

/*
 * Global data
 */

			/* Default fragment size and number */
static int		fsize = 16384;

/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_AudioFormat		daf;

    int				data_size, data_start;

    double			bufferDelay;
    long			bufferSize;
    long			bufferSSize;
};

/*
 * Filter info
 */

#define	WAVE_AUDIO_FORMATS	18

static XMM_AudioFormat wave_af[WAVE_AUDIO_FORMATS] =
{
    { XMM_AUDIO_FMT_U8, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "PCM (8bit)" },
    { XMM_AUDIO_FMT_S16LE, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "PCM (16bit,signed,LE)" },
    { XMM_AUDIO_FMT_S24LE, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "PCM (16bit,signed,LE)" },
    { XMM_AUDIO_FMT_S32LE, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "PCM (16bit,signed,LE)" },
    { XMM_AUDIO_FMT_IEEE32LE, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "IEEE FP(32bit,signed,LE)" },
    { XMM_AUDIO_FMT_IEEE64LE, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "IEEE FP(64bit,signed,LE)" },

    { XMM_AUDIO_CODEC_MPEG, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "MPEG (Audio)" },
    { XMM_AUDIO_CODEC_AC3, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "AC3" },
    { XMM_AUDIO_CODEC_GSM610MS, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "GSM 6.10 (WAVE type)" },
    { XMM_AUDIO_CODEC_WMA1, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "DivX Audio (WMA v1)" },
    { XMM_AUDIO_CODEC_WMA2, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "DivX Audio (WMA v2)" },

    { XMM_AUDIO_CODEC_ULAW, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "u-Law" },
    { XMM_AUDIO_CODEC_ALAW, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "A-Law" },
    { XMM_AUDIO_CODEC_G721, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "G.721 (4-bit)" },
    { XMM_AUDIO_CODEC_G723, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "G.723 (3,5-bit)" },
    { XMM_AUDIO_CODEC_G726, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "G.726 (2-bit)" },
    { XMM_AUDIO_CODEC_ADPCM_MS, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "MS-ADPCM" },
    { XMM_AUDIO_CODEC_ADPCM_IMA, XMM_AUDIO_ANY_SRATE, XMM_AUDIO_ANY_CHANN, 0, XMM_AUDIO_ANY_BRATE, 0, 0, "IMA-ADPCM" },
};

static XMM_FilterAudioInfo	wave_fai =
{
    0,
    "",					/* Filename. Will be initialized later */
    "WAVE",				/* Name */
    "",					/* Description */
    "Copyright (c) 2000 Arthur Kleer",	/* Copyright */
    WAVE_AUDIO_FORMATS,			/* Number of supported formats */
    wave_af				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Prototypes
 */

static int WRITE32( XMM_PluginIO *pIO, uint32_t val );
static int WRITE16( XMM_PluginIO *pIO, uint16_t val );
static int WRITE8( XMM_PluginIO *pIO, unsigned char val );
static int WRITEn( XMM_PluginIO *pIO, void *data, uint32_t size );

static int wave_check( void *xmm, XMM_AudioFormat *saf, int query, WAVEFORMATEX *wfx, int *wfx_size );

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *wave_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  int			ret = XMM_RET_OK;
  const char		*ofn;
  int			wfx_size;
  WAVEFORMATEX		wfx;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  if( wave_check( xmm, saf, flags & XMM_FILTER_AOF_QUERY, &wfx, &wfx_size ) != XMM_RET_OK )
	return (void *)ret;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(WAVE) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Open output */
  if(( ofn = xmm_AcOutputFilename( xmm, NULL, ".wav" )) == NULL )
	return NULL;

  if(( priv->pIO = xmm_IOOpen( xmm, ofn, XMM_IO_WRITE )) == NULL )
	return NULL;

  /* Write header */
  WRITE32( priv->pIO, RIFF_ID_RIFF );
  WRITE32( priv->pIO, 0 );
  WRITE32( priv->pIO, RIFF_ID_WAVE );
  WRITE32( priv->pIO, RIFF_ID_fmt );
  WRITE32( priv->pIO, wfx_size + wfx.cbSize );
  WRITEn( priv->pIO, &wfx, wfx_size );
  if( wfx.cbSize )	WRITEn( priv->pIO, &saf[1], wfx.cbSize );
  WRITE32( priv->pIO, RIFF_ID_data );
  WRITE32( priv->pIO, 0 );

  priv->data_start = priv->pIO->Tell( priv->pIO );
  priv->data_size = 0;

  /* correct fsize */
  if( fsize <= 0 )	fsize = PCM_BUFFER_SIZE;

  xmm_logging( 2, "WAVE! Started ( %i Hz, %i channel(s), format %x, bs = %i )\n",
			saf->samprate, saf->channels, saf->format, fsize );

  /* Initialize data */
  priv->bufferDelay = fsize / (double)saf->bitrate;
  priv->bufferSSize = ( saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8 );
  priv->bufferSize = fsize;

  /* Set initialized format */
  priv->daf.format = saf->format;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE );
  priv->daf.blockSize = saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8;
  priv->daf.extraSize = 0;
  priv->daf.extraType = 0;
  priv->daf.desc[0] = '\0';

  /* return filter address */
  return pFilter;
}

/*
 * Free filter
 */
static int wave_Close( XMM_PluginFilterAudio *filter )
{
  struct priv_t *priv = filter->sys.priv;
  int		pos, len;
  char		software[256];

  /**/
  memset( software, 0, 256 );
  strcpy( software, XMM_PROJECT_NAME" WAVE Plugin v"XMM_VERSION_STRING );
  len = strlen( software ) + 1;
  if( len & 1 )	len++;

  /* Add info */
  WRITE32( priv->pIO, RIFF_ID_LIST );
  WRITE32( priv->pIO, 3 * sizeof(uint32_t) + len );
  WRITE32( priv->pIO, RIFF_ID_INFO );
  WRITE32( priv->pIO, RIFF_ID_ISFT );
  WRITE32( priv->pIO, len );
  WRITEn( priv->pIO, software, len );

  /* Fix / Rewrite Header */
  pos = priv->pIO->Tell( priv->pIO );
  priv->pIO->Seek( priv->pIO, 4, XMM_SEEK_SET );
  WRITE32( priv->pIO, pos - 8 );

  priv->pIO->Seek( priv->pIO, priv->data_start - 4, XMM_SEEK_SET );
  WRITE32( priv->pIO, priv->data_size );

  priv->pIO->Close( priv->pIO );

  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int wave_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLQUERY_AFORMAT:
		if( wave_check( filter->sys.xmm, (XMM_AudioFormat *)data, 1, NULL, NULL ) == XMM_RET_OK )
			return XMM_CTLRET_TRUE;
		else	return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_MAX_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_TOTAL:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_FREE:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data) = param;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &wave_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_FLUSH:
		return XMM_CTLRET_TRUE;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(WAVE) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(WAVE) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int wave_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t	*priv = filter->sys.priv;
  int		done;

  done = priv->pIO->Write( priv->pIO, src, 1, isize );
  priv->data_size += done;

  if( done == 0 )	return XMM_RET_ERROR;

  if( osize )	*osize = 0;	/* Nothing in dest */
  return done;
}

/*
 * Process data (BQ)
 */
static int wave_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin info
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_AOUTPUT,
				XMM_VERSION_NUM,
				"",
				"WAVE",
				"Sound: WAVE",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				wave_Open, wave_Close, wave_Control,
				wave_Process, wave_ProcessBQ };

/*
 * Internal code
 */

/*
 * Data output
 */

static int WRITEn( XMM_PluginIO *pIO, void *data, uint32_t size )
{
  return pIO->Write( pIO, data, size, 1 );
}

static int WRITE32( XMM_PluginIO *pIO, uint32_t val )
{
  return pIO->Write( pIO, &val, 4, 1 );
}

static int WRITE16( XMM_PluginIO *pIO, uint16_t val )
{
  return pIO->Write( pIO, &val, 2, 1 );
}

static int WRITE8( XMM_PluginIO *pIO, unsigned char val )
{
  return pIO->Write( pIO, &val, 1, 1 );
}

/*
 * Convert XMM codec to wFormatID
 */

/* XMMP sound format --> WAVE sound format */
struct format_conv
{
	int	ms;
	int	mask_ss;
	int	xmm;
	int	mask;
};

static struct format_conv format_table[] =
{
    { WAVE_FORMAT_PCM		, 0xFFFF, XMM_AUDIO_FMT_U8		, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE | XMM_AUDIO_MASK_SIGNED },
    { WAVE_FORMAT_PCM		, 0xFFFF, XMM_AUDIO_FMT_S16LE		, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE | XMM_AUDIO_MASK_SIGNED },
    { WAVE_FORMAT_PCM		, 0xFFFF, XMM_AUDIO_FMT_S24LE		, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE | XMM_AUDIO_MASK_SIGNED },
    { WAVE_FORMAT_PCM		, 0xFFFF, XMM_AUDIO_FMT_S32LE		, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE | XMM_AUDIO_MASK_SIGNED },
    { WAVE_FORMAT_IEEE_FLOAT	, 0xFFFF, XMM_AUDIO_FMT_IEEE32LE	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE },
    { WAVE_FORMAT_IEEE_FLOAT	, 0xFFFF, XMM_AUDIO_FMT_IEEE64LE	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE },
    { WAVE_FORMAT_MULAW		, 0xFFFF, XMM_AUDIO_CODEC_ULAW | 8	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { WAVE_FORMAT_ALAW		, 0xFFFF, XMM_AUDIO_CODEC_ALAW | 8	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { WAVE_FORMAT_MPEGLAYER3	, 0	, XMM_AUDIO_CODEC_MPEG		, XMM_AUDIO_MASK_CODEC },
    { WAVE_FORMAT_AC3		, 0	, XMM_AUDIO_CODEC_AC3		, XMM_AUDIO_MASK_CODEC },
    { WAVE_FORMAT_GSM610	, 0	, XMM_AUDIO_CODEC_GSM610MS	, XMM_AUDIO_MASK_CODEC },
    { WAVE_FORMAT_G721_ADPCM	, 0xFFFF, XMM_AUDIO_CODEC_G721 | 4	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { WAVE_FORMAT_G723_ADPCM	, 0xFFFF, XMM_AUDIO_CODEC_G723 | 3	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { WAVE_FORMAT_G723_ADPCM	, 0xFFFF, XMM_AUDIO_CODEC_G723 | 5	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { WAVE_FORMAT_G726_ADPCM	, 0xFFFF, XMM_AUDIO_CODEC_G726 | 2	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { WAVE_FORMAT_WMA1		, 0xFFFF, XMM_AUDIO_CODEC_WMA1		, XMM_AUDIO_MASK_CODEC },
    { WAVE_FORMAT_WMA2		, 0xFFFF, XMM_AUDIO_CODEC_WMA2		, XMM_AUDIO_MASK_CODEC },
    { WAVE_FORMAT_ADPCM		, 0xFFFF, XMM_AUDIO_CODEC_ADPCM_MS | 4	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { WAVE_FORMAT_DVI_ADPCM	, 0xFFFF, XMM_AUDIO_CODEC_ADPCM_IMA | 4	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { 0, 0, 0, 0 }
};

/*
 * Initialize WAVE
 */
static int wave_check( void *xmm, XMM_AudioFormat *saf, int query, WAVEFORMATEX *wfx, int *wfx_size )
{
  int		xmm_fmt_idx;

  /* Find XMM format */
  for( xmm_fmt_idx = 0; format_table[xmm_fmt_idx].xmm; xmm_fmt_idx++ )
	if(( saf->format & format_table[xmm_fmt_idx].mask ) == format_table[xmm_fmt_idx].xmm )
		break;

  if( format_table[xmm_fmt_idx].xmm == 0 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(WAVE) Unknown/Unsupported XMM format 0x%x", saf->format );

  /* Query only */
  if( query )	return XMM_RET_OK;

  /* Fill wave format with XMMP audio format info */
  wfx->wFormatTag	= format_table[xmm_fmt_idx].ms;
  wfx->nChannels	= saf->channels;
  wfx->nSamplesPerSec	= saf->samprate;
  wfx->nAvgBytesPerSec	= saf->bitrate / 8;
  wfx->nBlockAlign	= saf->blockSize;
  wfx->wBitsPerSample	= ( saf->format & XMM_AUDIO_MASK_SIZE ) & format_table[xmm_fmt_idx].mask_ss;
  wfx->cbSize		= 0;

  if(( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM )
  {
	if( saf->extraType & XMM_AUDIO_EXT_WAVE )
		wfx->cbSize = saf->extraSize;

	*wfx_size = 18;
  }
  else	*wfx_size = 16;

  /* return */
  return XMM_RET_OK;
}
