/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * sdl.c
 * SDL ( Simple DirectMedia Layer ) graph output
 *
 * TODO:
 *	- XMM_CTLQUERY_GFORMAT returns XMM_CTLRET_TRUE for all YUV formats.
 *	  During init it should be checked if formats really in hardware available.
 */

#include <stdio.h>
#include <stdlib.h>
#include <SDL/SDL.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpgraph.h>
#include <libxmm/lpfilterv.h>
#include <libxmm/util/timer.h>
#include <libxmm/util/mutex.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>
#include <libxmm/event.h>
#include <libxmm/event_ksym.h>

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#define	DEBUG
#endif

/* change from 'undef' to 'define' to disable event checking with timer */
#undef DISABLE_TIMER_EVENT_CHECK

/*
 * Global data
 */

extern XMM_PluginGraph	plugin_info;

/*
 * Types
 */

struct priv_t
{
    int			pixelsize;
    int			gflags;

    /* Dest size */
    int			dwidth;
    int			dheight;
    uint32_t		dformat;
    int			dyuv;
    SDL_Rect		drect;

    /* planar YUV */
    uint32_t		ysize;
    uint32_t		uvsize;

    /* Original size of the input */
    int			swidth;
    int			sheight;
    uint32_t		sformat;

    /* No blitting while resizing */
    XMM_Mutex		*mutex;

    SDL_Surface		*surface;
    SDL_Overlay		*overlay;
    int			Fullscreen;

    /* Video modes */
    int			sdl_flags;
    SDL_Rect		**sdl_modes;
    int			sdl_fflags;
    SDL_Rect		**sdl_fmodes;
};

/*
 * Prototypes
 */

static int sdl_Resize( XMM_PluginGraph *graph, struct priv_t *priv, int width, int height, int flags );
static int sdl_CheckForEvent( void *xmm );

/*
 * Initialize
 */

static XMM_PluginGraph *sdl_Init( void *xmm )
{
  XMM_PluginGraph	*graph;
  const SDL_VideoInfo	*vi;
  struct priv_t		*priv;
  char			driver[256];

  /*
   * Alocate plugin data and private data
   */
  if(( graph = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginGraph ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(SDL) Unable to duplicate plugin_info" );
	return NULL;
  }

  /*
   * These have to be initialized
   */
  priv = (struct priv_t *) &graph[1];
  graph->sys.priv = (void *) priv;
  graph->sys.xmm = xmm;

  /* Initialize SDL */
  if( SDL_Init( SDL_INIT_VIDEO ) < 0 )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, "(SDL) Video init error: %s.", SDL_GetError());
	return NULL;
  }

  SDL_VideoDriverName( driver, 255 );

  SDL_EnableKeyRepeat( 0, 0 );	/* Send key press signals only once */
  SDL_ShowCursor( 0 );

  /* Get 'best' video mode ( format ). We do not want SDL do the conversion */
  vi = SDL_GetVideoInfo();
  priv->pixelsize = vi->vfmt->BytesPerPixel;

  xmm_logging( 2, "SDL! driver = %s: hw surface = %i, blit hw -> hw = %i blit sw -> hw = %i\n",
					    driver, vi->hw_available, vi->blit_hw, vi->blit_sw );

  /* Get video modes for 'best' format */
  priv->sdl_fflags = SDL_FULLSCREEN | SDL_HWSURFACE | SDL_DOUBLEBUF | SDL_HWACCEL;
  if(( priv->sdl_fmodes = SDL_ListModes( vi->vfmt, priv->sdl_fflags )) == NULL )
  {
	priv->sdl_fflags &= ~SDL_HWSURFACE;
	priv->sdl_fmodes = SDL_ListModes( vi->vfmt, priv->sdl_fflags );
  }

  priv->sdl_flags = SDL_HWSURFACE | SDL_HWACCEL;
  if(( priv->sdl_modes = SDL_ListModes( vi->vfmt, priv->sdl_flags )) == NULL )
  {
	priv->sdl_flags &= ~SDL_HWSURFACE;
	priv->sdl_modes = SDL_ListModes( vi->vfmt, priv->sdl_flags );
  }

  /* Create mutex */
  priv->mutex = xmmMutex_Create();

  /*
   * Return object
   */
  return graph;
}

/*
 * Free plugin data
 */

static void sdl_Close( XMM_PluginGraph *graph )
{
  struct priv_t *priv = graph->sys.priv;

  if( priv->surface )	SDL_FreeSurface( priv->surface );
  if( priv->overlay )	SDL_FreeYUVOverlay( priv->overlay );

  SDL_QuitSubSystem( SDL_INIT_VIDEO );

  /* Stop video chain */
  xmm_FilterVChainStop( graph->sys.xmm );

  /* Unlock video chain */
  xmm_FilterVChainUnlock( graph->sys.xmm );

  /* Destroy mutex */
  xmmMutex_Destroy( priv->mutex );

  /*
   * Free plugin memory
   */
  free( graph );
}

/*
 * Control call
 */
static int sdl_Control( XMM_PluginGraph *graph, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = graph->sys.priv;
  XMM_ControlScale	*scale = NULL;
  XMM_ControlSurface	*surface = NULL;

  switch( cmd )
  {
	case XMM_CTLQUERY_HWSCALE:
		return XMM_CTLRET_FALSE;
	    
	case XMM_CTLQUERY_YFLIP:
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_FULLSCREEN:
		return XMM_CTLRET_TRUE;

	case XMM_CTLQUERY_GFORMAT:
		if((uint32_t)data == XMM_GRAPH_FMT_RGB( priv->pixelsize * 8 ))	return XMM_CTLRET_TRUE;
		if((uint32_t)data == XMM_GRAPH_FMT_YV12 )	return XMM_CTLRET_TRUE;
		if((uint32_t)data == XMM_GRAPH_FMT_IYUV )	return XMM_CTLRET_TRUE;
		if((uint32_t)data == XMM_GRAPH_FMT_YUY2 )	return XMM_CTLRET_TRUE;
		if((uint32_t)data == XMM_GRAPH_FMT_UYVY )	return XMM_CTLRET_TRUE;
		if((uint32_t)data == XMM_GRAPH_FMT_YVYU )	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_SURFACE_READ:
		return XMM_CTLRET_TRUE;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_GFORMAT:
		if(( priv->sformat & XMM_GRAPH_FMT_RGB(0)) == XMM_GRAPH_FMT_RGB(0))
		    *((uint32_t *)data) = xmmFOURCC( 'R','G','B', priv->pixelsize * 8 );
		else
		    *((uint32_t *)data) = xmmFOURCC( 'Y','U','Y','2' );

		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_SURFACE:
		surface = (XMM_ControlSurface *)data;

		/* Surface only for internal use, if filter chain active */
		if( xmm_FilterVChainActive( graph->sys.xmm ) == 1 )	return XMM_CTLRET_FALSE;

		/* Return surface type */
		surface->fourcc = priv->dformat;

		/* Return surface pointer and stride */
    		if( priv->dyuv == 1 )
		{
		    surface->data[0] = priv->overlay->pixels[0];
		    surface->data[1] = NULL;
		    surface->data[2] = NULL;
		    surface->stride[0] = priv->overlay->w * 2;
		}
		else if( priv->dyuv == 2 )
		{
		    surface->data[0] = priv->overlay->pixels[0];
		    surface->data[1] = priv->overlay->pixels[1];
		    surface->data[2] = priv->overlay->pixels[2];
		    surface->stride[0] = priv->overlay->w;
		    surface->stride[1] = priv->overlay->w >> 2;
		    surface->stride[2] = priv->overlay->w >> 2;
		}
		else
		{
		    surface->data[0] = priv->surface->pixels;
		    surface->stride[0] = priv->surface->w * priv->pixelsize;
		}

		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_SCREEN:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		scale = (XMM_ControlScale *)data;
#if 0
		if(( scale->width - priv->dwidth ) || ( scale->height - priv->dheight ))
#endif
		    if( sdl_Resize( graph, priv, scale->width, scale->height, scale->flags ) >= 0 )
			return XMM_CTLRET_TRUE;

		return XMM_CTLRET_FALSE;

	case XMM_CTLSET_POSITION:
		return XMM_CTLRET_FALSE;		/* TODO: set window position */

	case XMM_CTLSET_TITLE:
		if( data != NULL )	SDL_WM_SetCaption((char *)data, (char *)data );
		else	SDL_WM_SetCaption( XMM_PROJECT_NAME, XMM_PROJECT_NAME );
		return XMM_CTLRET_TRUE;

	case XMM_CTLSET_FRAMERATE:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_GRAPH )
	return xmm_SetError( graph->sys.xmm, XMM_RET_NOTSUPPORTED, "(SDL) cmd = 0x%x" );

  return xmm_SetError( graph->sys.xmm, XMM_RET_INVALID_ARG, "(SDL) cmd ( 0x%x )" );
}

/*
 * Start graph output
 */

static int sdl_Start( XMM_PluginGraph *graph, int width, int height, uint32_t format, int flags )
{
  struct priv_t *priv = graph->sys.priv;
  char		buffer[5];

  /*
   * Save the original width, height, format and flags for later
   */
  priv->swidth = width;
  priv->sheight = height;
  priv->sformat = format;
  priv->gflags = flags;

  /*
   * Set input format/size for video chain
   */
  xmm_FilterVChainInput( graph->sys.xmm, format, width, height, flags );

  /*
   * Check whether conversion needed
   */
  if( sdl_Control( graph, XMM_CTLQUERY_GFORMAT, 0, (void *)priv->sformat ) != XMM_CTLRET_TRUE )
  {
	sdl_Control( graph, XMM_CTLGET_GFORMAT, 0, &priv->dformat );
  }
  else	priv->dformat = priv->sformat;

  /* Check if YUV overlay possible */
  if(( priv->dformat & XMM_GRAPH_FMT_RGB(0)) == XMM_GRAPH_FMT_RGB(0))	priv->dyuv = 0;
  else switch( priv->dformat )
  {
	case XMM_GRAPH_FMT_YUY2:
	case XMM_GRAPH_FMT_UYVY:
	case XMM_GRAPH_FMT_YVYU:
			priv->dyuv = 1;
			break;

	case XMM_GRAPH_FMT_YV12:
	case XMM_GRAPH_FMT_IYUV:
			priv->dyuv = 2;
			priv->ysize = priv->swidth * priv->sheight;
			priv->uvsize = priv->swidth * priv->sheight / 4;
			break;

	default:	xmm_logging( 1, "SDL! Unknown destination format %s\n", xmm_FOURCC_string( priv->dformat ));
			break;
  }

  strcpy( buffer, xmm_FOURCC_string( priv->sformat ));
  xmm_logging( 2, "SDL! Initialized. Format conversion: %s -> %s\n", buffer, xmm_FOURCC_string( priv->dformat ));

  /* Check periodicaly for events */
#ifndef DISABLE_TIMER_EVENT_CHECK
  xmmTimer_Add((XMM_TimerProc) sdl_CheckForEvent, (void *)graph->sys.xmm );
#endif

  return sdl_Resize( graph, priv, width, height, 0 );
}

/*
 * Draw frame ( or part of it )
 */

static void sdl_Draw( XMM_PluginGraph *graph, uint8_t *data[], int stride[], int x, int y, int width, int height, int flag )
{
  struct priv_t	*priv = graph->sys.priv;
  uint8_t	*src, *dest;
  int		i, j, sstride, dstride, shift;

  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_logging( 1, "SDL! Surface locked\n" );
	return;
  }

  if( xmm_FilterVChainActive( graph->sys.xmm ) == 1 )
  {
	xmm_FilterVChainData( graph->sys.xmm, data, stride, x, y, width, height, 0 );
  }
  else
  {
    if( priv->dyuv == 0 )
    {
	if( data[0] == priv->surface->pixels )
	{
#ifdef DEBUG
	    xmm_logging( 2, "SDL! data is internal surface. (RGB)\n" );
#endif
	    xmmMutex_Unlock( priv->mutex );
	    return;
	}

	src = data[0];
	dest = priv->surface->pixels + ( y * priv->surface->w + x ) * priv->pixelsize;

	width *= priv->pixelsize;
	dstride = priv->surface->w * priv->pixelsize;
	sstride = (( stride == NULL ) ? priv->swidth * priv->pixelsize : stride[0] );

	if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )
	{
	    dest = priv->surface->pixels + (( priv->dheight - y - 1 ) * priv->surface->w + x ) * priv->pixelsize;
	    dstride = -dstride;
	}

	for( i = 0; i < height; i++, src += sstride, dest += dstride )
		memcpy( dest, src, width );
    }
    else if( priv->dyuv == 1 )	/* Packed YUV overlays */
    {
	if( data[0] == priv->overlay->pixels[0] )
	{
#ifdef DEBUG
	    xmm_logging( 2, "SDL! data is internal surface.(YUY2)\n" );
#endif
	    xmmMutex_Unlock( priv->mutex );
	    return;
	}

	if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )
	{
	    /* width is not used. so we set it to ( image width * 2 ) */
	    width = priv->overlay->w * 2;
	    height = priv->overlay->h;

	    src = data[0];
	    dest = priv->overlay->pixels[0] + ( priv->overlay->h - y - 1 ) * priv->overlay->w * 2;

	    dstride = -priv->overlay->w * 2;
	    sstride = (( stride == NULL ) ? priv->overlay->w * 2 : stride[0] );

	    for( i = 0; i < height; i++, src += sstride, dest += dstride )
		memcpy( dest, src, width );
	}
	else
	{
	    memcpy( priv->overlay->pixels[0], data[0], priv->overlay->w * priv->overlay->h * 2 );
	}
    }
    else if( priv->dyuv == 2 )	/* Planar YUV overlays */
    {
	if(( data[0] == priv->overlay->pixels[0] ) && ( data[1] == priv->overlay->pixels[1] ) && ( data[2] == priv->overlay->pixels[2] ))
	{
#ifdef DEBUG
	    xmm_logging( 2, "SDL! data is internal surface.(YV12)\n" );
#endif
	    xmmMutex_Unlock( priv->mutex );
	    return;
	}

	if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )
	{
	    for( j = 0, shift = 0; j < priv->overlay->planes; j++ )
	    {
		dstride = -( priv->overlay->w >> shift );
		sstride = (( stride == NULL ) ? ( priv->swidth >> shift ) : stride[j] );

		src = data[j];
		dest = priv->overlay->pixels[j] + ( priv->dheight - y - 1 ) * dstride;

		for( i = 0; i < height; i++, src += sstride, dest += dstride )
		    memcpy( dest, src, width );

		if( j == 0 )
		{
		    shift++;
		    y >>= 1;
		    width >>= 1;
		    height >>= 1;
		}
	    }
	}
	else
	{
	    memcpy( priv->overlay->pixels[0], data[0], priv->ysize );
	    memcpy( priv->overlay->pixels[1], data[1], priv->uvsize );
	    memcpy( priv->overlay->pixels[2], data[2], priv->uvsize );
	}
    }
    else xmm_logging( 1, "SDL! " __FUNCTION__ "(): unknown destination color space ( internal error )\n" );
  }

  xmmMutex_Unlock( priv->mutex );
}

/*
 * Blit to screen
 */

static void sdl_Blit( XMM_PluginGraph *graph )
{
  struct priv_t *priv = graph->sys.priv;

  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_logging( 1, "SDL! Surface locked\n" );
	return;
  }

  /* Process filter chain */
  if( xmm_FilterVChainActive( graph->sys.xmm ) == 1 )
	xmm_FilterVChain( graph->sys.xmm, priv->surface->pixels );

  if( priv->dyuv )	SDL_DisplayYUVOverlay( priv->overlay, &priv->drect );
  else	SDL_UpdateRect( priv->surface, 0, 0, 0, 0 );

  /* Check for events */
  sdl_CheckForEvent( graph->sys.xmm );

  xmmMutex_Unlock( priv->mutex );
}

/*
 * Plugin info
 */

XMM_PluginGraph	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_GRAPH,
				0,
				XMM_VERSION_NUM,
				"",
				"SDL",
				"Graph: SDL",
				"Copyright (c) 1999 Arthur Kleer",
				NULL, NULL },
				sdl_Init, sdl_Close, sdl_Control, sdl_Start,
				sdl_Draw, sdl_Blit };

/*
 * Internal code
 */

/*
 * Resize window
 */

static int sdl_Resize( XMM_PluginGraph *graph, struct priv_t *priv, int width, int height, int flags )
{
  int			i, mxs = 0x7FFF, mys = 0x7FFF, vm = -1, stretch;
  SDL_Rect		*mode;

  /* Lock surface */
  xmmMutex_Lock( priv->mutex );

  /*
   * Check parameters 
   */
  if(( !width || !height ) && !flags )
  {
	xmmMutex_Unlock( priv->mutex );
	return xmm_SetError( graph->sys.xmm, XMM_RET_ERROR, "(SDL) Wrong parameters: width = 0, height = 0 only with flags ( FULLSCREEN )" );
  }

  if(( flags & XMM_GRAPH_RESIZE_ORIG ) == XMM_GRAPH_RESIZE_ORIG )
  {
	width = priv->swidth;
	height = priv->sheight;
  }

  if(( flags & XMM_GRAPH_RESIZE_DOUBLE ) == XMM_GRAPH_RESIZE_DOUBLE )
  {
	width = priv->swidth << 1;
	height = priv->sheight << 1;
  }

  if(( flags & XMM_GRAPH_RESIZE_FULLSCREEN ) == XMM_GRAPH_RESIZE_FULLSCREEN )
  {
	/* Stretch to full screen ? */
	if( !( width && height ))
	{
		width = priv->swidth;
		height = priv->sheight;
		stretch = 1;
	}
	else	stretch = 0;

	/* Find proper mode */
	if( priv->sdl_fmodes )
	{
	    for( i = 0; priv->sdl_fmodes[i]; i++ )
	    {
		mode = priv->sdl_fmodes[i];
		if(( mode->w > width ) && ( mode->h > height ))
		{
			if(( mode->w < mxs ) && ( mode->h < mys ))
			{
				mxs = mode->w;
				mys = mode->h;
				vm = i;
			}
		}
	    }
	}

	if( vm == -1 )
	{
	    xmmMutex_Unlock( priv->mutex );
	    return xmm_SetError( graph->sys.xmm, XMM_RET_ERROR, "(SDL) Video init error: No matching mode found." );
	}

	if( priv->surface )	SDL_FreeSurface( priv->surface );

	if(( priv->surface = SDL_SetVideoMode( priv->sdl_fmodes[vm]->w, priv->sdl_fmodes[vm]->h, priv->pixelsize * 8, priv->sdl_fflags )) == NULL )
	{
	    xmmMutex_Unlock( priv->mutex );
	    return xmm_SetError( graph->sys.xmm, XMM_RET_ERROR, "(SDL) Coludn't set video mode to %xx%x: %s", priv->sdl_fmodes[vm]->w, priv->sdl_fmodes[vm]->h, SDL_GetError());
	}

	/* Stretch to full screen ? */
	if( stretch )
	{
		width = priv->surface->w;
		height = priv->surface->h;
	}
  }
  else
  {
	/* Free old surface */
	if( priv->surface )	SDL_FreeSurface( priv->surface );

	/* Create new surface */
	if(( priv->surface = SDL_SetVideoMode( width, height, priv->pixelsize * 8, priv->sdl_flags )) == NULL )
	{
	    xmmMutex_Unlock( priv->mutex );
	    return xmm_SetError( graph->sys.xmm, XMM_RET_ERROR, "(SDL) Coludn't set video mode to %xx%x: %s", width, height, SDL_GetError());
	}

	/* Set window title */
	SDL_WM_SetCaption( XMM_PROJECT_NAME, XMM_PROJECT_NAME );
  }

  /*
   * YUV Overlay
   */
  if( priv->dyuv )
  {
	priv->overlay = SDL_CreateYUVOverlay( priv->swidth, priv->sheight, priv->dformat, priv->surface );

#ifdef DEBUG
	if( priv->overlay == NULL )
		xmm_logging( 1, "SDL! Unable to create overlay: %s\n", SDL_GetError());
	else	xmm_logging( 1, "SDL! Overlay created (%s): hw-accelarated = %i\n", xmm_FOURCC_string(priv->dformat), priv->overlay->hw_overlay );
#endif

	priv->drect.x = 0;
	priv->drect.y = 0;
	priv->drect.w = width;
	priv->drect.h = height;

	/* Set destination parameters */
	priv->dwidth = width;
	priv->dheight = height;

	/* Unlock surface */
	xmmMutex_Unlock( priv->mutex );
	return XMM_RET_OK;
  }


  /* Set output end of filter chain */
  xmm_FilterVChainOutput( graph->sys.xmm, priv->dformat, width, height, priv->gflags, priv->surface->w, priv->surface->h );

  /* Set destination size */
  priv->dwidth = width;
  priv->dheight = height;

  /* Unlock surface */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}

/*
 * Get Events
 */

static int sdl_CheckForEvent( void *xmm )
{
  SDL_Event	event;
  XMM_Event_Key	key;

  while( SDL_PollEvent( &event ))
  {
	switch( event.type )
	{
	    case SDL_KEYDOWN:
		    key.type = XMM_EVENT_KEY;
		    key.scode = event.key.keysym.scancode - 8;
		    key.state = XMM_KEY_PRESSED;

		    xmm_PushEvent( xmm, (XMM_Event *) &key, 1 );
		    return 1;

	    case SDL_KEYUP:
		    key.type = XMM_EVENT_KEY;
		    key.scode = event.key.keysym.scancode - 8;
		    key.state = XMM_KEY_RELEASED;

		    xmm_PushEvent( xmm, (XMM_Event *) &key, 1 );
		    return 1;

	    default:
		    break;
	}
  }

  return 0;
}
