/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * svga.c
 * SVGA library graph output
 *
 * TODO:
 */

#include <stdio.h>
#include <stdlib.h>
#include <vga.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpgraph.h>
#include <libxmm/lpfilterv.h>
#include <libxmm/util/timer.h>
#include <libxmm/util/mutex.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>
#include <libxmm/event.h>
#include <libxmm/event_ksym.h>
#include "svga_kb.h"

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* We assume this as the 'best' ( fastest ) format */
#define	SVGA_BEST_FORMAT_BPP	16

/* change from 'undef' to 'define' to disable event checking with timer */
#undef DISABLE_TIMER_EVENT_CHECK

/*
 * Global data
 */

extern XMM_PluginGraph	plugin_info;

/*
 * Types
 */

struct priv_t
{
    int			oldvmode;
    int			pixelsize;
    int			gflags;

    /* Dest size */
    int			dwidth;
    int			dheight;
    uint32_t		dformat;

    /* Screen size */
    int			wwidth;
    int			wheight;
    int			wssize;

    /* Original size of the input */
    int			swidth;
    int			sheight;
    uint32_t		sformat;

    /* No blitting while resizing */
    XMM_Mutex		*mutex;

    uint8_t		*image;
};

/*
 * Prototypes
 */

static int svga_Resize( XMM_PluginGraph *graph, struct priv_t *priv, int width, int height, int flags );
static int svga_findmode( int width, int height, int bpp );
static int svga_CheckForEvent( void *xmm, struct priv_t *priv );

/*
 * Initialize plugin
 */

static XMM_PluginGraph *svga_Init( void *xmm )
{
  XMM_PluginGraph	*graph;
  struct priv_t		*priv;

  /*
   * Alocate plugin data and private data
   */
  if(( graph = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginGraph ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(SVGA) Unable to duplicate plugin_info" );
	return NULL;
  }

  /*
   * These have to be initialized
   */
  priv = (struct priv_t *) &graph[1];
  graph->sys.priv = (void *) priv;
  graph->sys.xmm = xmm;

  /* Save current mode */
  priv->oldvmode = vga_getcurrentmode();

  /* Initialize Svgalib */
  vga_init();

  /* Create mutex */
  priv->mutex = xmmMutex_Create();

  /* Return object */
  return graph;
}

/*
 * Free plugin data
 */

static void svga_Close( XMM_PluginGraph *graph )
{
  struct priv_t *priv = graph->sys.priv;

  free( priv->image );


  /* Restore old mode */
  vga_setmode( priv->oldvmode );

  /* Close keyboard */
  svga_keyboard_Exit();

  /* Stop video chain */
  xmm_FilterVChainStop( graph->sys.xmm );

  /* Unlock video chain */
  xmm_FilterVChainUnlock( graph->sys.xmm );

  /* Destroy mutex */
  xmmMutex_Destroy( priv->mutex );

  free( graph );
}

/*
 * Control call
 */

static int svga_Control( XMM_PluginGraph *graph, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = graph->sys.priv;
  XMM_ControlScale	*scale = NULL;
  XMM_ControlSurface	*surface = NULL;
  XMM_ControlRect	*rect;

  switch( cmd )
  {
	case XMM_CTLQUERY_HWSCALE:
		return XMM_CTLRET_FALSE;
	    
	case XMM_CTLQUERY_YFLIP:
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_FULLSCREEN:
		return XMM_CTLRET_TRUE;

	case XMM_CTLQUERY_GFORMAT:
		if(((uint32_t)data & xmmFOURCC( 'R','G','B', 0 )) == xmmFOURCC( 'R','G','B', 0 ))
		    if( svga_findmode( 0, 0, ( (uint32_t)data >> 24 ) & 0xFF ) > -1 )
			return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_SURFACE_READ:
		return XMM_CTLRET_TRUE;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_GFORMAT:
		*((uint32_t *)data) = xmmFOURCC( 'R','G','B', SVGA_BEST_FORMAT_BPP );
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_SURFACE:
		surface = (XMM_ControlSurface *)data;

		/* Return surface type */
		surface->fourcc = priv->dformat;
		/* Return surface pointer */
    		surface->data[0] = priv->image;
		/* Return surface stride */
		surface->stride[0] = priv->wwidth * priv->pixelsize;

		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_SCREEN:
		rect = (XMM_ControlRect *) data;
		rect->width = priv->wwidth;
		rect->height = priv->wheight;

		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLSET_SCALE:
		scale = (XMM_ControlScale *)data;
#if 0
		if(( scale->width - priv->dwidth ) || ( scale->height - priv->dheight ))
#endif
		    if( svga_Resize( graph, priv, scale->width, scale->height, scale->flags ) >= 0 )
			return XMM_CTLRET_TRUE;

		return XMM_CTLRET_FALSE;

	case XMM_CTLSET_POSITION:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_TITLE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_FRAMERATE:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_GRAPH )
	return xmm_SetError( graph->sys.xmm, XMM_RET_NOTSUPPORTED, "(SVGA) cmd = 0x%x" );

  return xmm_SetError( graph->sys.xmm, XMM_RET_INVALID_ARG, "(SVGA) cmd ( 0x%x )" );
}

/*
 * Start graph output
 */

static int svga_Start( XMM_PluginGraph *graph, int width, int height, uint32_t format, int flags )
{
  struct priv_t *priv = graph->sys.priv;

  /*
   * Save the original width, height, format and flags for later
   */
  priv->swidth = width;
  priv->sheight = height;
  priv->sformat = format;
  priv->gflags = flags;

  /*
   * Set input format/size for video chain
   */
  xmm_FilterVChainInput( graph->sys.xmm, format, width, height, flags );

  /*
   * Check whether conversion needed
   */
  if( svga_Control( graph, XMM_CTLQUERY_GFORMAT, 0, (void *)priv->sformat ) != XMM_CTLRET_TRUE )
  {
	svga_Control( graph, XMM_CTLGET_GFORMAT, 0, &priv->dformat );
  }
  else	priv->dformat = priv->sformat;

  /* Initialize keyboard */
  svga_keyboard_Init();

  /* Check periodicaly for events */
#ifndef DISABLE_TIMER_EVENT_CHECK
  xmmTimer_Add((XMM_TimerProc) svga_CheckForEvent, (void *)priv );
#endif

  return svga_Resize( graph, priv, width, height, 0 );
}

/*
 * Draw frame ( or part of it )
 */

static void svga_Draw( XMM_PluginGraph *graph, uint8_t *data[], int stride[], int x, int y, int width, int height, int flags )
{
  struct priv_t *priv = graph->sys.priv;
  uint8_t *src, *dest;
  int i, sstride, dstride;

  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_logging( 1, "SVGA! Surface locked\n" );
	return;
  }

  if( xmm_FilterVChainActive( graph->sys.xmm ) == 1 )
  {
	xmm_FilterVChainData( graph->sys.xmm, data, stride, x, y, width, height, 0 );
  }
  else
  {
	if( data[0] == priv->image )
	{
#ifdef DEBUG
	    xmm_logging( 2, "SDL! data is internal surface. (RGB)\n" );
#endif
	    xmmMutex_Unlock( priv->mutex );
	    return;
	}

	src = data[0];
	dest = priv->image + ( y * priv->wwidth + x ) * priv->pixelsize;

	width *= priv->pixelsize;
	dstride = priv->wwidth * priv->pixelsize;
	sstride = (( stride == NULL ) ? priv->swidth * priv->pixelsize : stride[0] );

	if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )
	{
	    dest = priv->image + (( priv->dheight - y - 1 ) * priv->wwidth + x ) * priv->pixelsize;
	    dstride = -dstride;
	}

	for( i = 0; i < height; i++, src += sstride, dest += dstride )
		memcpy( dest, src, width );
  }

  xmmMutex_Unlock( priv->mutex );
}

/*
 * Blit to screen
 */

static void svga_Blit( XMM_PluginGraph *graph )
{
  struct priv_t *priv = graph->sys.priv;
  int i;

  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_logging( 1, "SVGA! Surface locked\n" );
	return;
  }

  /* Process filter chain */
  if( xmm_FilterVChainActive( graph->sys.xmm ) == 1 )
	xmm_FilterVChain( graph->sys.xmm, &priv->image );

  /* Draw buffer */
  for( i = 0; i < priv->dheight; i++ )
	vga_drawscansegment( priv->image + i * priv->wwidth * priv->pixelsize, 0, i, priv->wwidth * priv->pixelsize );

  /* Check for events */
  svga_CheckForEvent( graph->sys.xmm, priv );

  xmmMutex_Unlock( priv->mutex );
}

/*
 * Plugin info
 */

XMM_PluginGraph	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_GRAPH,
				0,
				XMM_VERSION_NUM,
				"",
				"SVGA",
				"Graph: Svgalib",
				"Copyright (c) 1999 Arthur Kleer",
				NULL, NULL },
				svga_Init, svga_Close, svga_Control, svga_Start,
				svga_Draw, svga_Blit };

/*
 * Internal code
 */

/*
 * Resize window
 */

static int svga_Resize( XMM_PluginGraph *graph, struct priv_t *priv, int width, int height, int flags )
{
  int			vm, stretch;
  vga_modeinfo		*pModeinfo;

  /* Lock surface */
  xmmMutex_Lock( priv->mutex );

  /*
   * Check parameters
   */
  if(( !width || !height ) && !flags )
  {
	xmmMutex_Unlock( priv->mutex );
	return xmm_SetError( graph->sys.xmm, XMM_RET_ERROR, "(SVGA) Wrong parameters: width = 0, height = 0 only with flags ( FULLSCREEN )" );
  }

  if(( flags & XMM_GRAPH_RESIZE_ORIG ) == XMM_GRAPH_RESIZE_ORIG )
  {
	width = priv->swidth;
	height = priv->sheight;
  }

  if(( flags & XMM_GRAPH_RESIZE_DOUBLE ) == XMM_GRAPH_RESIZE_DOUBLE )
  {
	width = priv->swidth << 1;
	height = priv->sheight << 1;
  }

  if(( flags & XMM_GRAPH_RESIZE_FULLSCREEN ) == XMM_GRAPH_RESIZE_FULLSCREEN )
  {
	/* Stretch to full screen ? */
	if( !( width && height ))
	{
		width = priv->swidth;
		height = priv->sheight;
		stretch = 1;
	}
	else	stretch = 0;
  }
  else	stretch = 0;


  /*
   * Find mode
   */
  if(( vm = svga_findmode( width, height, ( priv->dformat >> 24 ) & 0xFF )) == -1 )
  {
	xmmMutex_Unlock( priv->mutex );
	return xmm_SetError( graph->sys.xmm, XMM_RET_ERROR, "(SVGA) Video init error: No matching mode found." );
  }

  /* Get mode info */
  pModeinfo = vga_getmodeinfo( vm );
  priv->wwidth = pModeinfo->width;
  priv->wheight = pModeinfo->height;
  priv->wssize = priv->wwidth * priv->wheight * pModeinfo->bytesperpixel;
  priv->pixelsize = pModeinfo->bytesperpixel;

  /* Stretch to full screen ? */
  if(( flags & XMM_GRAPH_RESIZE_FULLSCREEN ) == XMM_GRAPH_RESIZE_FULLSCREEN )
  {
	if( stretch )
	{
		width = priv->wwidth;
		height = priv->wheight;
	}
  }

  /* Set output end of filter chain */
  xmm_FilterVChainOutput( graph->sys.xmm, priv->dformat, width, height, priv->gflags, priv->wwidth, priv->wheight );

  /* Set destination size */
  priv->dwidth = width;
  priv->dheight = height;

  /* Free existing screen buffer */
  if( priv->image )	free( priv->image );

  /* Allocate new screen buffer */
  if(( priv->image = malloc( priv->wssize )) == NULL )
  {
	xmmMutex_Unlock( priv->mutex );
	return xmm_SetError( graph->sys.xmm, XMM_RET_ALLOC, "(SVGA) Unable to allocate virtual page." );
  }

  /* Set mode */
  vga_setmode( vm );

  xmm_logging( 2, "SVGA! mode(%d): %dx%dx%d (%dbyte) (=total: %ldbytes) \n",
	vm, priv->wwidth, priv->wheight, pModeinfo->colors, pModeinfo->bytesperpixel, priv->wssize );

  /* Unlock surface */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}

/*
 * Find proper mode
 */

static int svga_findmode( int width, int height, int bpp )
{
  int i, vm = -1;
  int mxs = 0x7FFF, mys = 0x7FFF;
  vga_modeinfo *pModeinfo;

  for( i = 1; i <= vga_lastmodenumber(); i++ )
  {
	if( vga_hasmode( i ))
	{
		pModeinfo = vga_getmodeinfo( i );

		if(( pModeinfo->width >= width ) && ( pModeinfo->height >= height ))
		{
			if( pModeinfo->colors == ( 1 << bpp ))
			{
			    if(( pModeinfo->width < mxs ) && ( pModeinfo->height < mys ))
			    {
				mxs = pModeinfo->width;
				mys = pModeinfo->height;
				vm = i;
			    }
			}
		}
	}
  }

  return vm;
}

/*
 * Check for Window event
 */

static int svga_CheckForEvent( void *xmm, struct priv_t *priv )
{
  XMM_Event_Key	key;
  int		scancode, state;

  if( svga_keyboard_PollEvent( &scancode, &state ))
  {
	if( state == 1 )
	{
		key.type = XMM_EVENT_KEY;
		key.scode = scancode;
		key.state = XMM_KEY_PRESSED;

		xmm_PushEvent( xmm, (XMM_Event *) &key, 1 );
		return 1;
	}

	if( state == 0 )
	{
		key.type = XMM_EVENT_KEY;
		key.scode = scancode;
		key.state = XMM_KEY_RELEASED;

		xmm_PushEvent( xmm, (XMM_Event *) &key, 1 );
		return 1;
	}
  }

  return 0;
}
