/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* x11.c
 * X11 graph output
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#ifdef HAVE_XSHM
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#endif

#ifdef HAVE_XV
#include <X11/extensions/Xvlib.h>
#include "x11_xv.h"
#endif

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpgraph.h>
#include <libxmm/lpfilterv.h>
#include <libxmm/util/timer.h>
#include <libxmm/util/mutex.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>
#include <libxmm/event.h>
#include <libxmm/event_ksym.h>

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

#ifdef HAVE_XSHM
#define XSHM_BUFFERS			1
#endif

/* change from 'undef' to 'define' to disable event checking with timer */
#undef DISABLE_TIMER_EVENT_CHECK

/*
 * Global data
 */

extern XMM_PluginGraph	plugin_info;

/*
 * Prototypes
 */

static int x11_Resize( XMM_PluginGraph *graph, int width, int height, int flags );
static int x11_CheckForEvent( XMM_PluginGraph *graph );
static int x11_screen_saver( Display *display, int timeout );

/* XF86-DGA Extension 1.x */
#ifdef HAVE_XF86DGA1x
char *xf86dga_start( Display *display, int screen, int *xsize, int *ysize, int *pwidth );
void xf86dga_exit( Display *display, int screen );
#endif

/* XF86-DGA Extension 2.0 */
#ifdef HAVE_XF86DGA2
char *xf86dga2_start( Display *display, int screen, int pixelsize, int *xsize, int *ysize, int *pwidth );
void xf86dga2_exit( Display *display, int screen );
#endif

/* X Shared Memory extension */
#ifdef HAVE_XSHM
XImage *xshm_Create( Display *display, Visual *visual, XShmSegmentInfo *Shminfo, int width, int height, int bpp );
void xshm_Destroy( Display *display, XShmSegmentInfo *Shminfo );
#endif

/* XVideo extension */
#ifdef HAVE_XV
int xv_initData( Display *display, xv_data_t *xvd );
void xv_exit( Display *display, xv_data_t *xvd );
int xv_start( Display *display, xv_data_t *xvd, uint32_t format, int xsize, int ysize );
#endif

/* Xdpms extension */
#ifdef HAVE_XDPMS
int x11_dpms_off( Display *display );
int x11_dpms_on( Display *display );
#endif

/*
 * Types
 */

struct priv_t
{
    /* Initialized by x11_Init */
    Display		*display;
    int			screen;
    Visual		*visual;

    /* Initialized by x11_Start */
    Window		window;
    GC			gc;
    int			dbpp;
    int			pixelsize;
    int			bgr;

    /* Initialized by x11_Resize */
    XImage		*image;
    uint8_t		*image_data;

    /* Flags */
    int			gflags;

    /* Dest size */
    int			dwidth;
    int			dheight;
    uint32_t		dformat;
    int			dxoff;
    int			dyoff;

    /* Screen size */
    int			wwidth;
    int			wheight;
    int			wssize;

    /* Original size of the input */
    int			swidth;
    int			sheight;
    uint32_t		sformat;

    /* No blitting while resizing */
    XMM_Mutex		*mutex;

    /* Fullscreen */
    int			Fullscreen;

#if defined( HAVE_XF86DGA1x ) || defined( HAVE_XF86DGA2 )
    char		*MemBase;
    int			PhysWidth;
#endif

#ifdef HAVE_XSHM
    XShmSegmentInfo	ShmInfo[XSHM_BUFFERS];
    int			xshm_mode;
#endif

#ifdef HAVE_XV
    int			dyuv;

    xv_data_t		xvd;
    uint32_t		ysize;
    uint32_t		uvsize;
    uint32_t		compOffset[3];
#endif

#ifdef HAVE_XDPMS
    int			dpms_disabled;
#endif
    int			ss_timeout;
};

/*
 * Initialize
 */
static XMM_PluginGraph *x11_Init( void *xmm )
{
  XMM_PluginGraph	*graph;
  struct priv_t		*priv;
#if defined( HAVE_XV ) && defined( DEBUG )
  int			i;
#endif

  /*
   * Alocate plugin data and private data
   */
  if(( graph = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginGraph ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(X11) Unable to duplicate plugin_info" );
	return NULL;
  }

  /*
   * These have to be initialized
   */
  priv = (struct priv_t *) &graph[1];
  graph->sys.priv = (void *) priv;
  graph->sys.xmm = xmm;

  /*
   * Open Display, get Screen and Visual ( we need this for Control() )
   */
  if(( priv->display = XOpenDisplay( NULL )) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, "(X11) Couldn't open display." );
	return NULL;
  }

  priv->screen = DefaultScreen( priv->display );
  priv->visual = DefaultVisual( priv->display, priv->screen );

  /* Get BPP of screen */
  priv->dbpp = DefaultDepth( priv->display, priv->screen );

  {
	XImage *img;

	img = XGetImage( priv->display, RootWindow( priv->display, priv->screen ),
				0, 0, 1, 1, AllPlanes, ZPixmap );

	/* Get pixelsize */
	priv->pixelsize = img->bits_per_pixel / 8;

	/* RGB or BGR ? */
	priv->bgr = ( img->blue_mask & 1 ) ? 0 : 1;

	XDestroyImage( img );
  }

  /* Output some information */
  xmm_logging( 2, "X11! [vendor] %s [relase] %i [protocol] %i.%i\n", 
					    ServerVendor( priv->display ),
					    VendorRelease( priv->display ),
					    ProtocolVersion( priv->display ),
					    ProtocolRevision( priv->display ));

#ifdef HAVE_XV
  /* Check for XV extension */
  if( xv_initData( priv->display, &priv->xvd ) < 0 )
	xmm_logging( 1, "X11! Unable to initialize XVideo extension. Deactivating...\n" );

#ifdef DEBUG
  xmm_logging( 1, "X11! XVideo extension: %i formats available.\n", priv->xvd.formats );

    for( i = 0; i < priv->xvd.formats; i++ )
	xmm_logging( 1, "\t0x%x (%s)\n", priv->xvd.aFormat[i], xmm_FOURCC_string( priv->xvd.aFormat[i] ));

#endif	/* DEBUG */
#endif	/* HAVE_XV */

  /* Create mutex */
  priv->mutex = xmmMutex_Create();

  /*
   * Return object
   */
  return graph;
}

/*
 * Free plugin data
 */
static void x11_Close( XMM_PluginGraph *graph )
{
  struct priv_t *priv = graph->sys.priv;

#ifdef HAVE_XV
  /* Free XV resources */
  if( priv->xvd.image )	xv_exit( priv->display, &priv->xvd );
#endif

#ifdef HAVE_XSHM
  /* Free XShm resources */
  if( priv->xshm_mode && priv->image )
  {
	xshm_Destroy( priv->display, &priv->ShmInfo[0] );
	priv->image = NULL;
	priv->image_data = NULL;
  }
#endif

#ifdef HAVE_XDPMS
  if( priv->dpms_disabled )	x11_dpms_on( priv->display );
#endif

  if( priv->ss_timeout )	x11_screen_saver( priv->display, priv->ss_timeout );

  /*
   * Free X stuff
   */
  XAutoRepeatOn( priv->display );
  XCloseDisplay( priv->display );

  /* Free / Destroy Image ( data ) */
  if( priv->image )	XDestroyImage( priv->image );
  else	if( priv->image_data )	free( priv->image_data );

  /* Stop video chain */
  xmm_FilterVChainStop( graph->sys.xmm );

  /* Unlock video chain */
  xmm_FilterVChainUnlock( graph->sys.xmm );

  /* Destroy mutex */
  xmmMutex_Destroy( priv->mutex );

  /*
   * Free plugin memory
   */
  free( graph );
}

/*
 * Control call
 */
static int x11_Control( XMM_PluginGraph *graph, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = graph->sys.priv;
  XMM_ControlScale	*scale;
  XMM_ControlSurface	*surface = NULL;
  XMM_ControlRect	*rect;
#ifdef HAVE_XV
  int			i;
#endif

  switch( cmd )
  {
	case XMM_CTLQUERY_HWSCALE:
		return XMM_CTLRET_FALSE;
	    
	case XMM_CTLQUERY_YFLIP:
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_FULLSCREEN:
		return XMM_CTLRET_TRUE;

	case XMM_CTLQUERY_GFORMAT:
#ifdef HAVE_XV
		/* YCrCb formats */
		if( priv->xvd.aFormat != NULL )
		{
		    for( i = 0; i < priv->xvd.formats; i++ )
			if((uint32_t) data == priv->xvd.aFormat[i] )	break;

		    if( i != priv->xvd.formats )	return XMM_CTLRET_TRUE;
		}
#endif
		/* RGB formats */
		if( priv->bgr )
		{
		    if((uint32_t) data == xmmFOURCC( 'B','G','R', priv->pixelsize * 8 ))
			return XMM_CTLRET_TRUE;
		}
		else
		{
		    if((uint32_t) data == xmmFOURCC( 'R','G','B', priv->pixelsize * 8 ))
			return XMM_CTLRET_TRUE;
		}

		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_SURFACE_READ:
		return XMM_CTLRET_TRUE;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;
    
	case XMM_CTLGET_GFORMAT:
#ifdef HAVE_XV
		/* YCrCb formats */
		if( priv->xvd.aFormat != NULL )
		{
		    *((uint32_t *)data) = priv->xvd.aFormat[0];
		}
		else
#endif
		if( priv->bgr )
		{
		    *((uint32_t *)data) = xmmFOURCC( 'B','G','R', priv->pixelsize * 8 );
		}
		else
		{
		    *((uint32_t *)data) = xmmFOURCC( 'R','G','B', priv->pixelsize * 8 );
		}

		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_SURFACE:
		surface = (XMM_ControlSurface *)data;

		/* Surface only for internal use, if filter chain active */
		if( xmm_FilterVChainActive( graph->sys.xmm ) == 1 )	return XMM_CTLRET_FALSE;

		/* Return surface type */
		surface->fourcc = priv->dformat;

		/* Return surface pointer */
#ifdef HAVE_XV
    		if( priv->dyuv == 1 )
		{
		    surface->data[0] = priv->xvd.image->data;
		    surface->data[1] = NULL;
		    surface->data[2] = NULL;
		    surface->stride[0] = priv->xvd.image->width * 2;
		}
		else if( priv->dyuv == 2 )
		{
		    surface->data[0] = priv->xvd.image->data;
		    surface->data[1] = (uint8_t *)priv->xvd.image->data + priv->ysize;
		    surface->data[2] = (uint8_t *)priv->xvd.image->data + priv->ysize + priv->uvsize;
		    surface->stride[0] = priv->xvd.image->width;
		    surface->stride[1] = priv->xvd.image->width >> 2;
		    surface->stride[2] = priv->xvd.image->width >> 2;
		}
		else
		{
#endif
		    surface->data[0] = priv->image_data;
		    surface->stride[0] = priv->wwidth * priv->pixelsize;
#ifdef HAVE_XV
		}
#endif

		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_SCREEN:
		rect = (XMM_ControlRect *) data;
		rect->width = DisplayWidth( priv->display, priv->screen );
		rect->height = DisplayHeight( priv->display, priv->screen );

		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLSET_SCALE:
		scale = (XMM_ControlScale *)data;
#if 0
		if(( scale->width - priv->dwidth ) || ( scale->height - priv->dheight ))
#endif
		    if( x11_Resize( graph, scale->width, scale->height, scale->flags ) >= 0 )
			return XMM_CTLRET_TRUE;

		return XMM_CTLRET_FALSE;

	case XMM_CTLSET_POSITION:
		XMoveWindow( priv->display, priv->window,  ((XMM_ControlRect *)data)->x, ((XMM_ControlRect *)data)->y );
		return XMM_CTLRET_TRUE;

	case XMM_CTLSET_TITLE:
		if( data == NULL )
		    XSetStandardProperties( priv->display, priv->window, XMM_PROJECT_NAME, XMM_PROJECT_NAME, None, NULL, 0, NULL );
		else
		    XSetStandardProperties( priv->display, priv->window, (char *)data, (char *)data, None, NULL, 0, NULL );
		return XMM_CTLRET_TRUE;

	case XMM_CTLSET_FRAMERATE:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_GRAPH )
	return xmm_SetError( graph->sys.xmm, XMM_RET_NOTSUPPORTED, "(X11) cmd = 0x%x" );

  return xmm_SetError( graph->sys.xmm, XMM_RET_INVALID_ARG, "(X11) cmd ( 0x%x )" );
}

/*
 * Start graph output
 */
static int x11_Start( XMM_PluginGraph *graph, int width, int height, uint32_t format, int flags )
{
  struct priv_t *priv = graph->sys.priv;
  XEvent	event;
  Pixmap	cursor_bitmap;
  Cursor	cursor;
  XColor	fg, bg;
  char		data[8]={ 0, 0, 0, 0, 0, 0, 0, 0 };
  char		buffer[5];
  int		ret;
  uint32_t	aFmt[] = 
  {
    XMM_GRAPH_FMT_RGB( priv->pixelsize * 8 ),
    XMM_GRAPH_FMT_BGR( priv->pixelsize * 8 ),
  };

  /* Check display size */
  if(( DisplayWidth( priv->display, priv->screen ) < width ) ||
	( DisplayHeight( priv->display, priv->screen ) < height ))
  {
	return xmm_SetError( graph->sys.xmm, XMM_RET_ERROR, "(X11) Needs at least %ix%i to run.\n", width, height );
  }

  /* Create Window */
  priv->window = XCreateSimpleWindow( priv->display, RootWindow( priv->display, priv->screen ),
				0, 0, width, height, CopyFromParent, 0 /*CopyFromParent*/,
				BlackPixel( priv->display, priv->screen ));


  /* Create GC */
  priv->gc = XCreateGC( priv->display, priv->window, 0, NULL );
  XSetForeground( priv->display, priv->gc, 0 );

  /* Select Input and map window */
  XSelectInput( priv->display, priv->window, ExposureMask | KeyPressMask );
  XMapWindow( priv->display, priv->window );

  /* wait for exposure */
  do
  {
	XNextEvent( priv->display, &event );
  } while( event.type != Expose );

  /* no interest in Exposure */
  XSelectInput( priv->display, priv->window, KeyPressMask | KeyReleaseMask );

  /* Disable mouse pointer ( set it to nothing ) */
  fg.pixel = WhitePixel( priv->display, priv->screen );
  XQueryColor( priv->display, DefaultColormap( priv->display, priv->screen ), &fg );
  bg.pixel = BlackPixel( priv->display, priv->screen );
  XQueryColor( priv->display, DefaultColormap( priv->display, priv->screen ), &bg );

  cursor_bitmap = XCreateBitmapFromData( priv->display, priv->window, data, 8, 8 );
  cursor = XCreatePixmapCursor( priv->display, cursor_bitmap, cursor_bitmap, &fg, &bg, 8, 8 );
  XFreePixmap( priv->display, cursor_bitmap );

  XDefineCursor( priv->display, priv->window, cursor );
  XFreeCursor( priv->display, cursor );

  /*
   * Check for XShm extension
   */

#ifdef HAVE_XSHM

  if( XShmQueryExtension( priv->display ))
  {
	priv->xshm_mode = 1;
	xmm_logging( 2, "X11! MIT X Shared memory extension enabled\n" );
  }
  else
  {
	priv->xshm_mode = 0;
	xmm_logging( 2, "X11! MIT X Shared memory extension disabled\n" );
  }

#endif

  /*
   * Save the original width, height, format and flags for later
   */
  priv->swidth = width;
  priv->sheight = height;
  priv->sformat = format;
  priv->gflags = flags;

  /*
   * Set input format/size for video chain
   */
  xmm_FilterVChainInput( graph->sys.xmm, format, width, height, flags );

  /*
   * Check whether conversion needed
   */
  if( x11_Control( graph, XMM_CTLQUERY_GFORMAT, 0, (void *)priv->sformat ) != XMM_CTLRET_TRUE )
  {
	x11_Control( graph, XMM_CTLGET_GFORMAT, 0, &priv->dformat );

	if(( priv->sformat & XMM_GRAPH_FMT_RGB(0)) == XMM_GRAPH_FMT_RGB(0))
	    priv->dformat = aFmt[priv->bgr];

	if(( priv->sformat & XMM_GRAPH_FMT_BGR(0)) == XMM_GRAPH_FMT_BGR(0))
	    priv->dformat = aFmt[priv->bgr];
  }
  else	priv->dformat = priv->sformat;

#ifdef HAVE_XV

  /* Check if YUV overlay possible */
  if(( priv->dformat & XMM_GRAPH_FMT_RGB(0)) == XMM_GRAPH_FMT_RGB(0))	priv->dyuv = 0;
  else switch( priv->dformat )
  {
	case XMM_GRAPH_FMT_YUY2:
	case XMM_GRAPH_FMT_UYVY:
	case XMM_GRAPH_FMT_YVYU:
			priv->dyuv = 1;
			break;

	case XMM_GRAPH_FMT_YV12:
	case XMM_GRAPH_FMT_IYUV:
			priv->dyuv = 2;
			priv->ysize = priv->swidth * priv->sheight;
			priv->uvsize = priv->swidth * priv->sheight / 4;

			priv->compOffset[0] = 0;
			priv->compOffset[1] = priv->ysize;
			priv->compOffset[2] = priv->ysize + priv->uvsize;
			break;

	default:	xmm_logging( 1, "X11! Unknown destination format %s\n", xmm_FOURCC_string( priv->dformat ));
			break;
  }

#endif

  strcpy( buffer, xmm_FOURCC_string( priv->dformat ));

#ifdef HAVE_XV
  xmm_logging( 2, "X11! Init. Format: %x [%s] -> %x [%s] YF = %s Overlay: %s\n", priv->sformat, xmm_FOURCC_string( priv->sformat ), priv->dformat, buffer, ( priv->gflags & XMM_GRAPH_FLAG_YFLIP ) ? "yes" : "no", priv->dyuv ? "YUV" : "no" );
#else
  xmm_logging( 2, "X11! Init. Format: %x [%s] -> %x [%s] YF = %s\n", priv->sformat, xmm_FOURCC_string( priv->sformat ), priv->dformat, buffer, ( priv->gflags & XMM_GRAPH_FLAG_YFLIP ) ? "yes" : "no" );
#endif

  /* Check periodicaly for events */
#ifndef DISABLE_TIMER_EVENT_CHECK
  xmmTimer_Add((XMM_TimerProc) x11_CheckForEvent, (void *)graph );
#endif


  ret = x11_Resize( graph, width, height, 0 );

#ifdef HAVE_XDPMS
  if( ret == XMM_RET_OK )	priv->dpms_disabled = x11_dpms_off( priv->display );
#endif

  /* Disable screen-saver */
  priv->ss_timeout = x11_screen_saver( priv->display, 0 );

  /* no auto repeat for keys */
  XAutoRepeatOff( priv->display );

  return ret;  
}

/*
 * Draw frame ( or part of it )
 */
static void x11_Draw( XMM_PluginGraph *graph, uint8_t *data[], int stride[], int x, int y, int width, int height, int flag )
{
  struct priv_t *priv = graph->sys.priv;
  uint8_t	*src, *dest;
  int		i, sstride, dstride;
#ifdef HAVE_XV
  int		j, shift;
#endif

  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_logging( 1, "X11! Surface locked\n" );
	return;
  }

  if( xmm_FilterVChainActive( graph->sys.xmm ) == 1 )
  {
	xmm_FilterVChainData( graph->sys.xmm, data, stride, x, y, width, height, 0 );
  }
  else
  {
#ifdef HAVE_XV
    if( priv->dyuv == 0 )	/* RGB */
    {
#endif
	src = data[0];
	dest = priv->image_data + ( y * priv->wwidth + x ) * priv->pixelsize;

	width *= priv->pixelsize;
	dstride = priv->wwidth * priv->pixelsize;
	sstride = (( stride == NULL ) ? priv->swidth * priv->pixelsize : stride[0] );

	if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )
	{
	    dest = priv->image_data + (( priv->dheight - y - 1 ) * priv->wwidth + x ) * priv->pixelsize;
	    dstride = -dstride;
	}

	for( i = 0; i < height; i++, src += sstride, dest += dstride )
		memcpy( dest, src, width );
#ifdef HAVE_XV
    }
    else if( priv->dyuv == 1 )	/* Packed YUV overlays */
    {
	if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )
	{
	    /* width is not used. so we set it to ( image width * 2 bpp ) */
	    width = priv->xvd.image->width * 2;
	    height = priv->xvd.image->height;

	    src = data[0];
	    dest = priv->xvd.image->data + ( priv->xvd.image->height - y - 1 ) * priv->xvd.image->width * 2;

	    dstride = -priv->xvd.image->width * 2;
	    sstride = (( stride == NULL ) ? priv->xvd.image->width * 2 : stride[0] );

	    for( i = 0; i < height; i++, src += sstride, dest += dstride )
		memcpy( dest, src, width );
	}
	else
	{
	    memcpy( priv->xvd.image->data, data[0], priv->xvd.image->data_size );
	}
    }
    else if( priv->dyuv == 2 )	/* Planar YUV overlays */
    {
	if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )
	{
	    for( j = 0, shift = 0; j < 3; j++ )
	    {
		dstride = -( priv->xvd.image->width >> shift );
		sstride = (( stride == NULL ) ? ( priv->swidth >> shift ) : stride[j] );

		src = data[j];
		dest = (uint8_t *)priv->xvd.image->data + priv->compOffset[j] + (( priv->sheight - y - 1 ) >> shift ) * -dstride;

		for( i = 0; i < height; i++, src += sstride, dest += dstride )
		    memcpy( dest, src, width );

		if( shift == 0 )
		{
		    shift = 1;
		    width >>= 1;
		    height >>= 1;
		}
	    }
	}
	else
	{
	    memcpy( (uint8_t *)priv->xvd.image->data, data[0], priv->ysize );
	    memcpy( (uint8_t *)priv->xvd.image->data + priv->ysize, data[1], priv->uvsize );
	    memcpy( (uint8_t *)priv->xvd.image->data + priv->ysize + priv->uvsize, data[2], priv->uvsize );
	}
    }
    else xmm_logging( 1, "X11! Draw(): unknown destination color space ( internal error )\n" );
#endif
  }

  xmmMutex_Unlock( priv->mutex );
}

/*
 * Blit to screen
 */
static void x11_Blit( XMM_PluginGraph *graph )
{
  struct priv_t *priv = graph->sys.priv;

  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_logging( 1, "X11! Surface locked\n" );
	return;
  }

  /* Process filter chain */
  if( xmm_FilterVChainActive( graph->sys.xmm ) == 1 )
	xmm_FilterVChain( graph->sys.xmm, &priv->image_data );

#ifdef HAVE_XV
  if( priv->xvd.image && priv->dyuv )
  {
	XvShmPutImage( priv->display, priv->xvd.portID, priv->window, priv->gc,
	    priv->xvd.image,
	    0, 0, priv->xvd.image->width, priv->xvd.image->height,
	    priv->dxoff, priv->dyoff, priv->dwidth, priv->dheight, False );
  }
  else
#endif
#if defined( HAVE_XF86DGA1x ) || defined( HAVE_XF86DGA2 )
  if( priv->Fullscreen && ( priv->MemBase != NULL ))
  {
	/* Nothing has to be done if DGA is used */
  }
  else
#endif
#ifdef HAVE_XSHM
  if( priv->xshm_mode )
  {
	XShmPutImage( priv->display, priv->window, priv->gc, priv->image, 0, 0, 0, 0, priv->image->width, priv->image->height, True );
  }
  else
#endif
	XPutImage( priv->display, priv->window, priv->gc, priv->image, 0, 0, 0, 0, priv->image->width, priv->image->height );

  XFlush( priv->display );

  /* Check for events */
  x11_CheckForEvent( graph );

  xmmMutex_Unlock( priv->mutex );
}

/*
 * Plugin info
 */

XMM_PluginGraph	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_GRAPH,
				0,
				XMM_VERSION_NUM,
				"",
				"X11",
				"Graph: X11",
				"Copyright (c) 1999 Arthur Kleer",
				NULL, NULL },
				x11_Init, x11_Close, x11_Control, x11_Start,
				x11_Draw, x11_Blit };

/*
 * Internal code
 */

/*
 * Resize window
 */
static int x11_Resize( XMM_PluginGraph *graph, int width, int height, int flags )
{
  struct priv_t 	*priv = graph->sys.priv;
  XSizeHints		hints;
  int			stretch = 0, ret;

  /* Lock surface */
  xmmMutex_Lock( priv->mutex );

  /*
   * Check parameters
   */
  if(( !width || !height ) && !flags )
  {
	xmmMutex_Unlock( priv->mutex );
	return xmm_SetError( graph->sys.xmm, XMM_RET_ERROR, "(X11) Wrong parameters: width = 0, height = 0 only with flags ( FULLSCREEN )" );
  }

  if(( flags & XMM_GRAPH_RESIZE_ORIG ) == XMM_GRAPH_RESIZE_ORIG )
  {
	width = priv->swidth;
	height = priv->sheight;

	priv->dxoff = 0;
	priv->dyoff = 0;
  }

  if(( flags & XMM_GRAPH_RESIZE_DOUBLE ) == XMM_GRAPH_RESIZE_DOUBLE )
  {
	width = priv->swidth << 1;
	height = priv->sheight << 1;

	priv->dxoff = 0;
	priv->dyoff = 0;
  }

  /* Manage fullscreen */
  if(( flags & XMM_GRAPH_RESIZE_FULLSCREEN ) == XMM_GRAPH_RESIZE_FULLSCREEN )
  {
	if( priv->Fullscreen )
	{
		xmmMutex_Unlock( priv->mutex );
		return xmm_SetError( graph->sys.xmm, XMM_RET_ERROR, "(X11) Already fullscreen." );
	}

	priv->dxoff = 0;
	priv->dyoff = 0;

	/* Stretch to full screen ? */
	if( !( width && height ))
	{
		width = priv->swidth;
		height = priv->sheight;
		stretch = 1;
	}

	/* Set desired screen size */
	priv->wwidth = width;
	priv->wheight = height;

	/*
	 * Initialize Fullscreen
	 */

#if defined( HAVE_XF86DGA1x ) || defined( HAVE_XF86DGA2 )
	priv->MemBase = NULL;
#endif

#ifdef HAVE_XV
	if( priv->dyuv == 0 )	/* Use DGA only if overlay not available */
	{
#endif
#ifdef HAVE_XF86DGA2
	    priv->MemBase = xf86dga2_start( priv->display, priv->screen, priv->pixelsize * 8, &priv->wwidth, &priv->wheight, &priv->PhysWidth );
#elif defined( HAVE_XF86DGA1x )
	    priv->MemBase = xf86dga_start( priv->display, priv->screen, &priv->wwidth, &priv->wheight, &priv->PhysWidth );
#endif
#ifdef HAVE_XV
	}
#endif

#if defined( HAVE_XF86DGA1x ) || defined( HAVE_XF86DGA2 )
	if( priv->MemBase == NULL )	priv->PhysWidth = 0;
#endif

#if defined( HAVE_XF86DGA1x ) || defined( HAVE_XF86DGA2 )
	if( priv->MemBase == NULL )
#endif
	{
	    XMM_ControlRect rect;

	    /* Query screen size */
	    x11_Control( graph, XMM_CTLGET_SCREEN, 0, &rect );

	    priv->wwidth = rect.width;
	    priv->wheight = rect.height;

	    /* Set position to 0,0 */
	    rect.x = 0;
	    rect.y = 0;
	    x11_Control( graph, XMM_CTLSET_POSITION, 0, &rect );
	}

	/* Stretch to full screen ? */
	if( stretch )
	{
		width = priv->wwidth;
		height = priv->sheight * width / priv->swidth;

		priv->dyoff = ( priv->wheight - height ) >> 1;
	}

	/* framebuffer width correction */
#if defined( HAVE_XF86DGA1x ) || defined( HAVE_XF86DGA2 )
	if( priv->PhysWidth )	priv->wwidth = priv->PhysWidth / priv->pixelsize;
#endif

	/* Set some data */
	priv->Fullscreen = 1;
	priv->wssize = priv->wwidth * priv->wheight * priv->pixelsize;

	/* Clear screen */
#if defined( HAVE_XF86DGA1x ) || defined( HAVE_XF86DGA2 )
	if( priv->MemBase )	memset( priv->MemBase, 0, priv->wssize );
#endif
  }
  else
  {
	if( priv->Fullscreen )
	{
#ifdef HAVE_XF86DGA2
		xf86dga2_exit( priv->display, priv->screen );
#elif defined( HAVE_XF86DGA1x )
		xf86dga_exit( priv->display, priv->screen );
#endif
		priv->Fullscreen = 0;
	}

	priv->wwidth = width;
	priv->wheight = height;
	priv->wssize = priv->wwidth * priv->wheight * priv->pixelsize;
  }

  /* Set destination size */
  priv->dwidth = width;
  priv->dheight = height;

  /*
   * Do we need resizing / Do we have Converter Plugin ?
   */

#ifdef HAVE_XV
  if( priv->dyuv == 0 )
  {
#endif
	/* Set output end of filter chain */
	ret = xmm_FilterVChainOutput( graph->sys.xmm, priv->dformat, priv->dwidth, priv->dheight, priv->gflags, priv->wwidth, priv->wheight );
#ifdef HAVE_XV
  }
  else
  {
	ret = xmm_FilterVChainOutput( graph->sys.xmm, priv->dformat, priv->swidth, priv->sheight, priv->gflags, priv->swidth, priv->sheight );
  }
#endif

  if( ret != XMM_RET_OK )	return ret;

  /*
   * Do X11 stuff
   */

  /* Disable resizing / Set WM window size */
  hints.flags = ( PSize | PMinSize | PMaxSize | PPosition );
  hints.x = 0;
  hints.y = 0;
  hints.min_width = hints.max_width = hints.width = priv->wwidth;
  hints.min_height = hints.max_height = hints.width = priv->wheight;

#if 0
  XSetWMNormalHints( priv->display, priv->window, &hints );
#endif
  XSetStandardProperties( priv->display, priv->window, XMM_PROJECT_NAME, XMM_PROJECT_NAME, None, NULL, 0, &hints );

  /* Resize window */
  XResizeWindow( priv->display, priv->window, priv->wwidth, priv->wheight );
  XSync( priv->display, True);
  XMapWindow( priv->display, priv->window );

  /* DGA Fullscreen */
  if( priv->Fullscreen )
  {
#if defined( HAVE_XF86DGA1x ) || defined( HAVE_XF86DGA2 )
	if( priv->MemBase )
	{
	    priv->image_data = priv->MemBase;
	    priv->image = NULL;

	    /* Unlock surface */
	    xmmMutex_Unlock( priv->mutex );

	    return XMM_RET_OK;
	}
#endif
  }

#ifdef HAVE_XV
  if( priv->dyuv )
  {
	if( priv->xvd.image == NULL )
	{
	    if( xv_start( priv->display, &priv->xvd, priv->dformat, priv->swidth, priv->sheight ) >= 0 )
	    {
		xmm_logging( 2, "X11! Overlay (type: %s) created using XVideo extension.\n", xmm_FOURCC_string( priv->dformat ));

		/* Unlock surface */
		xmmMutex_Unlock( priv->mutex );
		return XMM_RET_OK;
	    }
	    else
	    {
		xmmMutex_Unlock( priv->mutex );
		return xmm_SetError( graph->sys.xmm, XMM_RET_ERROR, "(X11) Error initializing XVideo." );
	    }
	}
  }
#endif /* HAVE_XV */

#ifdef HAVE_XSHM
  if( priv->xshm_mode )
  {
	/* Free old image */
	if( priv->image )	xshm_Destroy( priv->display, &priv->ShmInfo[0] );

	/* Create new one */
	if(( priv->image = xshm_Create( priv->display, priv->visual, &priv->ShmInfo[0], priv->wwidth, priv->wheight, priv->dbpp )) != NULL )
	{
	    priv->image_data = priv->ShmInfo[0].shmaddr;
	    priv->image->data = priv->image_data;

	    /* Unlock surface */
	    xmmMutex_Unlock( priv->mutex );
	    return XMM_RET_OK;
	}
	else
	{
	    xmm_logging( 1, "X11! Error initializing XShm. Disabling...\n" );
	    priv->xshm_mode = 0;
	}
  }
#endif	/* HAVE_XSHM */

  /* Free / Destroy Image ( data ) */
  if( priv->image )	XDestroyImage( priv->image );
  else	if( priv->image_data )	free( priv->image_data );

  /* Allocate image data */
  priv->image_data = malloc( priv->wssize );

  /* Create Image */
  priv->image = XCreateImage( priv->display, priv->visual, priv->dbpp, ZPixmap, 0, priv->image_data, priv->wwidth, priv->wheight, priv->pixelsize * 8, priv->wwidth * priv->pixelsize );
  priv->image->data = priv->image_data;

  /* Unlock surface */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}

/*
 * Check for Window event
 */
static int x11_CheckForEvent( XMM_PluginGraph *graph )
{
  struct priv_t *priv = graph->sys.priv;
  XEvent	event;
  XMM_Event_Key	key;

  /* Check for event */
  if( XCheckWindowEvent( priv->display, priv->window, KeyPressMask | KeyReleaseMask, &event ))
  {
	if( event.type == KeyPress )
	{
		key.type = XMM_EVENT_KEY;
		key.scode = ((XKeyEvent *)&event)->keycode - 8;
		key.state = XMM_KEY_PRESSED;

		xmm_PushEvent( graph->sys.xmm, (XMM_Event *) &key, 1 );
		return 1;
	}

	if( event.type == KeyRelease )
	{
		key.type = XMM_EVENT_KEY;
		key.scode = ((XKeyEvent *)&event)->keycode - 8;
		key.state = XMM_KEY_RELEASED;

		xmm_PushEvent( graph->sys.xmm, (XMM_Event *) &key, 1 );
		return 1;
	}
  }

  return 0;
}

/*
 * Set screen saver timeout
 */
static int x11_screen_saver( Display *display, int timeout )
{
  int interval, prefer_blank, allow_exp, toback;
#ifdef DEBUG
  int to;
#endif

  XGetScreenSaver( display, &toback, &interval, &prefer_blank, &allow_exp );

#ifdef DEBUG
  xmm_logging( 1, "X11! ScreenSaver: timeout = %i ( wanted %i ), interval = %i, blank = %i exp = %i\n",
			toback, timeout, interval, prefer_blank, allow_exp );
#endif

  if( toback != timeout )
  {
	XSetScreenSaver( display, timeout, interval, prefer_blank, allow_exp );

#ifdef DEBUG
	XGetScreenSaver( display, &to, &interval, &prefer_blank, &allow_exp );
	xmm_logging( 1, "X11! ScreenSaver(new): timeout = %i interval = %i blank = %i exp = %i\n",
			to, interval, prefer_blank, allow_exp );
#endif
  }

  return toback;
}
