/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * x11_dga1x.c ( based on 'The Guild' source [v0.90] )
 * XFree86 - Direct Graphic Access ( DGA )
 */

#include <unistd.h>
#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/xf86dga.h>
#include <X11/extensions/xf86vmode.h>

#include <libxmm/xmmp.h>

/*
 * Definitions
 */

#define	DGA_MINMAJOR	1
#define	DGA_MINMINOR	0
#define	VM_MINMAJOR	0
#define	VM_MINMINOR	8

/*
 * Global data
 */

static Window			window = 0;

static XF86VidModeModeInfo	**aVidMode = NULL;
static int			VidModes = 0;

/*
 * ( initialize and ) start X11/XF86-DGA stuff
 */

char *xf86dga_start( Display *display, int screen, int *xsize, int *ysize, int *pwidth )
{
  int		MajorVersion, MinorVersion;
  int		EventBase, ErrorBase, n;
  int		cmode, cxsize, cysize;
  char		*MemBase = NULL;
  int		BankSize, MemSize, DGAfeat = 0;

  if( window )
  {
	xmm_logging( 1, "X11(DGA)! Already in fullscreen mode.\n" );
	return NULL;
  }

  /* Check for root permissions */
  if( geteuid())
  {
	xmm_logging( 1, "X11(DGA)! Must run with root permissions.\n" );
	return NULL;
  }

  /* Get/Check XFree86-DGA version */
  if( !XF86DGAQueryVersion( display, &MajorVersion, &MinorVersion ))
  {
	xmm_logging( 1, "X11(DGA)! Cannot query DGA version.\n" );
	return NULL;
  }

  if( !XF86DGAQueryExtension( display, &EventBase, &ErrorBase )) 
  {
	xmm_logging( 1, "X11(DGA)! Cannot query DGA extension.\n" );
	return NULL;
  }

  if( !XF86DGAQueryDirectVideo( display, screen, &DGAfeat )) 
  {
	xmm_logging( 1, "X11(DGA)! Cannot query DGA direct video.\n" );
	return NULL;
  }

  if( !(DGAfeat & XF86DGADirectPresent ))
  {
	xmm_logging( 1, "X11(DGA)! No direct video support present.\n" );
	return NULL;
  }

  if( MajorVersion < DGA_MINMAJOR ||
	(( MajorVersion == DGA_MINMAJOR ) && ( MinorVersion < DGA_MINMINOR ))) 
  {
	xmm_logging( 1, "X11(DGA)! XFree86-DGA version %i.%i found.\n",
						MajorVersion, MinorVersion );
	xmm_logging( 1, "X11(DGA)! Minimum requiered XFree86-DGA version: %i.%i\n",
						DGA_MINMAJOR, DGA_MINMINOR );
	return NULL;
  }

  xmm_logging( 1, "X11(DGA)! XFree86-DGA extension version %i.%i found.\n",
						MajorVersion, MinorVersion );

  /* Get/Check XFree86-VidMode version */
  if( !XF86VidModeQueryVersion( display, &MajorVersion, &MinorVersion ))
  {
	xmm_logging( 1, "X11(DGA)! Cannot get VidMode version.\n" );
	return NULL;
  }

  if( MajorVersion < VM_MINMAJOR ||
	(( MajorVersion == VM_MINMAJOR ) && ( MinorVersion < VM_MINMINOR ))) 
  {
	xmm_logging( 1, "X11(DGA)! XFree86-VidMode version %i.%i found.\n",
						MajorVersion, MinorVersion );
	xmm_logging( 1, "X11(DGA)! Minimum requiered XFree86-VidMode version: %i.%i\n",
						VM_MINMAJOR, VM_MINMINOR );
	return NULL;
  }

  xmm_logging( 1, "X11(DGA)! XFree86-VidMode extension version %i.%i found.\n",
						MajorVersion, MinorVersion );


  /* Find/Set proper video mode */
  XF86VidModeGetAllModeLines( display, screen, &VidModes, &aVidMode );

  cmode = VidModes;
  cxsize = 0x7FFF;	/* Biggest positive value for signed short */
  cysize = 0x7FFF;

  for( n = 0; n < VidModes; n++ )
  {
	if(( aVidMode[n]->hdisplay > *xsize ) && ( aVidMode[n]->vdisplay > *ysize ))
	{
		if(( aVidMode[n]->hdisplay < cxsize ) && ( aVidMode[n]->vdisplay < cysize ))
		{
			cxsize = aVidMode[n]->hdisplay;
			cysize = aVidMode[n]->vdisplay;
			cmode = n;
		}
	}
  }

  if( cmode == VidModes )
  {
	xmm_logging( 1, "X11(DGA)! No %ix%i mode found.\n", *xsize, *ysize );
	return NULL;
  }

  xmm_logging( 1, "X11(DGA)! VidMode = %i ( %ix%i )\n",
						    cmode,
						    aVidMode[cmode]->hdisplay,
						    aVidMode[cmode]->vdisplay );

  *xsize = aVidMode[cmode]->hdisplay;
  *ysize = aVidMode[cmode]->vdisplay;

  /* Init DGA: 1. step */
  XF86DGAGetVideo( display, screen, &MemBase, pwidth, &BankSize, &MemSize );

  xmm_logging( 1, "X11(DGA)! MemBase = %p, Width = %i, BankSize = %i, Ram = %i\n",
                                      MemBase, *pwidth, BankSize, MemSize );

  if( BankSize != ( MemSize * 1024 ))
  {
	xmm_logging( 1, "X11(DGA)! Banked framebuffer not supported.\n" );
	return NULL;
  }

  /* Switch mode and Init DGA ( step 2 ) */
  if( cmode )	XF86VidModeSwitchToMode( display, screen, aVidMode[cmode] );

  XF86DGADirectVideo( display, screen,
	XF86DGADirectGraphics /* | XF86DGADirectMouse | XF86DGADirectKeyb */);

  /* Create Window */
  window = DefaultRootWindow( display );

  /* Map window */
  XMapWindow( display, window );

  /* Event handling */
  XGrabKeyboard( display, window, True, GrabModeAsync, GrabModeAsync, CurrentTime );

  /* Upper Left Corner */
  XF86DGASetViewPort( display, screen, 0, 0 );

  return MemBase;
}

/*
 * Restore X11 settings and free memory
 */

void xf86dga_exit( Display *display, int screen )
{
  int n;

  if( aVidMode )
  {
	XF86VidModeSwitchToMode( display, screen, aVidMode[0] );

	for( n = 0; n < VidModes; n++ )
		if( aVidMode[n]->privsize > 0 )	XFree( aVidMode[n]->private );

	XFree( aVidMode );
  }

  XF86DGADirectVideo( display, screen, 0 );

  XUngrabKeyboard( display, CurrentTime );
  XUnmapWindow( display, window );

  window = 0;
}
