/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * x11_xv.c
 * XVideo extension
 *
 * Note: Works only with X shared memory extension
 */

#include <stdlib.h>
#include <inttypes.h>
#include <sys/shm.h>

#include <X11/Xlib.h>
#include <X11/extensions/XShm.h>
#include <X11/extensions/Xvlib.h>
#include "x11_xv.h"

#include <libxmm/xmmp.h>
#include <libxmm/util/utils.h>

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to even more information */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/*
 * Global data
 */

/*
 * Prototypes
 */

static int xv_init( Display *display, xv_data_t *xvd );

/*
 * Get available formats
 */

int xv_initData( Display *display, xv_data_t *xvd )
{
  int		j;

  /* Initialize XV */
  if(( xvd->formats = xv_init( display, xvd )) < 0 )	return -1;

  /* Allocate formatID array */
  xvd->aFormat = malloc( sizeof(uint32_t) * xvd->formats );
  if( xvd->aFormat == NULL )	return -1;

  /* Copy formatID's */
  for( j = 0; j < xvd->formats; j++ )	xvd->aFormat[j] = xvd->ifv[j].id;

  return 0;
}

/*
 * Free XV resources
 */

void xv_exit( Display *display, xv_data_t *xvd )
{
  if( xvd->aFormat )	free( xvd->aFormat );

  XShmDetach( display, &xvd->shmInfo );
  shmdt( xvd->shmInfo.shmaddr );
  shmctl( xvd->shmInfo.shmid, IPC_RMID, 0 );
}

/*
 * start XV output
 */

int xv_start( Display *display, xv_data_t *xvd, uint32_t format, int xsize, int ysize )
{
  xvd->image = XvShmCreateImage( display, xvd->portID, format, NULL, xsize, ysize, &xvd->shmInfo );
  if( xvd->image == 0 )
  {
	xmm_logging( 1, "X11(XV)! Unable to created Xv shared memory image\n" );
	return -1;
  }

  xvd->shmInfo.shmid = shmget( IPC_PRIVATE, xvd->image->data_size, IPC_CREAT | 0777 );
  if( xvd->shmInfo.shmid < 0 )
  {
	xmm_logging( 1, "X11(XV)! Shared memory error, ( seg id error: %s )\n", strerror( errno ));
	return -1;
  }
	
  xvd->shmInfo.shmaddr = (char *) shmat( xvd->shmInfo.shmid, 0, 0 );
  if((int)( xvd->shmInfo.shmaddr = shmat( xvd->shmInfo.shmid, 0, 0 )) == -1 )
  {
	xmm_logging( 1, "X11(XV)! Shared memory error, ( address error )\n" );
	return 0;
  }
  xvd->shmInfo.readOnly = False;

  /**/
  XShmAttach( display, &xvd->shmInfo );
  XSync( display, False );

  xvd->image->data = xvd->shmInfo.shmaddr;

  return 0;
}


/*
 * Internal code
 */

/*
 * Check for the extension
 * and gets some information into global ifv
 * returns number of formats
 */

static int xv_init( Display *display, xv_data_t *xvd )
{
  unsigned int		version, revision, requestBase, eventBase, errorBase;
  XvAdaptorInfo		*ai;
  int			nAdaptors, formats, formats_ret = -1, i;
  XvImageFormatValues	*ifv;
#ifdef DEBUG
  int			j;
#endif

  if( xvd->ifv != NULL )	return xvd->formats;	/* Already initialized */

  /* Check for XVideo extension */
  if( XvQueryExtension( display, &version, &revision, &requestBase, &eventBase, &errorBase ) != Success )
  {
	xmm_logging( 1, "X11(XV)! Cannot query XVideo extension.\n" );
	return -1;
  }

  /* Check for XVideo Adaptors */
  if( XvQueryAdaptors( display, DefaultRootWindow( display ), &nAdaptors, &ai ) != Success )
  {
	xmm_logging( 1, "X11(XV)! Cannot query XVideo adaptors.\n" );
	return -1;
  }

  /* */
  for( i = 0; i < nAdaptors; i++ )
  {
#ifdef VERBOSE
	xmm_logging( 1, "X11(XV)! BaseID = %x Ports = %i Formats = %i Adaptors = %i\n", 
					    ai[i].base_id, ai[i].num_ports,
					    ai[i].num_formats,
					    ai[i].num_adaptors );

	xmm_logging( 1, "X11(XV)! Name = %s Type = %x ( %s%s%s%s%s)\n",
					    ai[i].name,
					    ai[i].type,
					    ai[i].type & XvInputMask ? "Input " : "",
					    ai[i].type & XvOutputMask ? "Output " : "",
					    ai[i].type & XvVideoMask ? "Video " : "",
					    ai[i].type & XvStillMask ? "Still " : "",
					    ai[i].type & XvImageMask ? "Image " : "" );
					    
	for( j = 0; j < ai[i].num_formats; j++ )
	{
	    xmm_logging( 1, "X11(XV)! Depth = %i VisualID = %x\n",
					    ai[i].formats[j].depth,
					    ai[i].formats[j].visual_id );
	}


#endif
	/* Get formats */
	ifv = XvListImageFormats( display, ai[i].base_id, &formats );
	if( ifv == NULL )
	{
	    xmm_logging( 1, "X11(XV)! Error calling XvListImageFormats()\n" );
	    return -1;
	}

	if(( xvd->ifv == NULL ) && ( ai[i].type & XvInputMask ) && ( ai[i].type & XvImageMask ))
	{
	    xvd->ifv = ifv;
	    xvd->portID = ai[i].base_id;
	    formats_ret = formats;
	}

#ifdef DEBUG

	for( j = 0; j < formats; j++ )
	{
	    xmm_logging( 1, "X11(XV)! ID = 0x%x (%s)\n", ifv[j].id, (char *) &ifv[j].id );
	    xmm_logging( 1, "\tguid: %s\n", ifv[j].guid );
	    xmm_logging( 1, "\tbits per pixel: %i number of planes: %i byte order: %s\n",
				ifv[j].bits_per_pixel, ifv[j].num_planes,
				( ifv[j].byte_order == LSBFirst ) ? "LSB" :
				(( ifv[j].byte_order == MSBFirst ) ? "MSB" :
				"unknown" ));
	    xmm_logging( 1, "\ttype: %s (%s)\n",
				( ifv[j].type == XvRGB ) ? "RGB" :
				(( ifv[j].type == XvYUV ) ? "YUV" :
				"unknown" ),
				( ifv[j].format == XvPacked ) ? "Packed" :
				(( ifv[j].format == XvPlanar ) ? "Planar" :
				"unknown" ));

	    if( ifv[j].type == XvRGB )
	    {
		xmm_logging( 1, "\ttype specific: depth = %i rmask = %x gmask = %x bmask = %x\n",
					ifv[j].depth, ifv[j].red_mask,
					ifv[j].green_mask, ifv[j].blue_mask );
	    }
	    else if( ifv[j].type == XvYUV )
	    {
		xmm_logging( 1, "\ttype specific: order: %s (%s) bits: %i,%i,%i\n",
			ifv[j].component_order,
			( ifv[j].scanline_order == XvTopToBottom ) ? "Top to Bottom" :
			(( ifv[j].scanline_order == MSBFirst ) ? "Bottom to Top" :
			"unknown" ),
			ifv[j].y_sample_bits, ifv[j].u_sample_bits,
			ifv[j].v_sample_bits );
		xmm_logging( 1, "\ttype specific: H-Period: %i,%i,%i V-Period: %i,%i,%i\n",
			ifv[j].horz_y_period, ifv[j].horz_u_period,
			ifv[j].horz_v_period, ifv[j].vert_y_period,
			ifv[j].vert_u_period, ifv[j].vert_v_period );
	    }
	}
#endif

  }

  /* Free adaptor info */
  XvFreeAdaptorInfo( ai );

  return formats_ret;
}
