/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * aiff.c
 * Audio IFF DeMUX
 *
 * TODO: READ_IEEE_EXT() only a hack. Some tests should be added
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/endian.h>
#include <libxmm/util/utils.h>
#include "aiff.h"

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#define	VERBOSE
#endif
 
/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_AudioInfo		ai;

    /* Clip info */
    XMM_ClipInfo		ci;

    char 			*filename;
    int				seekable;
    uint32_t			data_offset, data_size;
    int				byterate;

    /* AIFF headers */
    comm_header_t		comm;
    ssnd_header_t		ssnd;
    mark_header_t		mark;
    inst_header_t		inst;
    comt_header_t		comt;

    marker_t			*marker;
};

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Prototypes
 */

static uint32_t	READn( XMM_PluginIO *pIO, void *data, uint32_t size );
static uint32_t	READ32( XMM_PluginIO *pIO );
static uint16_t	READ16( XMM_PluginIO *pIO );
static uint8_t	READ8( XMM_PluginIO *pIO );
static uint32_t READ_IEEE_EXT( XMM_PluginIO *pIO );
static int	SEEK( XMM_PluginIO *pIO, long offset, int seekable );


static int Read_comm( XMM_PluginIO *pIO, uint32_t size, comm_header_t *comm );
static int Read_ssnd( XMM_PluginIO *pIO, uint32_t size, ssnd_header_t *ssnd );
static int Read_mark( XMM_PluginIO *pIO, uint32_t size, mark_header_t *mark, marker_t **marker );
static int Read_inst( XMM_PluginIO *pIO, uint32_t size, inst_header_t *inst );
static int Read_comt( XMM_PluginIO *pIO, uint32_t size, comt_header_t *comt );
static int Read_text_chunk( XMM_PluginIO *pIO, XMM_ClipInfo *ci, uint32_t dwName, uint32_t dwSize );

/*
 * Initialize
 */
static XMM_PluginInput *aiff_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(AIFF) plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int aiff_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  uint32_t		dwSize, dwName, dwTemp;
  char			*buffer, *ptr;
  int			loop = 1;

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(AIFF) Only 'read' mode supported" );

  /* Open file */
  priv->pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->pIO == NULL )	return XMM_RET_ERROR;

  /* Seekable ? */
  if( priv->pIO->Seek( priv->pIO, 0, XMM_SEEK_CUR ) == XMM_RET_OK )
	priv->seekable = 1;
  else	priv->seekable = 0;
#ifdef VERBOSE
  xmm_logging( 2, "AIFF! Stream is %s seekable\n", priv->seekable ? "" : "NOT" );
#endif

  /* Clip info */
  memset( &priv->ci, 0, sizeof( XMM_ClipInfo ));

  /* Parse file */
  while( loop )
  {
	dwName = READ32( priv->pIO );
	if( priv->pIO->Eof( priv->pIO ))	break;

	dwSize = READ32( priv->pIO );

#ifdef VERBOSE
	xmm_logging( 2, "AIFF! Chunk ID = %x [%s] Size = %x ( %li )\n", dwName, xmm_FOURCC_string( XMM_INT32_BE(dwName)), dwSize, dwSize );
#endif

	switch( dwName )
	{
		case FORM_ID_FORM:
			dwTemp = READ32( priv->pIO );
			if( dwTemp != FORM_ID_AIFF )
			{
				priv->pIO->Close( priv->pIO );
				return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(AIFF) No AIFF form ( '%s' )", xmm_FOURCC_string( XMM_INT32_BE( dwTemp )));
			}
			xmm_logging( 2, "AIFF! FORM type = '%s'\n", xmm_FOURCC_string( XMM_INT32_BE(dwTemp)));
			break;

		case FORM_ID_COMM:
			Read_comm( priv->pIO, dwSize, &priv->comm );
			break;

		case FORM_ID_SSND:
			Read_ssnd( priv->pIO, dwSize, &priv->ssnd );

			priv->data_offset = priv->pIO->Tell( priv->pIO );
			priv->data_size = dwSize - 2 * sizeof(uint32_t) - sizeof(ssnd_header_t);

			if(( priv->data_size + priv->pIO->Tell( priv->pIO )) > priv->pIO->Size( priv->pIO ))
				priv->data_size = priv->pIO->Size( priv->pIO ) - priv->pIO->Tell( priv->pIO );

			if( priv->seekable )	priv->pIO->Seek( priv->pIO, priv->data_size, XMM_SEEK_CUR );
			else	loop = 0;

			xmm_logging( 2, "AIFF! data_offset = %li ( %lx )\n", priv->data_offset, priv->data_offset );
			break;

			break;

		case FORM_ID_MARK:
			Read_mark( priv->pIO, dwSize, &priv->mark, NULL );
			break;

		case FORM_ID_INST:
			Read_inst( priv->pIO, dwSize, &priv->inst );
			break;

		case FORM_ID_COMT:
			Read_comt( priv->pIO, dwSize, &priv->comt );
			break;

		/* INFO chunks */
		case FORM_ID_NAME:
		case FORM_ID_AUTH:
		case FORM_ID_COPY:
		case FORM_ID_DATE:
			Read_text_chunk( priv->pIO, &priv->ci, dwName, dwSize );
			break;

		case FORM_ID_AESD:
		case FORM_ID_MIDI:
		case FORM_ID_APPL:

		default:
			xmm_logging( 2, "AIFF! Ignoring chunk %x [%s]\n", dwName, xmm_FOURCC_string(XMM_INT32_BE(dwName)));
			SEEK( priv->pIO, dwSize, priv->seekable );
			break;
	}
  }

  /* Seek to the beginning of data */
  if( priv->seekable )	priv->pIO->Seek( priv->pIO, priv->data_offset, XMM_SEEK_SET );

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* AudioInfo struct */
  priv->ai.fmt.format = XMM_AUDIO_CODEC_PCM | XMM_AUDIO_MASK_SIGNED;
  priv->ai.fmt.format |= ( priv->comm.wSampleSize & XMM_AUDIO_MASK_SIZE );
  if( priv->comm.wSampleSize > 8 )	priv->ai.fmt.format |= XMM_AUDIO_MASK_BE;

  priv->ai.fmt.samprate = priv->comm.dwSampleRate;
  priv->ai.fmt.channels = priv->comm.wChannels;
  priv->ai.fmt.blockSize = DivideUP( priv->comm.wSampleSize, 8 ) * priv->comm.wChannels;
  priv->ai.fmt.bitrate = priv->ai.fmt.blockSize * priv->ai.fmt.samprate * 8;
  priv->ai.fmt.extraSize = 0;
  priv->ai.fmt.desc[0] = '\0';
  priv->ai.tSamples = (uint32_t)(((double)priv->data_size /
		( priv->ai.fmt.bitrate / 8 )) * priv->ai.fmt.samprate );
  priv->ai.tSize = priv->data_size;
  priv->ai.offset = 0;

  /* Byte rate */
  priv->byterate = priv->ai.fmt.samprate * priv->ai.fmt.blockSize;

  /* Clip info */
  priv->ci.size = priv->pIO->Size( priv->pIO );
  priv->ci.playtime = (double)priv->ai.tSamples / priv->ai.fmt.samprate;

  if( priv->ci.content[0] == '\0' )
  {
	strncpy( priv->ci.content, priv->filename, XMM_CIL_CONTENT - 1 );
	priv->ci.content[XMM_CIL_CONTENT - 1] = '\0';
  }

  return XMM_RET_OK;
}

/*
 * Close
 */
static int aiff_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  if( priv->filename )	free( priv->filename );
  if( priv->pIO )	priv->pIO->Close( priv->pIO );
  if( priv->marker )	free( priv->marker );
  free( input );

  return XMM_RET_OK;
}

/*
 * Control
 */
static int aiff_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_AudioFormat	*format;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_AFORMAT:			/* Query sound format */
		format = (XMM_AudioFormat *)data;
		if(( format->format == priv->ai.fmt.format ) &&
		    ( format->samprate == priv->ai.fmt.samprate ) &&
		    ( format->channels == priv->ai.fmt.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_AFORMAT:			/* Get sound format */
		memcpy( data, &priv->ai.fmt, sizeof( XMM_AudioFormat ));
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->ai.fmt;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_CAPS:				/* Get capabilities */
		*((uint32_t *) data ) = XMM_INPUT_CF_MODE_READ | XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(AIFF) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AIFF) cmd ( 0x%x )" );
}

/*
 * Seek to position
 */
static int aiff_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  struct priv_t	*priv = input->sys.priv;

  return input->AudioSeek( input, astream, (uint32_t)( seek * priv->ai.tSize / priv->byterate * 1000 ));
}

/*
 * Get Information
 */
static int aiff_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )	memcpy( ci, &priv->ci, sizeof( XMM_ClipInfo ));

  return XMM_RET_OK;
}

/*
 * Get audio stream number
 */
static int aiff_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int aiff_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int aiff_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  struct priv_t *priv = input->sys.priv;
  uint32_t	read;

  if(( priv->pIO->Tell( priv->pIO ) + size ) >= ( priv->data_offset + priv->data_size ))
  {
    size = priv->data_offset + priv->data_size - priv->pIO->Tell( priv->pIO );
    if( size <= 0 )	return XMM_RET_EOS;
  }

  read = priv->pIO->Read( priv->pIO, buffer, 1, size );
  if(( read == 0 ) && priv->pIO->Tell( priv->pIO ))	return XMM_RET_EOS;

  return read;
}

/*
 * Audio timestamp.
 */
static int aiff_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  struct priv_t	*priv = input->sys.priv;

  if( pts )	*pts = (uint32_t)((double)( priv->pIO->Tell( priv->pIO ) -
				priv->data_offset ) / priv->byterate * 1000 );

  return XMM_RET_OK;
}

/*
 * Seek to position in audio stream
 */
static int aiff_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts )
{
  struct priv_t	*priv = input->sys.priv;
  double	bytes;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(AIFF) Stream is not seekable." );

  bytes = (double)pts * priv->byterate / 1000;

  /* I/O Seek */
  return priv->pIO->Seek( priv->pIO, priv->data_offset + (uint32_t) bytes, XMM_SEEK_SET );
}

/*
 * AIFF files do not contain video data
 */
static int aiff_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int aiff_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int aiff_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  return XMM_RET_NOTSUPPORTED;
}

static int aiff_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int aiff_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if WAV file
 */
static int aiff_Check( void *xmm, char *filename )
{
  uint32_t		dwSize, dwName, dwTemp;
  XMM_PluginIO		*pIO;

  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  dwName = READ32( pIO );
  dwSize = READ32( pIO );
  dwTemp = READ32( pIO );

  pIO->Close( pIO );

  if(( dwName == FORM_ID_FORM ) && ( dwTemp == FORM_ID_AIFF ))	return 1;
  else	return 0;
}

/*
 * Get file info
 */
static int aiff_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */
XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"AIFF",
				"Input: Audio IFF",
				"Copyright (c) 2002 Arthur Kleer",
				NULL, NULL },
				aiff_Init, aiff_Open, aiff_Close,
				aiff_Control, aiff_Seek, aiff_Info, 
				aiff_AudioStreams, aiff_AudioInfo,
				aiff_AudioRead, aiff_AudioPTS, aiff_AudioSeek,
				aiff_VideoStreams, aiff_VideoInfo,
				aiff_VideoRead, aiff_VideoPTS, aiff_VideoSeek,
				aiff_Check, aiff_FileInfo };

/*
 * Internal code
 */

/*
 * Misc functions for file i/o
 */

static uint32_t READn( XMM_PluginIO *pIO, void *data, uint32_t size )
{
  return pIO->Read( pIO, data, size, 1 );
}

static uint32_t READ32( XMM_PluginIO *pIO )
{
  uint32_t tmp;

  pIO->Read( pIO, &tmp, 4, 1 );

  return XMM_INT32_BE( tmp );
}

static uint16_t READ16( XMM_PluginIO *pIO )
{
  uint16_t tmp;

  pIO->Read( pIO, &tmp, 2, 1 );

  return XMM_INT16_BE( tmp );
}

static uint8_t READ8( XMM_PluginIO *pIO )
{
  uint8_t tmp;

  pIO->Read( pIO, &tmp, 1, 1 );

  return tmp;
}

static uint32_t READ_IEEE_EXT( XMM_PluginIO *pIO )
{
  uint32_t	tmp;
  uint8_t	buffer[10];

  pIO->Read( pIO, buffer, 1, 10 );

  /* 1 bit: sign, 15 bit: exponent, 64 bit: mantissa */
  tmp = ( buffer[2] << 24 ) | ( buffer[3] << 16 ) | ( buffer[4] << 8 ) | buffer[0];
  tmp >>= ( 30 - buffer[1] );
	
  return tmp;
}

static int SEEK( XMM_PluginIO *pIO, long offset, int seekable )
{
  if( seekable )	return pIO->Seek( pIO, offset, XMM_SEEK_CUR );
  else
  {
	uint8_t tmp;
	int i;

	for( i = 0; i < offset; i++ )	pIO->Read( pIO, &tmp, 1, 1 );
	return i;
  }
}

/*
 * Read COMM header
 */
static int Read_comm( XMM_PluginIO *pIO, uint32_t size, comm_header_t *comm )
{
  if( size < 0x12 )
  {
	xmm_logging( 1, "AIFF! comm: FILE ERROR: Size < 0x12 bytes ( is %li )\n", size );
	return -1;
  }

  comm->wChannels = READ16( pIO );
  comm->dwFrames = READ32( pIO );
  comm->wSampleSize = READ16( pIO );
  comm->dwSampleRate = READ_IEEE_EXT( pIO );

  for( size -= 0x12; size; size-- )	READ8( pIO );	/* Padding bytes */

#ifdef VERBOSE
  xmm_logging( 1, "AIFF! comm: wChannels = %i\n", comm->wChannels );
  xmm_logging( 1, "AIFF! comm: dwFrames = %li\n", comm->dwFrames );
  xmm_logging( 1, "AIFF! comm: wSampleSize = %i\n", comm->wSampleSize );
  xmm_logging( 1, "AIFF! comm: dwSampleRate = %li\n", comm->dwSampleRate );
#endif

  return 0;
}

/*
 * Read SSND header
 */
static int Read_ssnd( XMM_PluginIO *pIO, uint32_t size, ssnd_header_t *ssnd )
{
  ssnd->dwOffset = READ32( pIO );
  ssnd->dwBlockSize = READ32( pIO );

#ifdef VERBOSE
  xmm_logging( 1, "AIFF! ssnd: dwOffset = %li\n", ssnd->dwOffset );
  xmm_logging( 1, "AIFF! ssnd: dwBlockSize = %li\n", ssnd->dwBlockSize );
#endif

  return 0;
}

/*
 * Read MARK header
 */
static int Read_mark( XMM_PluginIO *pIO, uint32_t size, mark_header_t *mark, marker_t **marker )
{
  int		i, j, len;
  marker_t	*pMarker;

  mark->wNumber = READ16( pIO );
  if( mark->wNumber == 0 )	return 0;

#ifdef VERBOSE
  xmm_logging( 1, "AIFF! mark: wNumber = %i\n", mark->wNumber );
#endif

  if(( pMarker = malloc( sizeof( marker_t ) * mark->wNumber )) == NULL )
	return xmm_SetError( pIO->sys.xmm, XMM_RET_ALLOC, __FUNCTION__ "() %i markers", mark->wNumber );

  for( i = 0; i < mark->wNumber; i++ )
  {
	pMarker[i].wID = READ16( pIO );
	pMarker[i].dwPosition = READ32( pIO );
	len = READ8( pIO );

	for( j = 0; j < len; j++ )	pMarker[i].pcName[j] = READ8( pIO );
	pMarker[i].pcName[j] = '\0';

#ifdef VERBOSE
  xmm_logging( 1, "AIFF! marker[%i]: wID = %i\n", i, pMarker[i].wID );
  xmm_logging( 1, "AIFF! marker[%i]: dwPosition = %li\n", i, pMarker[i].dwPosition );
  xmm_logging( 1, "AIFF! marker[%i]: pcName = %s\n", i, pMarker[i].pcName );
#endif
  }

  *marker = pMarker;
  return 0;
}

/*
 * Read INST header
 */
static int Read_inst( XMM_PluginIO *pIO, uint32_t size, inst_header_t *inst )
{
  if( size != 0x14 )
  {
	xmm_logging( 1, "AIFF! inst: FILE ERROR: Size != 0x14 bytes ( is %li )\n", size );
	return -1;
  }

  inst->bBaseNote = READ8( pIO );
  inst->cDetune = READ8( pIO );
  inst->bLowNote = READ8( pIO );
  inst->bHighNote = READ8( pIO );
  inst->bLowVelocity = READ8( pIO );
  inst->bHighVelocity = READ8( pIO );
  inst->sGain = READ16( pIO );
  inst->sustainLoop.wPlayMode = READ16( pIO );
  inst->sustainLoop.wLoopBegin = READ16( pIO );
  inst->sustainLoop.wLoopEnd = READ16( pIO );
  inst->releaseLoop.wPlayMode = READ16( pIO );
  inst->releaseLoop.wLoopBegin = READ16( pIO );
  inst->releaseLoop.wLoopEnd = READ16( pIO );

#ifdef VERBOSE
  xmm_logging( 1, "AIFF! inst: bBaseNote = %i\n", inst->bBaseNote );
  xmm_logging( 1, "AIFF! inst: cDetune = %i\n", inst->cDetune );
  xmm_logging( 1, "AIFF! inst: bLowNote = %i\n", inst->bLowNote );
  xmm_logging( 1, "AIFF! inst: bHighNote = %i\n", inst->bHighNote );
  xmm_logging( 1, "AIFF! inst: bLowVelocity = %i\n", inst->bLowVelocity );
  xmm_logging( 1, "AIFF! inst: bHighVelocity = %i\n", inst->bHighVelocity );
  xmm_logging( 1, "AIFF! inst: sGain = %i\n", inst->sGain );
  xmm_logging( 1, "AIFF! inst: sustainLoop.wPlayMode = %i\n", inst->sustainLoop.wPlayMode );
  xmm_logging( 1, "AIFF! inst: sustainLoop.wLoopBegin = %i\n", inst->sustainLoop.wLoopBegin );
  xmm_logging( 1, "AIFF! inst: sustainLoop.wLoopEnd = %i\n", inst->sustainLoop.wLoopEnd );
  xmm_logging( 1, "AIFF! inst: releaseLoop.wPlayMode = %i\n", inst->releaseLoop.wPlayMode );
  xmm_logging( 1, "AIFF! inst: releaseLoop.wLoopBegin = %i\n", inst->releaseLoop.wLoopBegin );
  xmm_logging( 1, "AIFF! inst: releaseLoop.wLoopEnd = %i\n", inst->releaseLoop.wLoopEnd );
#endif

  return 0;
}

/*
 * Read COMT header
 */
static int Read_comt( XMM_PluginIO *pIO, uint32_t size, comt_header_t *comt )
{
  int		i, j;
  comment_t	comment;

  comt->wNumber = READ16( pIO );

#ifdef VERBOSE
  xmm_logging( 1, "AIFF! comt: wNumber = %i\n", comt->wNumber );
#endif

  for( i = 0; i < comt->wNumber; i++ )
  {
	comment.dwTimeStamp = READ32( pIO );
	comment.wMarkerID = READ16( pIO );
	comment.wSize = READ16( pIO );
	for( j = 0; j < comment.wSize; j++ )	READ8( pIO );	/* Skip */

#ifdef VERBOSE
  xmm_logging( 1, "AIFF! comment[%i]: dwTimeStamp = %li\n", comment.dwTimeStamp );
  xmm_logging( 1, "AIFF! comment[%i]: wMarkerID = %i\n", comment.wMarkerID );
  xmm_logging( 1, "AIFF! comment[%i]: wSize = %i\n", comment.wSize );
#endif
  }

  return 0;
}

/*
 * Read text chunks
 */
static int Read_text_chunk( XMM_PluginIO *pIO, XMM_ClipInfo *ci, uint32_t dwName, uint32_t dwSize )
{
  char	info_buffer[4096];

  pIO->Read( pIO, info_buffer, dwSize, 1 );

  switch( dwName )
  {
	case FORM_ID_NAME:
		strncpy( ci->name, info_buffer, XMM_CIL_NAME - 1 );
		ci->name[XMM_CIL_NAME - 1] = '\0';
		break;

	case FORM_ID_AUTH:
		strncpy( ci->author, info_buffer, XMM_CIL_AUTHOR - 1 );
		ci->author[XMM_CIL_AUTHOR - 1] = '\0';
		break;

	case FORM_ID_COPY:
		strncpy( ci->copyright, info_buffer, XMM_CIL_COPYRIGHT - 1 );
		ci->copyright[XMM_CIL_COPYRIGHT - 1] = '\0';
		break;

	default:
		xmm_logging( 2, "AVI! Ignoring INFO chunk 0x%x [%s] [%li bytes]\n", dwName, xmm_FOURCC_string(XMM_INT32_BE(dwName)), dwSize );
		break;
  }

  return 1;
}
