/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * asf.c
 * ASF input plugin
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include <libxmm/util/mutex.h>
#include "asf.h"
#include "asfhead.h"

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/* Change 'undef' in 'define' to use IO mutex*/
#ifndef	ASF_USE_MUTEX_IO
#define	ASF_USE_MUTEX_IO
#endif

/* Change 'undef' in 'define' to use buffer queue mutex*/
#ifndef	ASF_USE_MUTEX_STREAM
#define	ASF_USE_MUTEX_STREAM
#endif

/* Change 'undef' in 'define' to debug seek data  */
#ifndef	ASF_DEBUG_SEEK_DATA
#define	ASF_DEBUG_SEEK_DATA	2
#endif

/* */
#define	MAX_STREAMS		0x10
#define	ASF_SEEK_BLOCK		4096

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Types
 */

typedef struct
{
	uint32_t		pts;
#define	ASF_SSI_PIDMASK_PID	0x7FFFFFFF
#define	ASF_SSI_PIDMASK_KEY	0x80000000
	uint32_t		pid;

} seek_entry_t;

typedef struct 
{
	uint32_t		rbuffer_fragmentOffset;
	uint32_t		entries;	/* used seek entries */
	uint32_t		size;		/* allocated seek entries */
	seek_entry_t		*se;		/* seek entries data */
	uint32_t		tssize;		/* total stream size */
} seek_stream_info_t;

/*
 * Private plugin data
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_FileInfo		fi;

    /* Clip info */
    XMM_ClipInfo		ci;

    /* Misc stuff */
    char 			*filename;
    int				seekable;
    uint32_t			data_offset;
    uint32_t			data_size;

    /* Stream stuff */
    int				nStreams, astreams, vstreams;
    int				asIDX[MAX_STREAMS];
    int				vsIDX[MAX_STREAMS];

    asf_header_t		ahh;
    asf_file_t			afh;
    asf_data_t			adh;

    uint8_t			*packet;

#ifdef ASF_USE_MUTEX_IO
    /* Mutex */
    XMM_Mutex			*mutex_io;
#endif

    /* Video / Audio streams */
    struct stream_t
    {
	/* Stream header */
	asf_stream_t			ash;

	/* Video stream */
	asf_video_type_t		avth;

	XMM_AudioFormat			*af;
	XMM_VideoFormat			*vf;

	/* Audio stream */
	asf_audio_spread_t		aash;
	WAVEFORMATEX			*wf;

	/* PTS */
	uint32_t			pts;

	/* Index stuff */
	uint32_t			idx1_entries;	/* number of idx1 entries */
	uint32_t			position;	/* Current position */

	/* buffering */
	XMM_BufferQueue			bq;

	/* Seek data */
	seek_stream_info_t		ssi;

#ifdef ASF_USE_MUTEX_STREAM
	/* Mutex */
	XMM_Mutex			*mutex;
#endif

	/* Read buffer */
	char				*rbuffer;
	uint32_t			rbuffer_fragmentOffset;
	int				rbuffersize;

    } 					aStream[MAX_STREAMS];
};

/*
 * Prototypes
 */

static int asf_AudioRead_idx( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags );
static int asf_AudioSeek_idx( XMM_PluginInput *input, int stream, uint32_t pts );
static int asf_VideoRead_idx( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags );
static int asf_VideoSeek_idx( XMM_PluginInput *input, int stream, uint32_t frame );

static struct stream_t *asf_lookup_astream( XMM_PluginInput *input, int stream );
static struct stream_t *asf_lookup_vstream( XMM_PluginInput *input, int stream );
static uint32_t codec_ms2xmm( uint16_t ms_codec );
static int read_asf_data_raw( struct priv_t *priv, struct stream_t *ps );
static int create_asf_seek_data( struct priv_t *priv );

/*
 * Initialize
 */
static XMM_PluginInput *asf_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(ASF) plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int asf_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  char			*buffer, *ptr;
  int			loop = 1, idx1_build = 1, idx1_found = 0;
  int			skip, used, chunk, i, j;
  asf_chunk_t		ach;
  asf_stream_t		ash;
  asf_content_t		amdh;
  struct stream_t	*ps;
  seek_stream_info_t	*ssi;

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(ASF) Only 'read' mode supported" );

  /* Open file */
  priv->pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->pIO == NULL )	return XMM_RET_ERROR;

  /* Seekable ? */
  if( priv->pIO->Seek( priv->pIO, 0, XMM_SEEK_CUR ) == XMM_RET_OK )
	priv->seekable = 1;
  else	priv->seekable = 0;
#ifdef VERBOSE
  xmm_logging( 2, "ASF! Stream is %sseekable\n", priv->seekable ? "" : "NOT " );
#endif

  /* Clip info */
  memset( &priv->ci, 0, sizeof( XMM_ClipInfo ));

  /* Init data */
  priv->data_offset = 0;
  priv->data_size = 0;

  /* Parse file */
  while( loop )
  {
	/* Read chunk header */
	used = read_asf_chunk( priv->pIO, &ach, 24 );
	if( used == -1 )	break;

	/* Will be skipped later */
	skip = ach.size - used;

	/* Get chunk type */
	for( chunk = 0; asf_guids[chunk].guid.v1; chunk++ )
	{
	    if( !memcmp( &ach.guid, &asf_guids[chunk].guid, sizeof( GUID_t )))
		    break;
	}

#ifdef VERBOSE
	print_guid( &ach.guid );
#endif

	switch( chunk )
	{
	    /* ASF 2.0 header */
	    case ASF_CHUNK_HEADER_ASF2:
		    xmm_logging( 1, "ASF! ASF v2.0 (public spec 1.0) files are not supported.\n" );
		    break;

	    /* ASF header */
	    case ASF_CHUNK_HEADER:
		    used = read_asf_header( priv->pIO, &priv->ahh, skip );
		    skip -= used;
		    skip = 0;
		    break;

	    /* ASF file header */
	    case ASF_CHUNK_FILE:
		    used = read_asf_file( priv->pIO, &priv->afh, skip );
		    skip -= used;
		    break;

	    /* ASF stream header */
	    case ASF_CHUNK_STREAM:
		    used = read_asf_stream( priv->pIO, &ash, skip );
		    skip -= used;

		    /* */
		    ps = &priv->aStream[ ash.stream - 1 ];
		    memcpy( &ps->ash, &ash, sizeof( asf_stream_t ));

		    /* Initialize data */
		    ps->rbuffersize = priv->afh.maxBitrate;
		    ps->rbuffer = malloc( ps->rbuffersize );

		    /* Initialize buffer queue */
		    xmmBQ_Init( &ps->bq );

#ifdef ASF_USE_MUTEX_STREAM
		    /* Create Mutex */
		    ps->mutex = xmmMutex_Create();
#endif

		    /* set start time */
		    ps->pts = priv->afh.startTime;

		    /* Video header */
		    if( !memcmp( &ash.type, &asf_guids[ASF_CHUNK_STREAM_VIDEO].guid, sizeof( GUID_t )))
		    {
#ifdef VERBOSE
			print_guid( &ash.type );
#endif

			/* Get type specific data */
			used = read_asf_video_type( priv->pIO, &ps->avth, skip );
			skip -= used;

			/* Allocate memory */
			if(( ps->vf = malloc( sizeof( XMM_VideoFormat ))) == NULL )
			    return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(ASF) Unable to allocate VideoFormat" );

			/* Fill video format struct */
			ps->vf->codec = ps->avth.bih.biCompression;
			ps->vf->width = ps->avth.bih.biWidth;
			ps->vf->height = ps->avth.bih.biHeight;
			ps->vf->planes = ps->avth.bih.biPlanes;
			ps->vf->bpp = ps->avth.bih.biBitCount;
			ps->vf->imgSize = ps->avth.bih.biSizeImage;
			ps->vf->bitrate = 0;
			ps->vf->framerate = 30;	/* TODO */
			ps->vf->aspect_val = 1;
			ps->vf->aspect_div = 1;
			ps->vf->extraSize = 0;
			ps->vf->extraType = 0;
			ps->vf->desc[0] = '\0';

			/* VideoInfo */
			memcpy( &priv->fi.vi[priv->fi.vstreams].fmt, ps->vf, sizeof( XMM_VideoFormat ));
			priv->fi.vi[priv->fi.vstreams].tFrames = 0;
			priv->fi.vi[priv->fi.vstreams].tSize = 0;
			priv->fi.vi[priv->fi.vstreams].offset = 0;

			/* Add stream to video stream table */
			priv->vsIDX[priv->fi.vstreams++] = ash.stream - 1;
			priv->nStreams++;
		    }

		    /* Audio header */
		    if( !memcmp( &ash.type, &asf_guids[ASF_CHUNK_STREAM_AUDIO].guid, sizeof( GUID_t )))
		    {
#ifdef VERBOSE
			print_guid( &ash.type );
#endif

			ps->wf = malloc( ash.typeSize );
			priv->pIO->Read( priv->pIO, ps->wf, ash.typeSize, 1 );
			skip -= ash.typeSize;

			if( ps->wf->wBitsPerSample == 0 )
				ps->wf->wBitsPerSample = 0x10;

#ifdef VERBOSE
			xmm_logging( 2, "ASF! wfx: wFormatTag = 0x%x\n", ps->wf->wFormatTag );
			xmm_logging( 2, "ASF! wfx: nChannels = %i\n", ps->wf->nChannels );
			xmm_logging( 2, "ASF! wfx: nSamplesPerSec = %li\n",  ps->wf->nSamplesPerSec );
			xmm_logging( 2, "ASF! wfx: nAvgBytesPerSec = %li\n", ps->wf->nAvgBytesPerSec );
			xmm_logging( 2, "ASF! wfx: nBlockAlign = %i\n", ps->wf->nBlockAlign );
			xmm_logging( 2, "ASF! wfx: wBitsPerSample = %i\n", ps->wf->wBitsPerSample );
			xmm_logging( 2, "ASF! wfx: cbSize = %i \n", ps->wf->cbSize );
#endif

			/* Get stream specific data */
			used = read_asf_audio_spread( priv->pIO, &ps->aash, skip );
			skip -= used;

			/* Allocate memory */
			if(( ps->af = malloc( sizeof( XMM_AudioFormat ) + ps->wf->cbSize )) == NULL )
			    return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(ASF) Unable to allocate AudioFormat" );

			/* Fill audio format struct */
			ps->af->format = codec_ms2xmm( ps->wf->wFormatTag );
			ps->af->format |= ( ps->wf->wBitsPerSample & XMM_AUDIO_MASK_SIZE );

			if(( ps->wf->wFormatTag == WAVE_FORMAT_IEEE_FLOAT ) ||
			    (( ps->wf->wBitsPerSample != 8 )))
				ps->af->format |= XMM_AUDIO_MASK_SIGNED;

			ps->af->samprate = ps->wf->nSamplesPerSec;
			ps->af->channels = ps->wf->nChannels;
			ps->af->bitrate = ps->wf->nAvgBytesPerSec * 8;
			ps->af->blockSize = ps->wf->nBlockAlign;
			ps->af->extraSize = ps->wf->cbSize;
			ps->af->extraType = XMM_AUDIO_EXT_WAVE;
			ps->af->desc[0] = '\0';

			if( ps->af->extraSize )
		    	    memcpy( &ps->af[1], &ps->wf[1], ps->af->extraSize );

			/* AudioInfo struct */
			memcpy( &priv->fi.ai[priv->fi.astreams].fmt, ps->af, sizeof( XMM_AudioFormat ));
			priv->fi.ai[priv->fi.astreams].fmt.extraSize = 0;
			priv->fi.ai[priv->fi.astreams].tSamples = 0; /* TODO */
			priv->fi.ai[priv->fi.astreams].tSize = 0; /* TODO */
			priv->fi.ai[priv->fi.astreams].offset = 0;

			/* Add stream to audio stream table */
			priv->asIDX[priv->fi.astreams++] = ash.stream - 1;
			priv->nStreams++;
		    }
		    break;

	    /* ASF content chunk */
	    case ASF_CHUNK_CONTENT:
		    used = read_asf_content( priv->pIO, &amdh, skip );
		    skip -= used;

		    strncpy( priv->ci.content, amdh.desc, XMM_CIL_CONTENT - 1 );
		    priv->ci.content[XMM_CIL_CONTENT - 1] = '\0';
		    strncpy( priv->ci.name, amdh.title, XMM_CIL_NAME - 1 );
		    priv->ci.name[XMM_CIL_NAME - 1] = '\0';
		    strncpy( priv->ci.author, amdh.author, XMM_CIL_AUTHOR - 1 );
		    priv->ci.author[XMM_CIL_AUTHOR - 1] = '\0';
		    strncpy( priv->ci.copyright, amdh.copyright, XMM_CIL_COPYRIGHT - 1 );
		    priv->ci.copyright[XMM_CIL_COPYRIGHT - 1] = '\0';
		    break;

	    /* ASF data chunk */
	    case ASF_CHUNK_DATA:
		    used = read_asf_data( priv->pIO, &priv->adh, skip );
		    skip -= used;

		    priv->data_offset = priv->pIO->Tell( priv->pIO );
		    priv->data_size = skip;
		    xmm_logging( 3, "ASF! data_offset = %li ( %lx )\n", priv->data_offset, priv->data_offset );

		    if( priv->seekable == 0 )
		    {
			loop = 0;
			skip = 0;
		    }
		    break;

	    /* ASF index chunk */
	    case ASF_CHUNK_INDEX:
		    break;

	    /* Skip unknown chunks */
	    default:
		    break;

	}

	/* Skip remaining ( unused ) data */
	priv->pIO->Seek( priv->pIO, skip, XMM_SEEK_CUR );
  }

  if( priv->data_offset == 0 )
  {
	priv->pIO->Close( priv->pIO );
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(ASF) No data found" );
  }

  /* Allocate packet buffer */
  priv->packet = malloc( priv->afh.maxPacketSize );
  if( priv->packet == NULL )
	return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(ASF) packet buffer ( packet size = %i )", priv->afh.maxPacketSize );

  /**/
  priv->astreams = priv->fi.astreams;
  priv->vstreams = priv->fi.vstreams;

  /* Seek to the beginning of data */
  if( priv->seekable )
  {
	/* Rebuild index */
	if(( idx1_found == 0 ) && idx1_build )
	{
	    /* Seek to data start */
	    priv->pIO->Seek( priv->pIO, priv->data_offset, XMM_SEEK_SET );

	    /* create data */
	    if( create_asf_seek_data( priv ) == XMM_RET_OK )
		idx1_found = 1;
	}

	/* Use idx1 chunk for reading */
	if( idx1_found == 1 )
	{
	    xmm_logging( 2, "ASF! INFO: Using IDX1 read mode\n" );
//	    input->VideoRead = asf_VideoRead_idx;
	    input->VideoSeek = asf_VideoSeek_idx;
//	    input->AudioRead = asf_AudioRead_idx;
	    input->AudioSeek = asf_AudioSeek_idx;

	    /* Audio */
	    for( i = 0; i < priv->astreams; i++ )
	    {
		j = priv->asIDX[i];			

		if( j >= priv->nStreams )
		{
		    xmm_logging( 1, "ASF! Wrong stream number: %i, Internal error (?).", j );
		    continue;
		}

		priv->fi.ai[i].tSamples = 0;
		priv->fi.ai[i].tSize = priv->aStream[j].ssi.tssize;
	    }

	    /* Video */
	    for( i = 0; i < priv->vstreams; i++ )
	    {
		j = priv->vsIDX[i];			

		if( j >= priv->nStreams )
		{
		    xmm_logging( 1, "ASF! Wrong stream number: %i, Internal error (?).", j );
		    continue;
		}
		ssi = &priv->aStream[j].ssi;

		priv->fi.vi[i].tFrames = ssi->entries;
		priv->fi.vi[i].tSize = ssi->tssize;

		/* Framerate */
		if( priv->aStream[j].ssi.entries > 1 )
		{
		    xmm_logging( 1, "%i, %i\n", 
				ssi->se[ssi->entries - 1].pts - ssi->se[0].pts );

		    priv->fi.vi[i].fmt.framerate = (double)1000 * ssi->entries
				    / ((double)ssi->se[ssi->entries - 1].pts
						- (double)ssi->se[0].pts );
		}
	    }
	}

	/* Seek to data start */
	priv->pIO->Seek( priv->pIO, priv->data_offset, XMM_SEEK_SET );
  }

#ifdef ASF_USE_MUTEX_IO
  /* Create Mutex */
  priv->mutex_io = xmmMutex_Create();
#endif

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* Clip info */
  priv->ci.size = priv->pIO->Size( priv->pIO );
  priv->ci.playtime = (((double)priv->afh.playTime - priv->afh.startTime * 10000 ) / 10000000 );

  if( priv->ci.content[0] == '\0' )
  {
	strncpy( priv->ci.content, priv->filename, XMM_CIL_CONTENT - 1 );
	priv->ci.content[XMM_CIL_CONTENT - 1] = '\0';
  }

  return XMM_RET_OK;
}

/*
 * Close
 */
static int asf_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;
  int		i;

  /* Close file and free resources */
  if( priv->filename )	free( priv->filename );
  if( priv->pIO )	priv->pIO->Close( priv->pIO );
  if( priv->packet )	free( priv->packet );

  for( i = 0; i < priv->nStreams; i++ )
  {
	if( priv->aStream[i].af )	free( priv->aStream[i].af );
	if( priv->aStream[i].vf )	free( priv->aStream[i].vf );
	if( priv->aStream[i].rbuffer )	free( priv->aStream[i].rbuffer );
	if( priv->aStream[i].wf )	free( priv->aStream[i].wf );
	if( priv->aStream[i].ssi.se )	free( priv->aStream[i].ssi.se );

#ifdef ASF_USE_MUTEX_STREAM
	/* Destroy mutex */
	if( priv->aStream[i].mutex )	xmmMutex_Destroy( priv->aStream[i].mutex );
#endif
  }

#ifdef ASF_USE_MUTEX_IO
  /* Destroy mutex */
  if( priv->mutex_io )	xmmMutex_Destroy( priv->mutex_io );
#endif

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int asf_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_AudioFormat	*format;
  struct stream_t	*ps;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:			/* Query graph format */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_AFORMAT:			/* Query sound format */
		format = (XMM_AudioFormat *)data;
		if(( format->format == priv->fi.ai[param].fmt.format ) &&
		    ( format->samprate == priv->fi.ai[param].fmt.samprate ) &&
		    ( format->channels == priv->fi.ai[param].fmt.channels ))
			return XMM_CTLRET_TRUE;

		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:			/* Query flip state */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_GFORMAT:			/* Get graph format */
		ps = asf_lookup_vstream( input, param );
		memcpy( data, ps->vf, sizeof( XMM_VideoFormat ));
		((XMM_VideoFormat *)data)->extraSize = 0;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_GFORMAT_PTR:			/* Get graph format */
		ps = asf_lookup_vstream( input, param );
		*((XMM_VideoFormat **) data) = ps->vf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT:			/* Get sound format */
		ps = asf_lookup_astream( input, param );
		memcpy( data, ps->af, sizeof( XMM_AudioFormat ));
		((XMM_AudioFormat *)data)->extraSize = 0;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		ps = asf_lookup_astream( input, param );
		*((XMM_AudioFormat **) data) = ps->af;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_CAPS:				/* Get capabilities */
		*((uint32_t *) data ) =	XMM_INPUT_CF_MODE_READ |
					XMM_INPUT_CF_AUDIO |
					XMM_INPUT_CF_VIDEO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:			/* Set graph format */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(ASF) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(ASF) cmd ( 0x%x )" );
}

/*
 * Seek to position
 */
static int asf_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;
  uint32_t		pts, idxl, idxr, idx;

  pts = (uint32_t)( seek * priv->ci.playtime * 1000 );

  if( priv->fi.vstreams )
  {
	ps = asf_lookup_vstream( input, vstream );

	idxl = 0;
	idxr = ps->ssi.entries - 1;

	while( idxl < ( idxr - 1 ))
	{
	    idx = ( idxl + idxr ) >> 1;

	    if( pts < ps->ssi.se[idx].pts )	idxr = idx;	/* Skip right part */
	    else	idxl = idx;				/* Skip left part */
	}

	/* Find next keyframe */
	while((( ps->ssi.se[idxl].pid & ASF_SSI_PIDMASK_KEY ) == 0 ) &&
		( idxl < ps->ssi.entries ))	idxl++;

	input->VideoSeek( input, vstream, idxl );
	input->VideoPTS( input, vstream, &pts );
  }

  if( priv->fi.astreams )
	input->AudioSeek( input, astream, pts );

  return XMM_RET_OK;
}

/*
 * Get Information
 */
static int asf_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;
  double		time;

  if( ci )	memcpy( ci, &priv->ci, sizeof( XMM_ClipInfo ));  

  if( priv->vstreams > 0 )	ps = asf_lookup_vstream( input, 0 );
  else	if( priv->astreams > 0 )	ps = asf_lookup_astream( input, 0 );
  else	ps = NULL;

  if( ps )	time = (double) ps->pts - priv->afh.startTime;
  else		time = (double) 0.0;

  if( seekval )		*seekval = time / ( priv->ci.playtime * 1000 );

  return XMM_RET_OK;
}

/*
 * Get audio stream number
 */
static int asf_AudioStreams( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  return priv->fi.astreams;
}

/*
 * Get audio stream information
 */
static int asf_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;

  ps = asf_lookup_astream( input, stream );

  if( ai )		memcpy( ai, &priv->fi.ai[stream], sizeof( XMM_AudioInfo ));

  return XMM_RET_OK;
}

/*
 * Read audio data ( idx1 )
 */
static int asf_AudioRead_idx( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(ASF) Internal error. Index mode not supported." );
}

/*
 * Read audio data
 */
static int asf_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;
  uint32_t		read;
  int			ret;

  ps = asf_lookup_astream( input, stream );

  /* Read raw data */
  while( xmmBQ_Size( &ps->bq ) < size )
  {
	/* Read object */
	ret = read_asf_data_raw( priv, ps );
	if( ret != XMM_RET_OK )	break;
  }

  if(( xmmBQ_Size( &ps->bq ) == 0 ) && ( size > 0 ))	return XMM_RET_EOS;

#ifdef ASF_USE_MUTEX_STREAM
  /* Lock bq */
  xmmMutex_Lock( ps->mutex );
#endif

  /* get data/PTS */
  read = xmmBQ_Size( &ps->bq );
  if( read > size )	read = size;

  xmmBQ_ReadSeq( &ps->bq, buffer, read, &ps->pts );

#ifdef ASF_USE_MUTEX_STREAM
  /* Unlock bq */
  xmmMutex_Unlock( ps->mutex );
#endif

  return read;
}

/*
 * Audio timestamp.
 */
static int asf_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;

  if( pts == NULL )
	return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(ASF) pts = NULL" );

  ps = asf_lookup_astream( input, stream );

  *pts = ps->pts - priv->afh.startTime;

  return XMM_RET_OK;
}

/*
 * Seek to position in audio stream
 */
static int asf_AudioSeek_idx( XMM_PluginInput *input, int stream, uint32_t pts )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;
  uint32_t		idxl, idxr, idx;

  ps = asf_lookup_astream( input, stream );

  idxl = 0;
  idxr = ps->ssi.entries - 1;

  while( idxl < ( idxr - 1 ))
  {
	idx = ( idxl + idxr ) >> 1;

	if( pts < ps->ssi.se[idx].pts )	idxr = idx;	/* Skip right part */
	else	idxl = idx;				/* Skip left part */
  }

  ps->pts = ps->ssi.se[idxl].pts + priv->afh.startTime;	/* Update PTS */
  ps->ssi.rbuffer_fragmentOffset = 0;

  /* Free buffered data */
  xmmBQ_Free( &ps->bq );

  /* Seek to packet */
  priv->pIO->Seek(	priv->pIO,
			priv->data_offset +
			( ps->ssi.se[idxl].pid & ASF_SSI_PIDMASK_PID ) *
			priv->afh.maxPacketSize, XMM_SEEK_SET );

  return XMM_RET_OK;
}

/*
 * Seek audio ( dummy function: no seeking supported )
 */
static int asf_AudioSeek( XMM_PluginInput *input, int stream, uint32_t position )
{
  return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(ASF) RAW mode. No seeking supported." );
}

/*
 * Get video stream number
 */
static int asf_VideoStreams( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  return priv->fi.vstreams;
}

/*
 * Get video stream information
 */
static int asf_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;

  ps = asf_lookup_vstream( input, stream );

  /* */
  if( vi )	memcpy( vi, &priv->fi.vi[stream], sizeof( XMM_VideoInfo ));
  if( cFrame )	*cFrame = ps->position;

  return XMM_RET_OK;
}

/*
 * Read video data ( idx1 )
 */
static int asf_VideoRead_idx( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(ASF) Internal error. Index mode not supported." );
}

/*
 * Read video data
 */
static int asf_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;
  int			ret, read;

  ps = asf_lookup_vstream( input, stream );

  /* Object buffered ? */
  if( xmmBQ_Size( &ps->bq ) == 0 )
  {
	/* Read object */
	ret = read_asf_data_raw( priv, ps );
	if( ret != XMM_RET_OK )	return ret;
  }

#ifdef ASF_USE_MUTEX_STREAM
  /* Lock bq */
  xmmMutex_Lock( ps->mutex );
#endif

  /* get data/PTS */
  read = xmmBQ_HeadReadSeq( &ps->bq, buffer[0], 0x7FFFFFFF, &ps->pts );

#ifdef ASF_USE_MUTEX_STREAM
  /* Unlock bq */
  xmmMutex_Unlock( ps->mutex );
#endif

  /* Update frame number */
  ps->position++;

  return read;
}

/*
 * Video PTS
 */
static int asf_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;

  if( videoPTS == NULL )
	return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(ASF) videoPTS = NULL" );

  ps = asf_lookup_vstream( input, stream );

  *videoPTS = ps->pts - priv->afh.startTime;

  return XMM_RET_OK;
}

/*
 * Seek to position in video stream
 */
static int asf_VideoSeek_idx( XMM_PluginInput *input, int stream, uint32_t frame )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;

  ps = asf_lookup_vstream( input, stream );

  /* Check for EOS */
  if( frame >= ps->ssi.entries )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AVI) Invalid frame number: %i, Stream contains only %i frames", frame, ps->ssi.entries );
	return XMM_RET_INVALID_ARG;
  }

  ps->position = frame;
  ps->pts = ps->ssi.se[frame].pts + priv->afh.startTime;	/* Update PTS */
  ps->ssi.rbuffer_fragmentOffset = 0;

  /* Free buffered data */
  xmmBQ_Free( &ps->bq );

  /* Seek to packet */
  priv->pIO->Seek(	priv->pIO,
			priv->data_offset +
			( ps->ssi.se[frame].pid & ASF_SSI_PIDMASK_PID ) *
			priv->afh.maxPacketSize, XMM_SEEK_SET );

  return XMM_RET_OK;
}

/*
 * Seek video ( dummy function: no seeking supported )
 */
static int asf_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(ASF) RAW mode. No seeking supported.\n" );
}

/*
 * Check if ASF file
 */
static int asf_CheckFile( void *xmm, char *filename )
{
  XMM_PluginIO		*pIO;
  asf_chunk_t		ach;

  /* Open file and read ( what maybe ) a ASF chunk header */
  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  pIO->Read( pIO, &ach, sizeof( asf_chunk_t ), 1 );

  pIO->Close( pIO );

  /* Check if ASF header chunk */
  if( !memcmp( &ach.guid, &asf_guids[ASF_CHUNK_HEADER].guid, sizeof( GUID_t )))
	return 1;

  return 0;
}

/*
 * Get file info
 */
static int asf_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"ASF",
				"Input: ASF",
				"Copyright (c) 2002 Arthur Kleer",
				NULL, NULL },
				asf_Init, asf_Open, asf_Close,
				asf_Control, asf_Seek, asf_Info, 
				asf_AudioStreams, asf_AudioInfo,
				asf_AudioRead, asf_AudioPTS, asf_AudioSeek,
				asf_VideoStreams, asf_VideoInfo,
				asf_VideoRead, asf_VideoPTS, asf_VideoSeek,
				asf_CheckFile, asf_FileInfo };

/*
 * Lookup audio stream
 */
static struct stream_t *asf_lookup_astream( XMM_PluginInput *input, int stream )
{
  int			idx;
  struct priv_t		*priv = input->sys.priv;

  /* Lookup stream */
  if( stream >= priv->astreams )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(ASF) Wrong stream number: %i, File has only %i audio streams.", stream, priv->astreams );
	return NULL;
  }
  idx = priv->asIDX[stream];			/* Audio */

  if( idx >= priv->nStreams )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(ASF) Wrong stream number: %i, Internal error (?).", idx );
	return NULL;
  }

  return &priv->aStream[idx];
}

/*
 * Lookup video stream
 */
static struct stream_t *asf_lookup_vstream( XMM_PluginInput *input, int stream )
{
  int			idx;
  struct priv_t		*priv = input->sys.priv;

  /* Lookup stream */
  if( stream >= priv->vstreams )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(ASF) Wrong stream number: %i, File has only %i video streams.", stream, priv->vstreams );
	return NULL;
  }
  /* TODO: ( - stream ) inverts the streams, the last contains the (best?) data */
  idx = priv->vsIDX[priv->vstreams - stream - 1];		/* Video */

  if( idx >= priv->nStreams )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(ASF) Wrong stream number: %i, Internal error (?).", idx );
	return NULL;
  }

  return &priv->aStream[idx];
}

/*
 * Convert M$ wFormatID to XMMP audio codec
 */
static uint32_t codec_ms2xmm( uint16_t ms_codec )
{
  int		xmm_fmt_idx;
  struct format_conv
  {
	int	xmm;
	int	ms;
  } format_table[] =
    {
	{ XMM_AUDIO_CODEC_PCM, WAVE_FORMAT_PCM },
	{ XMM_AUDIO_CODEC_IEEE, WAVE_FORMAT_IEEE_FLOAT },
	{ XMM_AUDIO_CODEC_ULAW, WAVE_FORMAT_MULAW },
	{ XMM_AUDIO_CODEC_ALAW, WAVE_FORMAT_ALAW },
	{ XMM_AUDIO_CODEC_MPEG, WAVE_FORMAT_MPEGLAYER3 },
	{ XMM_AUDIO_CODEC_AC3, WAVE_FORMAT_AC3 },
	{ XMM_AUDIO_CODEC_GSM610MS, WAVE_FORMAT_GSM610 },
	{ XMM_AUDIO_CODEC_G721, WAVE_FORMAT_G721_ADPCM },
	{ XMM_AUDIO_CODEC_G723, WAVE_FORMAT_G723_ADPCM },
	{ XMM_AUDIO_CODEC_G726, WAVE_FORMAT_G726_ADPCM },
	{ XMM_AUDIO_CODEC_WMA1, WAVE_FORMAT_WMA1 },
	{ XMM_AUDIO_CODEC_WMA2, WAVE_FORMAT_WMA2 },
	{ XMM_AUDIO_CODEC_ADPCM_MS, WAVE_FORMAT_ADPCM },
	{ XMM_AUDIO_CODEC_ADPCM_IMA, WAVE_FORMAT_DVI_ADPCM },
	{ XMM_AUDIO_CODEC_VOX_META, 0x75 },
	{ 0, 0 }
    };

  /* Check format */
  for( xmm_fmt_idx = 0; format_table[xmm_fmt_idx].xmm; xmm_fmt_idx++ )
	if( ms_codec == format_table[xmm_fmt_idx].ms )	break;

  if( xmm_fmt_idx == 10 )	return 0;

  return format_table[xmm_fmt_idx].xmm;
}

/*
 * Read one object
 */
static int read_asf_data_raw( struct priv_t *priv, struct stream_t *ps )
{
  struct stream_t		*psb;
  int				i, st, used = 0, todo, loop = 1;
  asf_data_erco_t		ade;
  asf_data_parse_t		adp;
  asf_segment_t			ahseg;

  /* Get data object */
  while( loop )
  {
#ifdef ASF_USE_MUTEX_IO
	/* Lock IO */
	xmmMutex_Lock( priv->mutex_io );
#endif

	/* Read packet */
	priv->pIO->Read( priv->pIO, priv->packet, priv->afh.maxPacketSize, 1 );
	if( priv->pIO->Eof( priv->pIO ))
	{
	    xmm_logging( 1, "ASF! End of File. Corrupted file ?\n" );
	    return XMM_RET_EOS;
	}
	used = 0;

#ifdef ASF_USE_MUTEX_IO
	/* Unlock IO */
	xmmMutex_Unlock( priv->mutex_io );
#endif

	if( priv->packet[0] & ASF_DFLAG_ERCO )
	{
	    /* Decode error correction */
	    used += decode_asf_packet_erco( priv->packet + used, &ade, priv->afh.maxPacketSize - used );
	    if( used >= priv->afh.maxPacketSize )	continue;
	}
	
	/* Decode v82 packet header */
	used += decode_asf_packet_parse( priv->packet + used, &adp, priv->afh.maxPacketSize - used );
#ifdef VERBOSE
	xmm_logging( 2, "ASF! %i bytes data in packet.\n", priv->afh.maxPacketSize - used - adp.paddingSize );
#endif

	/* Decode data */
	for( i = 0; i < ( adp.segments & ASF_DSNFLAG_NUMBER ); i++ )
	{
	    used += decode_asf_segment( priv->packet + used, &adp, &ahseg, priv->afh.maxPacketSize - used );

	    /* Lookup stream */
	    st = ( ahseg.stream & ASF_SSFLAG_STREAM ) - 1;
	    if( st >= priv->nStreams )	return	XMM_RET_INVALID_ARG;
	    psb = &priv->aStream[st];

#ifdef VERBOSE
	    xmm_logging( 2, "ASF! stream %i (internal) found.\n", st );
	    if( ahseg.offset && ( ahseg.replicated != ASF_SREP_GROUPING ))
		    xmm_logging( 1, "ASF! fragmentOffset = %i\n", ahseg.offset );
#endif

	    /* Read data */
	    if( ahseg.replicated != ASF_SREP_GROUPING )
	    {
		/* Did we miss a fragment ? */
		if( psb->rbuffer_fragmentOffset != ahseg.offset )
		{
		    /* TODO: This should result in an read error, as decoding will not be possible */
		    xmm_logging( 2, "ASF! Fragment error: Offset = %i, should be %i\n", ahseg.offset, psb->rbuffer_fragmentOffset );
		}

		/* Update fragments offset, object complete if all fragments read */
		psb->rbuffer_fragmentOffset += ahseg.dataSize;
		if( psb->rbuffer_fragmentOffset >= ahseg.objectSize )
			psb->rbuffer_fragmentOffset = 0;

		/* Copy data into object buffer */
		memcpy( psb->rbuffer + ahseg.offset, priv->packet + used, ahseg.dataSize );

		/* Buffer object */
		if( psb->rbuffer_fragmentOffset == 0 )
		{
#ifdef ASF_USE_MUTEX_STREAM
		    /* Lock bq */
		    xmmMutex_Lock( psb->mutex );
#endif
		    xmmBQ_AddSeq( &psb->bq, psb->rbuffer, ahseg.objectSize, ahseg.objectTime );

#ifdef ASF_USE_MUTEX_STREAM
		    /* Unlock bq */
		    xmmMutex_Unlock( psb->mutex );
#endif

		    /* Desired stream ? */
		    if( psb == ps )	loop = 0;
		}

		/* We used some data */
		used += ahseg.dataSize;
	    }
	    else
	    {
		/* loop until we run out of data */
		for( todo = ahseg.dataSize; todo > 0; )
		{
		    /* Get fragment size */
		    uint32_t len;

		    /* Get object size */
		    len = (uint32_t) *(priv->packet + used);
		    used++;

#ifdef ASF_USE_MUTEX_STREAM
		    /* Lock bq */
		    xmmMutex_Lock( psb->mutex );
#endif
		    xmmBQ_AddSeq( &psb->bq, priv->packet + used, len, ahseg.objectTime );

#ifdef ASF_USE_MUTEX_STREAM
		    /* Unlock bq */
		    xmmMutex_Unlock( psb->mutex );
#endif

		    /* Desired stream ? */
		    if( psb == ps )	loop = 0;

		    /* */
		    used += len;
		    todo -= ( len + 1 );
		}
	    }
	}
  }

  return XMM_RET_OK;
}

/*
 * Create seek data
 */

static int ssi_realloc( seek_stream_info_t *ssi );

static int create_asf_seek_data( struct priv_t *priv )
{
  int			i, st, used = 0, todo;
  asf_data_erco_t	ade;
  asf_data_parse_t	adp;
  asf_segment_t		ahseg;
  uint8_t		*packet;
  uint32_t		pid = 0;
  seek_stream_info_t	*pssi;
#if ASF_DEBUG_SEEK_DATA == 2
  int			j;
#endif

  /* Initialize */
  for( i = 0; i < priv->nStreams; i++ )
  {
	priv->aStream[i].ssi.entries = 0;
	priv->aStream[i].ssi.rbuffer_fragmentOffset = 0;
	priv->aStream[i].ssi.tssize = 0;

	priv->aStream[i].ssi.size = ASF_SEEK_BLOCK;
	priv->aStream[i].ssi.se = malloc( ASF_SEEK_BLOCK * sizeof( seek_entry_t ));
	if( priv->aStream[i].ssi.se == NULL )	return XMM_RET_ALLOC;
  }

  /* Allocate packet buffer */
  packet = malloc( priv->afh.maxPacketSize );
  if( packet == NULL )	return XMM_RET_ALLOC;

  xmm_logging( 2, "ASF! Creating seek data " );

  /* Get data object */
  while( pid < priv->afh.nPackets )
  {
	/* Read packet */
	priv->pIO->Read( priv->pIO, packet, priv->afh.maxPacketSize, 1 );
	if( priv->pIO->Eof( priv->pIO ))	break;

	used = 0;

	if( packet[0] & ASF_DFLAG_ERCO )
	{
	    /* Decode error correction */
	    used += decode_asf_packet_erco( packet + used, &ade, priv->afh.maxPacketSize - used );
	    if( used >= priv->afh.maxPacketSize )	continue;
	}
	
	/* Decode v82 packet header */
	used += decode_asf_packet_parse( packet + used, &adp, priv->afh.maxPacketSize - used );

	/* Decode data */
	for( i = 0; i < ( adp.segments & ASF_DSNFLAG_NUMBER ); i++ )
	{
	    used += decode_asf_segment( packet + used, &adp, &ahseg, priv->afh.maxPacketSize - used );

	    /* Lookup stream */
	    st = ( ahseg.stream & ASF_SSFLAG_STREAM ) - 1;
	    if( st >= priv->nStreams )	return	XMM_RET_INVALID_ARG;
	    pssi = &priv->aStream[st].ssi;

	    /* Read data */
	    if( ahseg.replicated != ASF_SREP_GROUPING )
	    {
		/* Did we miss a fragment ? */
		if( pssi->rbuffer_fragmentOffset != ahseg.offset )
		{
		    /* TODO: This should result in an read error, as decoding will not be possible */
		    xmm_logging( 2, "ASF! Fragment error: Offset = %i, should be %i\n", ahseg.offset, pssi->rbuffer_fragmentOffset );
		}

		/* Update fragments offset, object complete if all fragments read */
		pssi->rbuffer_fragmentOffset += ahseg.dataSize;
		if( pssi->rbuffer_fragmentOffset >= ahseg.objectSize )
			pssi->rbuffer_fragmentOffset = 0;

		/* Buffer object */
		if( pssi->rbuffer_fragmentOffset == 0 )
		{
		    if( pssi->entries >= pssi->size )	ssi_realloc( pssi );

		    pssi->se[pssi->entries].pts = ahseg.objectTime - priv->afh.startTime;
		    pssi->se[pssi->entries].pid = pid;
		    if( ahseg.stream & ASF_SSFLAG_KEYFRAME )
			pssi->se[pssi->entries].pid |= ASF_SSI_PIDMASK_KEY;
		    pssi->entries++;
		    pssi->tssize += ahseg.objectSize;

		}

		/* We used some data */
		used += ahseg.dataSize;
	    }
	    else
	    {
		/* loop until we run out of data */
		for( todo = ahseg.dataSize; todo > 0; )
		{
		    uint32_t len;

		    len = (uint32_t) *(packet + used);
		    used += ( len + 1 );
		    todo -= ( len + 1 );

		    if( pssi->entries >= pssi->size )	ssi_realloc( pssi );

		    pssi->se[pssi->entries].pts = ahseg.objectTime - priv->afh.startTime;
		    pssi->se[pssi->entries].pid = pid;
		    if( ahseg.stream & ASF_SSFLAG_KEYFRAME )
			pssi->se[pssi->entries].pid |= ASF_SSI_PIDMASK_KEY;
		    pssi->entries++;
		    pssi->tssize += len;

		}
	    }
	}
	pid++;

	if(( pid % ( DivideUP( priv->afh.nPackets, 50 ))) == 0 )
		xmm_logging( 1, "." );
  }

  xmm_logging( 1, "\n" );

#ifdef ASF_DEBUG_SEEK_DATA
  for( i = 0; i < priv->nStreams; i++ )
  {
	xmm_logging( 1, "ASF! Stream %i: %i entries\n", i, priv->aStream[i].ssi.entries );
#if ASF_DEBUG_SEEK_DATA == 2
	for( j = 0; j < priv->aStream[i].ssi.entries; j++ )
	{
	    xmm_logging( 1, "ASF! Entry %i: PTS = %i (PID = %i) Key = %s\n", j, priv->aStream[i].ssi.se[j].pts, priv->aStream[i].ssi.se[j].pid & ASF_SSI_PIDMASK_PID, ( priv->aStream[i].ssi.se[j].pid & ASF_SSI_PIDMASK_KEY ) ? "yes" : "no" );
	}
#endif
  }
#endif

  free( packet );
  return XMM_RET_OK;
}

/*
 * Misc
 */
static int ssi_realloc( seek_stream_info_t *ssi )
{
  seek_entry_t *temp;

  ssi->size += ASF_SEEK_BLOCK;
  temp = realloc( ssi->se, ssi->size * sizeof( seek_entry_t ));
  if( temp == NULL )	return XMM_RET_ALLOC;
  ssi->se = temp;

  return XMM_RET_OK;
}
