/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_asffmt_h
#define INC_asffmt_h

#include <inttypes.h>

/*
 * Types
 */

/*
 * GUID
 */
typedef struct __attribute__((packed)) GUID_s
{
    uint32_t		v1;
    uint16_t		v2;
    uint16_t		v3;
    uint8_t		v4[8];
} GUID_t;

/*
 * ASF chunk header
 */
typedef struct __attribute__((packed)) asf_chunk_s
{
    GUID_t		guid;
    uint64_t		size;
} asf_chunk_t;

/*
 * ASF header: first chunk in file
 */
typedef struct __attribute__((packed)) asf_header_s
{
    uint32_t		subchunks;		/* number of header chunks */
    uint16_t		unknown;
} asf_header_t;

/*
 * ASF file header
 */
typedef struct __attribute__((packed)) asf_file_s
{
    GUID_t		fileID;			/* File GUID */
    uint64_t		fileSize;		/* File size */
    uint64_t		creationTime;		/* File creation time */
    uint64_t		nPackets;		/* Number of packets  */
    uint64_t		playTime;		/* play time (100ns) */
    uint64_t		sendTime;		/* send time (100ns) */
    uint64_t		startTime;		/* start time (ms) */

#define	ASF_FFLAG_BROADCAST	0x01
#define	ASF_FFLAG_SEEKABLE	0x02
    uint32_t		flags;			/* flags */

    uint32_t		minPacketSize;		/* Packet size ( minimum ) */
    uint32_t		maxPacketSize;		/* Packet size ( maximum ) */
    uint32_t		maxBitrate;		/* Bitrate ( maximum ) */
} asf_file_t;

/*
 * ASF stream header
 */
typedef struct __attribute__((packed)) asf_stream_s
{
    GUID_t		type;			/* Stream type */
    GUID_t		erco;			/* Audio error correction type */
    uint64_t		timeOffset;		/* stream time offset (100ns) */
    uint32_t		typeSize;		/* Size: type specific data */
    uint32_t		ercoSize;		/* Size: error correction data */

#define	ASF_SFLAG_MASK_STREAM	0x007F
#define	ASF_SFLAG_ENCRYPTED	0x8000
    uint16_t		stream;			/* Stream number, encryption */

    uint32_t		reserved;
} asf_stream_t;


/*
 * Audio
 */

typedef struct __attribute__((packed)) _WAVEFORMATEX
{
#define WAVE_FORMAT_PCM			0x0001
#define WAVE_FORMAT_ADPCM		0x0002
#define	WAVE_FORMAT_IEEE_FLOAT		0x0003
#define	WAVE_FORMAT_ALAW		0x0006
#define	WAVE_FORMAT_MULAW		0x0007
#define	WAVE_FORMAT_DVI_ADPCM		0x0011
#define	WAVE_FORMAT_G723_ADPCM		0x0014
#define	WAVE_FORMAT_GSM610		0x0031
#define	WAVE_FORMAT_G721_ADPCM		0x0040
#define	WAVE_FORMAT_MPEG		0x0050
#define	WAVE_FORMAT_MPEGLAYER3		0x0055
#define	WAVE_FORMAT_G726_ADPCM		0x0064
#define	WAVE_FORMAT_WMA1		0x0160
#define	WAVE_FORMAT_WMA2		0x0161
#define	WAVE_FORMAT_AC3			0x2000
    uint16_t		wFormatTag;		/* Format */
    uint16_t		nChannels;		/* Channels */
    uint32_t		nSamplesPerSec;		/* Rate */
    uint32_t		nAvgBytesPerSec;	/* Average bytes / sec */
    uint16_t		nBlockAlign;		/* block size of data */
    uint16_t		wBitsPerSample;		/* Number of bits per sample */
    uint16_t		cbSize;			/* Size of extra data */
} WAVEFORMATEX;

/*
 * ASF audio spread header
 */
typedef struct __attribute__((packed)) asf_audio_spread_s
{
    uint8_t		span;			/* audio blocks / scramble group */
    uint16_t		packetSize;		/* size of scrambling chunk */
    uint16_t		chunkSize;		/* usually nBlockAlign */
    uint16_t		dataSize;		/* usually nBlockAlign */
    uint8_t		silence;
} asf_audio_spread_t;

/*
 * Video
 */

typedef struct __attribute__((packed)) _BITMAPINFOHEADER
{
    uint32_t 		biSize;			/* long */
    uint32_t  		biWidth;		/* long */
    uint32_t  		biHeight;		/* long */
    uint16_t		biPlanes;		/* short */
    uint16_t		biBitCount;		/* short */
    uint32_t		biCompression;		/* long */
    uint32_t		biSizeImage;		/* long */
    uint32_t		biXPelsPerMeter;	/* long */
    uint32_t		biYPelsPerMeter;	/* long */
    uint32_t		biClrUsed;		/* long */
    uint32_t		biClrImportant;		/* long */
} BITMAPINFOHEADER;

/*
 * ASF video stream header
 */
typedef struct __attribute__((packed)) asf_video_type_s
{
    uint32_t		width;			/* Width */
    uint32_t		height;			/* Height */
    uint8_t		flags;
    uint16_t		biSize;			/* BITMAPINFOHEADER size */
    BITMAPINFOHEADER	bih;
} asf_video_type_t;

/*
 * ASF content header
 */
typedef struct __attribute__((packed)) asf_content_s
{
    struct __attribute__((packed))
    {
	uint16_t		len_title;
	uint16_t		len_author;
	uint16_t		len_copyright;
	uint16_t		len_desc;
	uint16_t		len_unknown;
    } d;

    /* Rendered data */
    char		*title;				/* Title ( free this ) */
    char		*author;			/* Author */
    char		*copyright;			/* Copyright */
    char		*desc;				/* Description */
    char		*unknown;			/* ??? */
} asf_content_t;

/*
 * ASF data header
 */
typedef struct __attribute__((packed)) asf_data_s
{
    GUID_t		unknown1;
    uint64_t		nPackets;
    uint16_t		unknown2;
} asf_data_t;


#define	ASF_DFLAG_ERCO			0x80

/*
 * Error correction
 */

typedef struct asf_data_erco_s
{
#define	ASF_DEFLAG_SIZE			0x0F
#define	ASF_DEFLAG_TYPE			0x60	/* = 0, else unknown erco size */

    uint8_t		flags;
} asf_data_erco_t;


/*
 * Data parsing
 */

typedef struct asf_data_parse_s
{

#define	ASF_DPSFLAG_SEGMENTS		0x01
#define	ASF_DPSFLAG_SIZE_SEQ( val )	(( val >> 1 ) & 0x03 )
#define	ASF_DPSFLAG_SIZE_PADD( val )	(( val >> 3 ) & 0x03 )
#define	ASF_DPSFLAG_SIZE_PACK( val )	(( val >> 5 ) & 0x03 )
    uint8_t		sizeFlags;		/* size flags */
#define	ASF_DPPFLAG_SIZE_REP( val )	(( val >> 0 ) & 0x03 )
#define	ASF_DPPFLAG_SIZE_OFF( val )	(( val >> 2 ) & 0x03 )
#define	ASF_DPPFLAG_SIZE_SEQ( val )	(( val >> 4 ) & 0x03 )
#define	ASF_DPPFLAG_SIZE_STR( val )	(( val >> 6 ) & 0x03 )
    uint8_t		propFlags;		/* property flags: 0x55, 0x59, 0x5D */
    uint32_t		sendTime;		/* Send time (ms) */
    uint16_t		duration;		/* Duration (ms) */

    /* Data, size depends on flags */
    uint32_t		packetSize;		/* Packet size */
    uint32_t		sequenceNum;		/* Sequence number */
    uint32_t		paddingSize;		/* Padding size */

#define	ASF_DSNFLAG_NUMBER		0x3F
#define	ASF_DSNFLAG_SIZE( val )		(( val >> 6 ) & 0x03 )
#define	ASF_DSNFLAG_SIZESET( size )	(( size << 6 ) & 0xC0 )
    uint8_t		segments;		/* number and properties */

} asf_data_parse_t;

/*
 * ASF segment header
 */

typedef struct asf_segment_s
{
#define	ASF_SSFLAG_STREAM		0x7F
#define	ASF_SSFLAG_KEYFRAME		0x80
    uint8_t		stream;			/* Stream ID */

    uint32_t		sequence;		/* Sequence number */
    uint32_t		offset;			/* Fragment offset */
#define	ASF_SREP_GROUPING		0x01
    uint32_t		replicated;		/* Replicated size */

    /* Rendered data */
    uint32_t		objectTime;		/* Object start time: (ms) */
    uint32_t		objectSize;		/* Object size, only for no grouping */

    uint16_t		dataSize;		/* Data size */

} asf_segment_t;

/*
 * Data: GUIDs
 */

#define	ASF_CHUNK_HEADER		0x00	/* header */
#define	ASF_CHUNK_DATA			0x01	/* data */
#define	ASF_CHUNK_INDEX			0x02	/* index */
#define	ASF_CHUNK_FILE			0x03	/* File header */
#define	ASF_CHUNK_STREAM		0x04	/* stream header */
#define	ASF_CHUNK_CONTENT		0x05	/* Media description */
#define	ASF_CHUNK_STREAM_AUDIO		0x06	/* audio stream header */
#define	ASF_CHUNK_STREAM_VIDEO		0x07	/* video stream header */
#define	ASF_CHUNK_AUDIO_CONCEAL_NONE	0x08	/* no audio error concealment */
#define	ASF_CHUNK_AUDIO_CONCEAL_ILEAVE	0x09	/* interleaved audio error concealment */
#define	ASF_CHUNK_VIDEO_CONCEAL_NONE	0x0A	/* no video error concealment */
#define	ASF_CHUNK_HEADER_ASF2		0x0B	/* ASF 2.0 */

#define	ASF_CHUNK_HEADER_UNKNOWN	0x80

static const struct
{
    const GUID_t	guid;
    const char		*name;
} asf_guids[] = {


 {{ 0x75B22630, 0x668E, 0x11CF, { 0xA6, 0xD9, 0x00, 0xAA, 0x00, 0x62, 0xCE, 0x6C }},
 "header" },
 {{ 0x75b22636, 0x668e, 0x11cf, { 0xa6, 0xd9, 0x00, 0xaa, 0x00, 0x62, 0xce, 0x6c }},
 "data" },
 {{ 0x33000890, 0xe5b1, 0x11cf, { 0x89, 0xf4, 0x00, 0xa0, 0xc9, 0x03, 0x49, 0xcb }},
 "index" },

 /* Header types */
 {{ 0x8CABDCA1, 0xA947, 0x11CF, { 0x8E, 0xE4, 0x00, 0xC0, 0x0C, 0x20, 0x53, 0x65 }},
 "File header" },
 {{ 0xB7DC0791, 0xA9B7, 0x11CF, { 0x8E, 0xE6, 0x00, 0xC0, 0x0C, 0x20, 0x53, 0x65 }},
 "stream header" },
 {{ 0x75b22633, 0x668e, 0x11cf, { 0xa6, 0xd9, 0x00, 0xaa, 0x00, 0x62, 0xce, 0x6c }},
 "content header" },

 /* Stream types */
 {{ 0xF8699E40, 0x5B4D, 0x11CF, { 0xA8, 0xFD, 0x00, 0x80, 0x5F, 0x5C, 0x44, 0x2B }},
 "audio stream" },
 {{ 0xBC19EFC0, 0x5B4D, 0x11CF, { 0xA8, 0xFD, 0x00, 0x80, 0x5F, 0x5C, 0x44, 0x2B }},
 "video stream" },

 /* Stream error correction types */
 {{ 0x49f1a440, 0x4ece, 0x11d0, { 0xa3, 0xac, 0x00, 0xa0, 0xc9, 0x03, 0x48, 0xf6 }},
 "no audio error concealment" },
 {{ 0xbfc3cd50, 0x618f, 0x11cf, { 0x8b, 0xb2, 0x00, 0xaa, 0x00, 0xb4, 0xe2, 0x20 }},
 "interleaved audio error concealment" },
 {{ 0x20fb5700, 0x5b55, 0x11cf, { 0xa8, 0xfd, 0x00, 0x80, 0x5f, 0x5c, 0x44, 0x2b }},
 "no video error concealment" },

 {{ 0xD6E229D1, 0x35da, 0x11d1, { 0x90, 0x34, 0x00, 0xa0, 0xc9, 0x03, 0x49, 0xbe }},
 "ASF 2.0" },

 /* Other not handled chunks/headers ( from avifile 0.6cvs */
 {{ 0x7bf875ce, 0x468d, 0x11d1, { 0x8d, 0x82, 0x00, 0x60, 0x97, 0xc9, 0xa2, 0xb2 }},
 "stream bitrate properties" },
 {{ 0x298ae614, 0x2622, 0x4c17, { 0xb9, 0x35, 0xda, 0xe0, 0x7e, 0xe9, 0x28, 0x9c }},
 "extended content encryption" },
 {{ 0x1efb1a30, 0x0b62, 0x11d0, { 0xa3, 0x9b, 0x00, 0xa0, 0xc9, 0x03, 0x48, 0xf6 }},
 "script command" },
 {{ 0xf487cd01, 0xa951, 0x11cf, { 0x8e, 0xe6, 0x00, 0xc0, 0x0c, 0x20, 0x53, 0x65 }},
 "marker" },
 {{ 0x5fbf03b5, 0xa92e, 0x11cf, { 0x8e, 0xe3, 0x00, 0xc0, 0x0c, 0x20, 0x53, 0x65 }},
 "header extension" },
 {{ 0xd6e229dc, 0x35da, 0x11d1, { 0x90, 0x34, 0x00, 0xa0, 0xc9, 0x03, 0x49, 0xbe }},
 "bitrate mutual exclusion" },
 {{ 0x86d15240, 0x311d, 0x11d0, { 0xa3, 0xa4, 0x00, 0xa0, 0xc9, 0x03, 0x48, 0xf6 }},
 "codec list" },
 {{ 0xd2d0a440, 0xe307, 0x11d2, { 0x97, 0xf0, 0x00, 0xa0, 0xc9, 0x5e, 0xa8, 0x50 }},
 "extended content description" },
 {{ 0x75b22635, 0x668e, 0x11cf, { 0xa6, 0xd9, 0x00, 0xaa, 0x00, 0x62, 0xce, 0x6c }},
 "error correction" },
 {{ 0x1806d474, 0xcadf, 0x4509, { 0xa4, 0xba, 0x9a, 0xab, 0xcb, 0x96, 0xaa, 0xe8 }},
 "padding" },
 {{ 0x59dacfc0, 0x59e6, 0x11d0, { 0xa3, 0xac, 0x00, 0xa0, 0xc9, 0x03, 0x48, 0xf6 }},
 "command media" },
 {{ 0xd6e22a01, 0x35da, 0x11d1, { 0x90, 0x34, 0x00, 0xa0, 0xc9, 0x03, 0x49, 0xbe }},
 "mutex bitrate" },
 {{ 0xd6e22a02, 0x35da, 0x11d1, { 0x90, 0x34, 0x00, 0xa0, 0xc9, 0x03, 0x49, 0xbe }},
 "mutex unknown" }, 
 {{ 0x4B1ACBE3, 0x100B, 0x11D0, { 0xA3, 0x9B, 0x00, 0xA0, 0xC9, 0x03, 0x48, 0xF6 }},
 "reserved script command" },
 {{ 0x4CFEDB20, 0x75F6, 0x11CF, { 0x9C, 0x0F, 0x00, 0xA0, 0xC9, 0x03, 0x49, 0xCB }},
 "reserved marker" },

  /* */ 
 {{ 0, 0, 0, { 0, 0, 0, 0, 0, 0, 0, 0 }}, "" },

};

#endif
