/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Read ASF headers
 */

#include <stdlib.h>

#include "asfhead.h"
#include <libxmm/util/utils.h>

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/*
 * Read ASF chunk header
 */
int read_asf_chunk( XMM_PluginIO *pIO, asf_chunk_t *ach, int size )
{
  int i;

  pIO->Read( pIO, ach, sizeof( asf_chunk_t ), 1 );
  if( pIO->Eof( pIO ))	return -1;

  for( i = 0; asf_guids[i].guid.v1; i++ )
  {
	if( !memcmp( &ach->guid, &asf_guids[i].guid, sizeof( GUID_t )))
	{
	    xmm_logging( 2, "ASF! GUID: name = %s size = %i\n", asf_guids[i].name, ach->size - 24 );
	    return sizeof( asf_chunk_t );
	}
  }

  xmm_logging( 2, "ASF! GUID: %x, %x, %x,", ach->guid.v1, ach->guid.v2, ach->guid.v3 );
  for( i = 0; i < 8; i++ )	xmm_logging( 2, " %x", ach->guid.v4[i] );
  xmm_logging( 2, ", size = %lli\n", ach->size - 24 );

  return sizeof( asf_chunk_t );
}

/*
 * Read ASF header
 */
int read_asf_header( XMM_PluginIO *pIO, asf_header_t *ah, int size )
{
  pIO->Read( pIO, ah, sizeof( asf_header_t ), 1 );

#ifdef VERBOSE
  xmm_logging( 2, "ASF! Header: subchunks %i\n", ah->subchunks );
#endif

  return sizeof( asf_header_t );
}

/*
 * Read ASF file header
 */
int read_asf_file( XMM_PluginIO *pIO, asf_file_t *afh, int size )
{
  pIO->Read( pIO, afh, sizeof( asf_file_t ), 1 );

#ifdef VERBOSE
  xmm_logging( 2, "ASF! File header: FileSize %lli\n", afh->fileSize );
  xmm_logging( 2, "ASF! File header: creationTime %lli\n", afh->creationTime );
  xmm_logging( 2, "ASF! File header: nPackets %lli\n", afh->nPackets );
  xmm_logging( 2, "ASF! File header: playTime %lli\n", afh->playTime );
  xmm_logging( 2, "ASF! File header: sendTime %lli\n", afh->sendTime );
  xmm_logging( 2, "ASF! File header: startTime %lli\n", afh->startTime );
  xmm_logging( 2, "ASF! File header: flags %x\n", afh->flags );
  xmm_logging( 2, "ASF! File header: minPacketSize %i\n", afh->minPacketSize );
  xmm_logging( 2, "ASF! File header: maxPacketSize %i\n", afh->maxPacketSize );
  xmm_logging( 2, "ASF! File header: maxBitrate %i\n", afh->maxBitrate );
#endif

  return sizeof( asf_file_t );
}

/*
 * Read stream header
 */
int read_asf_stream( XMM_PluginIO *pIO, asf_stream_t *ash, int size )
{
  pIO->Read( pIO, ash, sizeof( asf_stream_t ), 1 );

#ifdef VERBOSE
  print_guid( &ash->type );
  print_guid( &ash->erco );
  xmm_logging( 2, "ASF! Stream header: timeOffset %lli\n", ash->timeOffset );
  xmm_logging( 2, "ASF! Stream header: typeSize %li\n", ash->typeSize );
  xmm_logging( 2, "ASF! Stream header: ercoSize %li\n", ash->ercoSize );
  xmm_logging( 2, "ASF! Stream header: stream %i ( encrypted: %s )\n", ash->stream & ASF_SFLAG_MASK_STREAM, ash->stream & ASF_SFLAG_ENCRYPTED ? "yes" : "no" );
#endif

  return sizeof( asf_stream_t );
}

/*
 * Read video stream header ( type specific )
 */
int read_asf_video_type( XMM_PluginIO *pIO, asf_video_type_t *avsh, int size )
{
  pIO->Read( pIO, avsh, sizeof( asf_video_type_t ), 1 );

#ifdef VERBOSE
  xmm_logging( 2, "ASF! Video (type): width %i\n", avsh->width );
  xmm_logging( 2, "ASF! Video (type): height %i\n", avsh->height );
  xmm_logging( 2, "ASF! Video (type): biSize %i\n", avsh->biSize );

  xmm_logging( 2, "\tbi_head: Size = %li\n", avsh->bih.biSize );
  xmm_logging( 2, "\tbi_head: Width = %li\n", avsh->bih.biWidth );
  xmm_logging( 2, "\tbi_head: Height = %li\n", avsh->bih.biHeight );
  xmm_logging( 2, "\tbi_head: Planes = %i\n", avsh->bih.biPlanes );
  xmm_logging( 2, "\tbi_head: BitCount = %i\n", avsh->bih.biBitCount );
  xmm_logging( 2, "\tbi_head: Compression = '%s' ( %lx )\n", xmm_FOURCC_string( avsh->bih.biCompression ), avsh->bih.biCompression );
  xmm_logging( 2, "\tbi_head: SizeImage = %li\n", avsh->bih.biSizeImage );
  xmm_logging( 2, "\tbi_head: XPels/m = %li\n", avsh->bih.biXPelsPerMeter );
  xmm_logging( 2, "\tbi_head: YPels/m = %li\n", avsh->bih.biYPelsPerMeter );
  xmm_logging( 2, "\tbi_head: ClrUsed = %li\n", avsh->bih.biClrUsed );
  xmm_logging( 2, "\tbi_head: ClrImportant = %li\n", avsh->bih.biClrImportant );
#endif

  return sizeof( asf_video_type_t );
}

/*
 * Read spread audio (erco) header
 */
int read_asf_audio_spread( XMM_PluginIO *pIO, asf_audio_spread_t *aash, int size )
{
  pIO->Read( pIO, aash, sizeof( asf_audio_spread_t ), 1 );

#ifdef VERBOSE
  xmm_logging( 2, "ASF! Audio (ErCo: Spread): span %i\n", aash->span );
  xmm_logging( 2, "ASF! Audio (ErCo: Spread): packetSize %i\n", aash->packetSize );
  xmm_logging( 2, "ASF! Audio (ErCo: Spread): chunkSize %i\n", aash->chunkSize );
  xmm_logging( 2, "ASF! Audio (ErCo: Spread): dataSize %i\n", aash->dataSize );
  xmm_logging( 2, "ASF! Audio (ErCo: Spread): silence %i\n", aash->silence );
#endif

  return sizeof( asf_audio_spread_t );
}

/*
 * Read data header
 */
int read_asf_data( XMM_PluginIO *pIO, asf_data_t *adh, int size )
{
  pIO->Read( pIO, adh, sizeof( asf_data_t ), 1 );

#ifdef VERBOSE
  xmm_logging( 2, "ASF! Data header: nPackets %lli\n", adh->nPackets );
#endif

  return sizeof( asf_data_t );
}

/*
 * Read content header
 */
int read_asf_content( XMM_PluginIO *pIO, asf_content_t *ach, int size )
{
  uint16_t	*ws, wlen;
  int		i = 0, t = 0;

  pIO->Read( pIO, &ach->d, sizeof( ach->d ), 1 );

  /* allocate temporary buffer */
  wlen = ach->d.len_title + ach->d.len_author + ach->d.len_copyright + ach->d.len_desc + ach->d.len_unknown;
  if(( ws = malloc( wlen )) == NULL )	return sizeof( ach->d );

  pIO->Read( pIO, ws, wlen, 1 );	/* Read data */

  /* allocate string memory */
  if(( ach->title = malloc(( wlen >> 1 ) + 5 )) == NULL )
	return sizeof( ach->d ) + wlen;

  memset( ach->title, 0, 5 );

  if( ach->d.len_title )
  {
	for( ; ws[t]; i++, t++ )	ach->title[i] = ws[t] & 0x00FF;
	ach->title[i] = '\0';
	t++;
  }

  ach->author = &ach->title[++i];
  if( ach->d.len_author )
  {
	for( ; ws[t]; i++, t++ )	ach->title[i] = ws[t] & 0x00FF;
	ach->title[i] = '\0';
	t++;
  }

  ach->copyright = &ach->title[++i];
  if( ach->d.len_copyright )
  {
	for( ; ws[t]; i++, t++ )	ach->title[i] = ws[t] & 0x00FF;
	ach->title[i] = '\0';
	t++;
  }

  ach->desc = &ach->title[++i];
  if( ach->d.len_desc )
  {
	for( ; ws[t]; i++, t++ )	ach->title[i] = ws[t] & 0x00FF;
	ach->title[i] = '\0';
	t++;
  }

  ach->unknown = &ach->title[++i];
  if( ach->d.len_unknown )
  {
	for( ; ws[t]; i++, t++ )	ach->title[i] = ws[t] & 0x00FF;
	ach->title[i] = '\0';
	t++;
  }

#ifdef VERBOSE
  xmm_logging( 2, "ASF! Content header: len_title %i '%s'\n", ach->d.len_title, ach->title );
  xmm_logging( 2, "ASF! Content header: len_author %i '%s'\n", ach->d.len_author, ach->author );
  xmm_logging( 2, "ASF! Content header: len_copyright %i '%s'\n", ach->d.len_copyright, ach->copyright );
  xmm_logging( 2, "ASF! Content header: len_desc %i '%s'\n", ach->d.len_desc, ach->desc );
  xmm_logging( 2, "ASF! Content header: len_unknown %i '%s'\n", ach->d.len_unknown, ach->unknown );
#endif

  free( ws );
  return sizeof( ach->d ) + wlen;
}

/*
 * Decode error correction data
 */
int decode_asf_packet_erco( uint8_t *data, asf_data_erco_t *ade, int size )
{
  int	done = 0;

  memset( ade, 0, sizeof( asf_data_erco_t ));

  /* Get flags */
  ade->flags = ((uint8_t *) data )[0];
  data += sizeof( uint8_t );
  done += sizeof( uint8_t );

  if( ade->flags & ASF_DEFLAG_TYPE )
  {
	xmm_logging( 1, "ASF! Unknown erco size ( flags = 0x%x )\n", ade->flags );
	return size;
  }

#ifdef VERBOSE
  xmm_logging( 2, "ASF! ErCo data: flags %x ( size = %i, type = %x )\n", ade->flags, ade->flags & ASF_DEFLAG_SIZE, ade->flags & ASF_DEFLAG_TYPE );
#endif

  return done + ( ade->flags & ASF_DEFLAG_SIZE );
}

/*
 * Decode parsing data
 */
int decode_asf_packet_parse( uint8_t *data, asf_data_parse_t *adp, int size )
{
  int	done = 0;

  memset( adp, 0, sizeof( asf_data_parse_t ));

  /* Get size and property flags */
  adp->sizeFlags = ((uint8_t *) data )[0];
  adp->propFlags = ((uint8_t *) data )[1];
  data += 2 * sizeof( uint8_t );
  done += 2 * sizeof( uint8_t );

  /* explicit packet size */
  switch( ASF_DPSFLAG_SIZE_PACK( adp->sizeFlags ))
  {
	case 1:	adp->packetSize = *((uint8_t *) data );
		data += sizeof( uint8_t );
		done += sizeof( uint8_t );
		break;

	case 2:	adp->packetSize = *((uint16_t *) data );
		data += sizeof( uint16_t );
		done += sizeof( uint16_t );
		break;

	case 3:	adp->packetSize = *((uint32_t *) data );
		data += sizeof( uint32_t );
		done += sizeof( uint32_t );
		break;

	default:
		adp->packetSize = size;
		break;
  }

  /* explicit sequence number */
  switch( ASF_DPSFLAG_SIZE_SEQ( adp->sizeFlags ))
  {
	case 1:	adp->sequenceNum = *((uint8_t *) data );
		data += sizeof( uint8_t );
		done += sizeof( uint8_t );
		break;

	case 2:	adp->sequenceNum = *((uint16_t *) data );
		data += sizeof( uint16_t );
		done += sizeof( uint16_t );
		break;

	case 3:	adp->sequenceNum = *((uint32_t *) data );
		data += sizeof( uint32_t );
		done += sizeof( uint32_t );
		break;
  }

  /* explicit padding size */
  switch( ASF_DPSFLAG_SIZE_PADD( adp->sizeFlags ))
  {
	case 1:	adp->paddingSize = *((uint8_t *) data );
		data += sizeof( uint8_t );
		done += sizeof( uint8_t );
		break;

	case 2:	adp->paddingSize = *((uint16_t *) data );
		data += sizeof( uint16_t );
		done += sizeof( uint16_t );
		break;

	case 3:	adp->paddingSize = *((uint32_t *) data );
		data += sizeof( uint32_t );
		done += sizeof( uint32_t );
		break;
  }

  /* Send time */
  adp->sendTime = ((uint32_t *) data )[0];
  data += sizeof( uint32_t );
  done += sizeof( uint32_t );

  /* Duration */
  adp->duration = ((uint16_t *) data )[0];
  data += sizeof( uint16_t );
  done += sizeof( uint16_t );

  /* explicit segment number and properties */
  if( adp->sizeFlags & ASF_DPSFLAG_SEGMENTS )
  {
	adp->segments = *((uint8_t *) data );
	data += sizeof( uint8_t );
	done += sizeof( uint8_t );
  }
  else
  {
	/* If segments not explicit defined, assume 1 segment and
	* one word for dataSize entry in non-grouping segment */
	adp->segments = ASF_DSNFLAG_SIZESET( 0x10 ) | 0x01;
  }

#ifdef VERBOSE
  xmm_logging( 2, "ASF! parse data: sizeFlags %x\n", adp->sizeFlags );
  xmm_logging( 2, "ASF! parse data: propFlags %x\n", adp->propFlags );
  xmm_logging( 2, "ASF! parse data: sendtime %i ms\n", adp->sendTime );
  xmm_logging( 2, "ASF! parse data: duration %i ms\n", adp->duration );

  xmm_logging( 2, "ASF! parse data: packetSize %i (type %i)\n",
		    adp->packetSize, ASF_DPSFLAG_SIZE_PACK( adp->sizeFlags ));
  xmm_logging( 2, "ASF! parse data: sequenceNum %i (type %i)\n",
		    adp->sequenceNum, ASF_DPSFLAG_SIZE_SEQ( adp->sizeFlags ));
  xmm_logging( 2, "ASF! parse data: paddingSize %i (type %i)\n",
		    adp->paddingSize, ASF_DPSFLAG_SIZE_PADD( adp->sizeFlags ));

  xmm_logging( 2, "ASF! parse data: segments %i (type %i)\n",
		    adp->segments & ASF_DSNFLAG_NUMBER, 
				ASF_DSNFLAG_SIZE( adp->segments ));
#endif

  return done;
}

/*
 * Decode segment header
 */
int decode_asf_segment( uint8_t *data, asf_data_parse_t *adp, asf_segment_t *asegh, int size )
{
  int		done = 0;

  memset( asegh, 0, sizeof( asf_segment_t ));

  /* Get common segment data */
  asegh->stream = ((uint8_t *) data )[0];
  data += sizeof( uint8_t );
  done += sizeof( uint8_t );

  /* explicit sequence number */
  switch( ASF_DPPFLAG_SIZE_SEQ( adp->propFlags ))
  {
	case 1:	asegh->sequence = *((uint8_t *) data );
		data += sizeof( uint8_t );
		done += sizeof( uint8_t );
		break;

	case 2:	asegh->sequence = *((uint16_t *) data );
		data += sizeof( uint16_t );
		done += sizeof( uint16_t );
		break;

	case 3:	asegh->sequence = *((uint32_t *) data );
		data += sizeof( uint32_t );
		done += sizeof( uint32_t );
		break;
  }

  /* explicit offset */
  switch( ASF_DPPFLAG_SIZE_OFF( adp->propFlags ))
  {
	case 1:	asegh->offset = *((uint8_t *) data );
		data += sizeof( uint8_t );
		done += sizeof( uint8_t );
		break;

	case 2:	asegh->offset = *((uint16_t *) data );
		data += sizeof( uint16_t );
		done += sizeof( uint16_t );
		break;

	case 3:	asegh->offset = *((uint32_t *) data );
		data += sizeof( uint32_t );
		done += sizeof( uint32_t );
		break;
  }

  /* explicit replicated size */
  switch( ASF_DPPFLAG_SIZE_REP( adp->propFlags ))
  {
	case 1:	asegh->replicated = *((uint8_t *) data );
		data += sizeof( uint8_t );
		done += sizeof( uint8_t );
		break;

	case 2:	asegh->replicated = *((uint16_t *) data );
		data += sizeof( uint16_t );
		done += sizeof( uint16_t );
		break;

	case 3:	asegh->replicated = *((uint32_t *) data );
		data += sizeof( uint32_t );
		done += sizeof( uint32_t );
		break;
  }

  /* Grouped data */
  if( asegh->replicated != ASF_SREP_GROUPING )
  {
	asegh->objectSize = ((uint32_t *) data )[0];
	asegh->objectTime = ((uint32_t *) data )[1];

	data += asegh->replicated;
	done += asegh->replicated;
  }
  else
  {
	asegh->objectTime = asegh->offset;

	/* Skip presentation time delta */
	data += sizeof( uint8_t );
	done += sizeof( uint8_t );
  }

  /* Get segment datasize */
  if( adp->sizeFlags & ASF_DPSFLAG_SEGMENTS )
  {
	switch( ASF_DSNFLAG_SIZE( adp->segments ))
	{
	    case 1:	asegh->dataSize = ((uint8_t *) data )[0];
			data += sizeof( uint8_t );
			done += sizeof( uint8_t );
			break;

	    case 2:	asegh->dataSize = ((uint16_t *) data )[0];
			data += sizeof( uint16_t );
			done += sizeof( uint16_t );
			break;

	    case 3:	asegh->dataSize = ((uint32_t *) data )[0];
			data += sizeof( uint32_t );
			done += sizeof( uint32_t );
			break;

	    default:
			xmm_logging( 2, "ASF! Unknown dataSize entry size in asf_segment_header, internal error ???\n" );
			break;
	}
  }
  else	asegh->dataSize = size - done - adp->paddingSize;

#ifdef VERBOSE
  xmm_logging( 2, "ASF! Segment header: stream %x ( keyframe: %s )\n", asegh->stream & ASF_SSFLAG_STREAM, asegh->stream & ASF_SSFLAG_KEYFRAME ? "yes" : "no"  );
  xmm_logging( 2, "ASF! Segment header: sequence %i\n", asegh->sequence );

  if( asegh->replicated != ASF_SREP_GROUPING )
  {
	xmm_logging( 2, "ASF! Segment header: offset %i\n", asegh->offset );
	xmm_logging( 2, "ASF! Segment header: replicated %i\n", asegh->replicated );
	xmm_logging( 2, "ASF! Segment header: objectSize %i\n", asegh->objectSize );
	xmm_logging( 2, "ASF! Segment header: objectTime %i\n", asegh->objectTime );
  }
  else
  {
	xmm_logging( 2, "ASF! Segment header: <group> objectTime %i\n", asegh->objectTime );
	xmm_logging( 2, "ASF! Segment header: <group> replicated %i\n", asegh->replicated );
  }

  xmm_logging( 2, "ASF! Segment header: dataSize %i\n", asegh->dataSize );

#endif

  return done;
}

/*
 * Misc: Print GUID
 */

void print_guid( GUID_t *guid )
{
  int i;

  for( i = 0; asf_guids[i].guid.v1; i++ )
  {
	if( !memcmp( guid, &asf_guids[i].guid, sizeof( GUID_t )))
	{
	    xmm_logging( 1, "ASF! GUID: name = %s\n", asf_guids[i].name );
	    return;
	}
  }

  xmm_logging( 1, "ASF! GUID: %x, %x, %x,", guid->v1, guid->v2, guid->v3 );
  for( i = 0; i < 8; i++ )	xmm_logging( 1, " %x", guid->v4[i] );
  xmm_logging( 1, "\n" );
}
