/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * au.c
 * Sun / NeXT audio format DeMUX
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/endian.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include "au.h"

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif
 
/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_AudioInfo		ai;

    au_header			head;
    int				le;
    int				byterate;

    int				seekable;
    char 			*filename;
};


/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/* AU encoding -> XMM codec conversion */
struct au2xmm_format
{
    uint32_t			au_id;
    uint32_t			format;
} au_format_table[] =
{	
    { 0,		0 },
    { AU_DATA_ULAW,	XMM_AUDIO_FMT_ULAW },
    { AU_DATA_PCM8,	XMM_AUDIO_FMT_S8 },
    { AU_DATA_PCM16,	XMM_AUDIO_FMT_S16BE },
    { AU_DATA_PCM24,	XMM_AUDIO_FMT_S24BE },
    { AU_DATA_PCM32,	XMM_AUDIO_FMT_S32BE },
    { AU_DATA_FLOAT,	XMM_AUDIO_FMT_IEEE32BE },
    { AU_DATA_DOUBLE,	XMM_AUDIO_FMT_IEEE64BE },
    { 8,		0 },
    { 9,		0 },
    { 10,		0 },
    { 11,		0 },
    { 12,		0 },
    { 13,		0 },
    { 14,		0 },
    { 15,		0 },
    { 16,		0 },
    { 17,		0 },
    { 18,		0 },
    { 19,		0 },
    { 20,		0 },
    { 21,		0 },
    { 22,		0 },
    { AU_DATA_G721,	XMM_AUDIO_CODEC_G721 | XMM_AUDIO_MASK_BE | 4 },
    { AU_DATA_G722,	0 },
    { AU_DATA_G723_3,	XMM_AUDIO_CODEC_G723 | XMM_AUDIO_MASK_BE | 3 },
    { AU_DATA_G723_5,	XMM_AUDIO_CODEC_G723 | XMM_AUDIO_MASK_BE | 5 },
    { AU_DATA_ALAW,	XMM_AUDIO_FMT_ALAW }
};

/*
 * Prototypes
 */

/*
 * Initialize
 */
static XMM_PluginInput *au_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(AU) plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int au_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  char			*buffer, *ptr, *comment;

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(AU) Only 'read' mode supported" );

  /* Open file */
  priv->pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->pIO == NULL )	return XMM_RET_ERROR;

  /* Seekable ? */
  if( priv->pIO->Seek( priv->pIO, 0, XMM_SEEK_CUR ) == XMM_RET_OK )
	priv->seekable = 1;
  else	priv->seekable = 0;
#ifdef VERBOSE
  xmm_logging( 2, "AU! Stream is %s seekable\n", priv->seekable ? "" : "NOT" );
#endif

  priv->pIO->Read( priv->pIO, &priv->head, sizeof( au_header ), 1 );

  if( XMM_INT32_BE( priv->head.dwID ) == AU_FORMAT_ID )
  {
	priv->le = 0;

	priv->head.dwOffset = XMM_INT32_BE( priv->head.dwOffset );
	priv->head.dwSize = XMM_INT32_BE( priv->head.dwSize );
	priv->head.dwFormat = XMM_INT32_BE( priv->head.dwFormat );
	priv->head.dwRate = XMM_INT32_BE( priv->head.dwRate );
	priv->head.dwChannels = XMM_INT32_BE( priv->head.dwChannels );
  }
  else if( XMM_INT32_LE( priv->head.dwID ) == AU_FORMAT_ID )
  {
	xmm_logging( 1, "AU! INFO: data in little endian. Swapping header values...\n" );
	priv->le = 1;

	priv->head.dwOffset = XMM_INT32_LE( priv->head.dwOffset );
	priv->head.dwSize = XMM_INT32_LE( priv->head.dwSize );
	priv->head.dwFormat = XMM_INT32_LE( priv->head.dwFormat );
	priv->head.dwRate = XMM_INT32_LE( priv->head.dwRate );
	priv->head.dwChannels = XMM_INT32_LE( priv->head.dwChannels );
  }

#ifdef VERBOSE
  xmm_logging( 1, "AU! INFO: dwID = 0x%x [%s]\n", priv->head.dwID, xmm_FOURCC_string(priv->head.dwID));
  xmm_logging( 1, "AU! INFO: dwOffset = 0x%x [%i]\n", priv->head.dwOffset, priv->head.dwOffset );
  xmm_logging( 1, "AU! INFO: dwSize = 0x%x [%i]\n", priv->head.dwSize, priv->head.dwSize );
  xmm_logging( 1, "AU! INFO: dwFormat = 0x%x [%i]\n", priv->head.dwFormat, priv->head.dwFormat );
  xmm_logging( 1, "AU! INFO: dwRate = 0x%x [%i]\n", priv->head.dwRate, priv->head.dwRate );
  xmm_logging( 1, "AU! INFO: dwChannels = 0x%x [%i]\n", priv->head.dwChannels, priv->head.dwChannels );
#endif

  /* Read comment */
  if(( comment = malloc( priv->head.dwOffset - sizeof( au_header ) + 1 )) == NULL )
	return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(AU) comment" );
  
  priv->pIO->Read( priv->pIO, comment, priv->head.dwOffset - sizeof( au_header ), 1 );
  comment[priv->head.dwOffset - sizeof( au_header )] = '\0';
#ifdef VERBOSE
  xmm_logging( 1, "AU! INFO: comment = %s\n", comment );
#endif
  free( comment );

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* AudioInfo struct */
  priv->ai.fmt.format = au_format_table[priv->head.dwFormat].format;
  priv->ai.fmt.samprate = priv->head.dwRate;
  priv->ai.fmt.channels = priv->head.dwChannels;
  priv->ai.fmt.bitrate = priv->head.dwChannels * priv->head.dwRate *
			( priv->ai.fmt.format & XMM_AUDIO_MASK_SIZE );
  priv->ai.fmt.blockSize = priv->head.dwChannels * ( priv->ai.fmt.format & XMM_AUDIO_MASK_SIZE ) / 8;
  priv->ai.fmt.extraSize = 0;
  priv->ai.fmt.desc[0] = '\0';

  priv->ai.tSamples = (uint32_t)(((double)priv->head.dwSize / ( priv->ai.fmt.bitrate / 8 )) * priv->ai.fmt.samprate );
  priv->ai.tSize = priv->head.dwSize;
  priv->ai.offset = 0;

  /* Byte rate */
  priv->byterate = priv->ai.fmt.samprate * priv->ai.fmt.blockSize;

  return XMM_RET_OK;
}

/*
 * Close
 */
static int au_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  if( priv->filename )	free( priv->filename );
  if( priv->pIO )	priv->pIO->Close( priv->pIO );
  free( input );

  return XMM_RET_OK;
}

/*
 * Control
 */
static int au_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_AudioFormat	*format;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_AFORMAT:			/* Query sound format */
		format = (XMM_AudioFormat *)data;
		if(( format->format == priv->ai.fmt.format ) &&
		    ( format->samprate == priv->ai.fmt.samprate ) &&
		    ( format->channels == priv->ai.fmt.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_AFORMAT:			/* Get sound format */
		memcpy( data, &priv->ai.fmt, sizeof( XMM_AudioFormat ));
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->ai.fmt;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_CAPS:				/* Get capabilities */
		*((uint32_t *) data ) = XMM_INPUT_CF_MODE_READ | XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(AU) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AU) cmd ( 0x%x )" );
}

/*
 * Seek to position
 */
static int au_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  struct priv_t	*priv = input->sys.priv;

  return input->AudioSeek( input, astream, (uint32_t)( seek * priv->ai.tSize / priv->byterate * 1000 ));
}

/*
 * Get Information
 */
static int au_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )
  {
	strncpy( ci->name, priv->filename, XMM_CIL_NAME - 1 );
	ci->name[XMM_CIL_NAME - 1] = '\0';
	ci->author[0] = '\0';
	ci->album[0] = '\0';
	ci->content[0] = '\0';
	ci->copyright[0] = '\0';
	ci->software[0] = '\0';
	ci->date[0] = '\0';
	ci->comment[0] = '\0';
	ci->size = priv->pIO->Size( priv->pIO );
	ci->playtime = (double)priv->ai.tSamples / priv->ai.fmt.samprate;
  }

  return XMM_RET_OK;
}

/*
 * Get audio stream number
 */
static int au_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int au_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int au_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  struct priv_t *priv = input->sys.priv;
  uint32_t	read;

  read = priv->pIO->Read( priv->pIO, buffer, 1, size );
  if(( read == 0 ) && priv->pIO->Eof( priv->pIO ))	return XMM_RET_EOS;

  return read;
}

/*
 * Audio timestamp.
 */
static int au_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  struct priv_t	*priv = input->sys.priv;

  if( pts )	*pts = (uint32_t)((double)( priv->pIO->Tell( priv->pIO ) -
				priv->head.dwOffset ) / priv->byterate * 1000 );

  return XMM_RET_OK;
}

/*
 * Seek to position in audio stream
 */
static int au_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts )
{
  struct priv_t	*priv = input->sys.priv;
  double	bytes;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(AU) Stream is not seekable." );

  bytes = (double)pts * priv->byterate / 1000;

  /* I/O Seek */
  return priv->pIO->Seek( priv->pIO, priv->head.dwOffset + (uint32_t) bytes, XMM_SEEK_SET );
}

/*
 * AU files do not contain video data
 */
static int au_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int au_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int au_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  return XMM_RET_NOTSUPPORTED;
}

static int au_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int au_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if AU file
 */
static int au_Check( void *xmm, char *filename )
{
  uint32_t		dwID;
  XMM_PluginIO		*pIO;

  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  pIO->Read( pIO, &dwID, 4, 1 );
  pIO->Close( pIO );

  if( XMM_INT32_BE( dwID ) == AU_FORMAT_ID )	return 1;
  if( XMM_INT32_LE( dwID ) == AU_FORMAT_ID )	return 1;

  return 0;
}

/*
 * Get file info
 */
static int au_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */
XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"AU",
				"Input: AU",
				"Copyright (c) 2001 Arthur Kleer",
				NULL, NULL },
				au_Init, au_Open, au_Close,
				au_Control, au_Seek, au_Info, 
				au_AudioStreams, au_AudioInfo,
				au_AudioRead, au_AudioPTS, au_AudioSeek,
				au_VideoStreams, au_VideoInfo,
				au_VideoRead, au_VideoPTS, au_VideoSeek,
				au_Check, au_FileInfo };
