/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * avi.c
 * AVI input plugin
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include <libxmm/util/mutex.h>
#include "avi.h"


/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#define	VERBOSE
#endif

/* Change 'undef' in 'define' to debug idx1 entries, 'idx1' block content  */
#ifndef	AVI_DEBUG_IDX1_ENTRIES
#undef	AVI_DEBUG_IDX1_ENTRIES
#endif

/* Change 'undef' in 'define' to force idx1 ckecking, will check streamID and chunk size  */
#ifndef	AVI_CHECK_IDX1_ENTRIES
#define	AVI_CHECK_IDX1_ENTRIES
#endif

/* Change 'undef' in 'define' to use IO mutex*/
#ifndef	AVI_USE_MUTEX_IO
#define	AVI_USE_MUTEX_IO
#endif

/* */
#define	AVI_IDX1_BLOCK		4096
#define	MAX_STREAMS		0x10

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Types
 */

/*
 * Private plugin data
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_FileInfo		fi;

    /* Clip info */
    XMM_ClipInfo		ci;

    /* AVI Header */
    avi_header_t		avi_head;

    /* Misc stuff */
    char 			*filename;
    int				seekable;
    uint32_t			data_offset;
    uint32_t			data_size;

    /* Stream stuff */
    int				nStreams, astreams, vstreams;
    int				asIDX[MAX_STREAMS];
    int				vsIDX[MAX_STREAMS];

    /* Video / Audio streams */
    struct stream_t
    {
	avi_stream_header_t	ash;			/* stream header */
	char			*asf_extra;		/* extra data */

	union
	{
	    WAVEFORMATEX	wfhead;
	    BITMAPINFOHEADER	bmhead;
	} asf;

	XMM_AudioFormat		*af;
	XMM_VideoFormat		*vf;

	/* Index stuff */
	avi_index_entry_t	*idx1_chunk;		/* idx1 entries */
	uint32_t		*idx1_total;		/* audio bytes, including this chunk */
	uint32_t		idx1_entries;		/* number of idx1 entries */
	uint32_t		idx1_size;		/* idx1_chunk size in number of entries */
	uint32_t		position;		/* Current position */

	/* Misc */
	uint32_t		bposition;		/* (byte) position in current chunk */
	uint32_t		total_size;

	/* buffering */
	XMM_BufferQueue		bq;
	char			*rbuffer;
	int			rbuffersize;

    } 				aStream[MAX_STREAMS];

    /* Mutex */
#ifdef AVI_USE_MUTEX_IO
    XMM_Mutex			*mutex_io;
#endif
};

/*
 * Prototypes
 */

static struct stream_t *avi_lookup_astream( XMM_PluginInput *input, int stream );
static struct stream_t *avi_lookup_vstream( XMM_PluginInput *input, int stream );

static uint8_t	READ8( XMM_PluginIO *pIO );
static uint16_t	READ16( XMM_PluginIO *pIO );
static uint32_t	READ32( XMM_PluginIO *pIO );
static int	SEEK( XMM_PluginIO *pIO, long offset, int seekable );

static int	Read_avih( XMM_PluginIO *pIO, avi_header_t *avi_head, uint32_t size );
static int	Read_strh( XMM_PluginIO *pIO, avi_stream_header_t *ash, uint32_t size );
static int	Read_strf_vids( XMM_PluginIO *pIO, uint32_t size, avi_stream_header_t *ash, BITMAPINFOHEADER *avi_bm_head );
static int	Read_strf_auds( XMM_PluginIO *pIO, uint32_t size, avi_stream_header_t *ash, WAVEFORMATEX *avi_wf_head, char **asf_extra );
static int	Read_strd( XMM_PluginIO *pIO, unsigned char *data, uint32_t size );
static int	Read_INFO_chunk( XMM_PluginIO *pIO, XMM_ClipInfo *ci, uint32_t dwName, uint32_t dwSize );
static int	Read_idx1( XMM_PluginIO *pIO, uint32_t size, avi_index_entry_t **_idx1_entry, uint32_t *_idx1_entries, uint32_t *_idx1_relative, struct priv_t *priv );
static int	reconstruct_idx1( XMM_PluginIO *pIO, avi_index_entry_t **_idx1_entry, uint32_t *_idx1_entries, uint32_t *_idx1_relative, struct priv_t *priv );

static uint32_t codec_ms2xmm( uint16_t ms_codec );
static int	buffer_realloc( char **buffer, int *oldsize, int newsize );

static int	avi_AudioRead_raw( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags );
static int	avi_AudioSeek_nop( XMM_PluginInput *input, int stream, uint32_t pts );
static int	avi_VideoRead_raw( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags );
static int	avi_VideoSeek_nop( XMM_PluginInput *input, int stream, uint32_t frame );

static int	avi_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags );
static int	avi_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts );
static int	avi_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags );
static int	avi_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame );


/*
 * Initialize
 */
static XMM_PluginInput *avi_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(AVI) plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int avi_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  char			*buffer, *ptr;
  struct stream_t	*ps;
  avi_stream_header_t	ash;
  uint32_t		dwSize, dwName, dwTemp, last_LIST_size = 0, stream;
  int			loop = 1, i, idx1_build = 1, idx1_found = 0;
  unsigned char		strd[100];
  avi_index_entry_t	*idx1_entry = NULL;
  uint32_t		idx1_entries = 0, idx1_relative = 0;

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(AVI) Only 'read' mode supported" );

  /* Open file */
  priv->pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->pIO == NULL )	return XMM_RET_ERROR;

  /* Seekable ? */
  if( priv->pIO->Seek( priv->pIO, 0, XMM_SEEK_CUR ) == XMM_RET_OK )
	priv->seekable = 1;
  else	priv->seekable = 0;
#ifdef VERBOSE
  xmm_logging( 2, "AVI! Stream is %sseekable\n", priv->seekable ? "" : "NOT " );
#endif

  /* Clip info */
  memset( &priv->ci, 0, sizeof( XMM_ClipInfo ));

  /* Init data */
  priv->data_offset = 0;
  priv->data_size = 0;

  /* Parse file */
  while( loop )
  {
	dwName = READ32( priv->pIO );
	if( priv->pIO->Eof( priv->pIO ))	break;

#ifdef VERBOSE
	xmm_logging( 2, "AVI! Chunk ID = '%s' ( %lx ) at %x\n", xmm_FOURCC_string( dwName ), dwName, priv->pIO->Tell( priv->pIO ) - 4 );
#endif

	/*
	 * Handle list chunks
	 */
	switch( dwName )
	{
		case RIFF_ID_hdrl:
		case RIFF_ID_strl:
		case RIFF_ID_odml:
			dwName = 0;
			break;

		case RIFF_ID_INFO:
			dwName = 0;
			break;

		case RIFF_ID_movi:
			priv->data_offset = priv->pIO->Tell( priv->pIO );
			priv->data_size = last_LIST_size;
			xmm_logging( 3, "AVI! data_offset = %li ( %lx )\n", priv->data_offset, priv->data_offset );

			if( priv->seekable == 0 )	loop = 0;
			else	SEEK( priv->pIO, last_LIST_size - 4, priv->seekable );

			dwName = 0;
			break;

		/* Non-list chunk: read size */
		default:
			dwSize = READ32( priv->pIO );
#ifdef VERBOSE
			xmm_logging( 2, "AVI! ID = %x [%s] Chunk Size = %li ( %lx )\n", dwName, xmm_FOURCC_string(dwName), dwSize, dwSize );
#endif
			break;
	}

	if( dwName == 0 )	continue;

	switch( dwName )
	{
		case RIFF_ID_RIFF:
			dwTemp = READ32( priv->pIO );
			if( dwTemp != RIFF_ID_AVI )
			{
			    priv->pIO->Close( priv->pIO );
			    return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(AVI) No AVI RIFF form ( '%s' )\n", xmm_FOURCC_string( dwTemp ));
			}
			xmm_logging( 2, "AVI! RIFF Type = '%s'\n", xmm_FOURCC_string( dwTemp ));
			break;

		case RIFF_ID_LIST:
			last_LIST_size = dwSize;
			break;

		/* Header chunks */
		case RIFF_ID_avih:
			Read_avih( priv->pIO, &priv->avi_head, dwSize );
			break;
			

		case RIFF_ID_strh:
			Read_strh( priv->pIO, &ash, dwSize );
			break;

		case RIFF_ID_strf:
			switch( ash.fccType )
			{
			    case RIFF_STREAM_VIDEO:
				    Read_strf_vids( priv->pIO, dwSize, &ash, &priv->aStream[priv->nStreams].asf.bmhead );
				    break;

			    case RIFF_STREAM_AUDIO:
				    Read_strf_auds( priv->pIO, dwSize, &ash, &priv->aStream[priv->nStreams].asf.wfhead, &priv->aStream[priv->nStreams].asf_extra );
				    break;

			    default:
				priv->pIO->Close( priv->pIO );
				return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(AVI) Unsupported stream type '%s'", xmm_FOURCC_string( ash.fccType ));
			}

			memcpy( &priv->aStream[priv->nStreams].ash, &ash, sizeof( avi_stream_header_t ));
			priv->nStreams++;
			break;

		case RIFF_ID_strd:
			Read_strd( priv->pIO, strd, dwSize );
			break;

		/* 'movi' chunk index */
		case RIFF_ID_idx1:
			Read_idx1( priv->pIO, dwSize, &idx1_entry, &idx1_entries, &idx1_relative, priv );
			idx1_found = 1;
			break;

		/* INFO chunks */
		case RIFF_ID_INAM:
		case RIFF_ID_ISBJ:
		case RIFF_ID_IART:
		case RIFF_ID_ICOP:
		case RIFF_ID_ISFT:
		case RIFF_ID_ICMT:
			Read_INFO_chunk( priv->pIO, &priv->ci, dwName, dwSize );
			break;

		/* Chunks to ignore */
		case RIFF_ID_JUNK:
			xmm_logging( 2, "AVI! Ignoring chunk '%s' (%lx) [%li bytes]\n", xmm_FOURCC_string(dwName), dwName, dwSize );
			if( dwSize & 1 )	dwSize++;
			SEEK( priv->pIO, dwSize, priv->seekable );
			break;

		/* X-tended AVI */
		case RIFF_ID_dmlh:
			if( dwSize & 1 )	dwSize++;
			SEEK( priv->pIO, dwSize, priv->seekable );
			break;

		/* Unknown chunks */
		default:
			xmm_logging( 2, "AVI! Ignoring chunk '%s' (%lx) [%li bytes]\n", xmm_FOURCC_string(dwName), dwName, dwSize );
			if( dwSize & 1 )	dwSize++;
			SEEK( priv->pIO, dwSize, priv->seekable );
			break;
	}
  }

  if( priv->data_offset == 0 )
  {
	priv->pIO->Close( priv->pIO );
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(AVI) No 'movi' LIST found" );
  }

  /* Set default reader proc */
  input->VideoRead = avi_VideoRead_raw;
  input->VideoSeek = avi_VideoSeek_nop;
  input->AudioRead = avi_AudioRead_raw;
  input->AudioSeek = avi_AudioSeek_nop;

  /* Seek to the beginning of data */
  if( priv->seekable )
  {
	/* Rebuild index */
	if(( idx1_found == 0 ) && idx1_build )
	{
	    if( 0 < reconstruct_idx1( priv->pIO, &idx1_entry, &idx1_entries,
		    &idx1_relative, priv ))	idx1_found = 1;
	}

	/* Use idx1 chunk for reading */
	if( idx1_found == 1 )
	{
	    xmm_logging( 2, "AVI! INFO: Using IDX1 read mode\n" );
	    input->VideoRead = avi_VideoRead;
	    input->VideoSeek = avi_VideoSeek;
	    input->AudioRead = avi_AudioRead;
	    input->AudioSeek = avi_AudioSeek;

	    /* Allocate memory for each stream */
	    for( stream = 0; stream < priv->avi_head.dwStreams; stream++ )
	    {
		ps = &priv->aStream[stream];

		if(( ps->idx1_chunk = malloc( ps->idx1_size * sizeof( avi_index_entry_t ))) == NULL )
		    return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(AVI) Error allocating memory..." );

		if(( ps->idx1_total = malloc( ps->idx1_size * sizeof( uint32_t )  )) == NULL )
		    return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(AVI) audio total table." );
	    }

	    /* Split idx1 chunk into an idx1 table for each stream */
	    for( i = 0; i < idx1_entries; i++ )
	    {
		if( !isdigit( ((char *)&idx1_entry[i].dwChunkID)[0] ) ||
		    !isdigit( ((char *)&idx1_entry[i].dwChunkID)[1] ))
			continue;

		stream = strtol( xmm_FOURCC_string( idx1_entry[i].dwChunkID & 0xFFFF ), &ptr, 16 );
		if(( stream < 0 ) || ( stream >= MAX_STREAMS ))	continue;

		ps = &priv->aStream[stream];

		memcpy( &ps->idx1_chunk[ps->idx1_entries], &idx1_entry[i], sizeof( avi_index_entry_t ));
		ps->idx1_chunk[ps->idx1_entries].dwChunkOffset += idx1_relative;
		ps->idx1_total[ps->idx1_entries] = ps->total_size;
		ps->idx1_entries++;

		ps->total_size += idx1_entry[i].dwChunkLength;
	    }

	    free( idx1_entry );
	}

	/* Seek to start of 'movi' chunk */
	priv->pIO->Seek( priv->pIO, priv->data_offset, XMM_SEEK_SET );
  }

  /*
   * Initialize XMM_FileInfo struct / Build audio total bytes table
   */

  for( i = 0; i < priv->nStreams; i++ )
  {
	ps = &priv->aStream[i];

	/* Initialize data */
	ps->rbuffersize = -1;
	ps->rbuffer = NULL;

	/* Initialize buffer queue */
	xmmBQ_Init( &ps->bq );

	/* Initialize stream specific data and audio/video index table */
	switch( ps->ash.fccType )
	{
	    case RIFF_STREAM_VIDEO:
		    /* Allocate memory */
		    if(( ps->vf = malloc( sizeof( XMM_VideoFormat ))) == NULL )
			return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(AVI) Unable to allocate VideoFormat" );

		    /* Fill video format struct */
		    ps->vf->codec = ps->asf.bmhead.biCompression;
		    ps->vf->width = ps->asf.bmhead.biWidth;
		    ps->vf->height = ps->asf.bmhead.biHeight;
		    ps->vf->planes = ps->asf.bmhead.biPlanes;
		    ps->vf->bpp = ps->asf.bmhead.biBitCount;
		    ps->vf->imgSize = ps->asf.bmhead.biSizeImage;
		    ps->vf->bitrate = 0;
		    ps->vf->framerate = (double)1000000 / priv->avi_head.dwMicroSecPerFrame;
		    ps->vf->aspect_val = 1;
		    ps->vf->aspect_div = 1;
		    ps->vf->extraSize = 0;
		    ps->vf->extraType = 0;
		    ps->vf->desc[0] = '\0';

		    /* VideoInfo */
		    memcpy( &priv->fi.vi[priv->fi.vstreams].fmt, ps->vf, sizeof( XMM_VideoFormat ));
		    priv->fi.vi[priv->fi.vstreams].tFrames = priv->avi_head.dwTotalFrames;
		    priv->fi.vi[priv->fi.vstreams].tSize = priv->avi_head.dwLength;
		    priv->fi.vi[priv->fi.vstreams].offset = 0;

		    priv->vsIDX[priv->fi.vstreams++] = i;
		    break;

	    case RIFF_STREAM_AUDIO:
		    /* Allocate memory */
		    if(( ps->af = malloc( sizeof( XMM_AudioFormat ) + ps->asf.wfhead.cbSize )) == NULL )
			return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(AVI) Unable to allocate AudioFormat" );

		    /* Fill audio format struct */
		    ps->af->format = codec_ms2xmm( ps->asf.wfhead.wFormatTag );
		    ps->af->format |= ( ps->asf.wfhead.wBitsPerSample & XMM_AUDIO_MASK_SIZE );

		    if(( ps->asf.wfhead.wFormatTag == WAVE_FORMAT_IEEE_FLOAT ) ||
			(( ps->asf.wfhead.wBitsPerSample != 8 )))
				ps->af->format |= XMM_AUDIO_MASK_SIGNED;

		    ps->af->samprate = ps->asf.wfhead.nSamplesPerSec;
		    ps->af->channels = ps->asf.wfhead.nChannels;
		    ps->af->bitrate = ps->asf.wfhead.nAvgBytesPerSec * 8;
		    ps->af->blockSize = ps->asf.wfhead.nBlockAlign;
		    ps->af->extraSize = ps->asf.wfhead.cbSize;
		    ps->af->extraType = XMM_AUDIO_EXT_WAVE;
		    ps->af->desc[0] = '\0';

		    if( ps->asf_extra )
		        memcpy( &ps->af[1], ps->asf_extra, ps->af->extraSize );

		    /* AudioInfo struct */
		    memcpy( &priv->fi.ai[priv->fi.astreams].fmt, ps->af, sizeof( XMM_AudioFormat ));
		    priv->fi.ai[priv->fi.astreams].fmt.extraSize = 0;
		    priv->fi.ai[priv->fi.astreams].tSamples = ps->ash.dwLength / ( priv->fi.ai[priv->fi.astreams].fmt.bitrate / 8 ) * ps->asf.wfhead.nSamplesPerSec;
		    priv->fi.ai[priv->fi.astreams].tSize = ps->ash.dwLength;
		    priv->fi.ai[priv->fi.astreams].offset = 0;

		    priv->asIDX[priv->fi.astreams++] = i;
		    break;

	    default:
		    priv->pIO->Close( priv->pIO );
		    return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(AVI) Unsupported stream type '%s'", xmm_FOURCC_string( ash.fccType ));
	}
  }

  priv->astreams = priv->fi.astreams;
  priv->vstreams = priv->fi.vstreams;

  /* Create Mutex */
#ifdef AVI_USE_MUTEX_IO
  priv->mutex_io = xmmMutex_Create();
#endif

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* Clip info */
  ps = avi_lookup_vstream( input, 0 );
  priv->ci.size = priv->pIO->Size( priv->pIO );
  priv->ci.playtime = ((double)ps->ash.dwLength * ps->ash.dwScale / ps->ash.dwRate );

  if( priv->ci.content[0] == '\0' )
  {
	strncpy( priv->ci.content, priv->filename, XMM_CIL_CONTENT - 1 );
	priv->ci.content[XMM_CIL_CONTENT - 1] = '\0';
  }

  return XMM_RET_OK;
}

/*
 * Close
 */
static int avi_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;
  int		i;

  /* Close file and free resources */
  if( priv->filename )	free( priv->filename );
  if( priv->pIO )	priv->pIO->Close( priv->pIO );

  for( i = 0; i < priv->nStreams; i++ )
  {
      if( priv->aStream[i].af )		free( priv->aStream[i].af );
      if( priv->aStream[i].vf )		free( priv->aStream[i].vf );
      if( priv->aStream[i].rbuffer )	free( priv->aStream[i].rbuffer );
      if( priv->aStream[i].idx1_chunk )	free( priv->aStream[i].idx1_chunk );
      if( priv->aStream[i].idx1_total )	free( priv->aStream[i].idx1_total );
  }

#ifdef AVI_USE_MUTEX_IO
  /* Destroy mutex */
  xmmMutex_Destroy( priv->mutex_io );
#endif

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int avi_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_AudioFormat	*format;
  struct stream_t	*ps;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:			/* Query graph format */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_AFORMAT:			/* Query sound format */
		format = (XMM_AudioFormat *)data;
		if(( format->format == priv->fi.ai[param].fmt.format ) &&
		    ( format->samprate == priv->fi.ai[param].fmt.samprate ) &&
		    ( format->channels == priv->fi.ai[param].fmt.channels ))
			return XMM_CTLRET_TRUE;

		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:			/* Query flip state */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_GFORMAT:			/* Get graph format */
		ps = avi_lookup_vstream( input, param );
		memcpy( data, ps->vf, sizeof( XMM_VideoFormat ));
		((XMM_VideoFormat *)data)->extraSize = 0;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_GFORMAT_PTR:			/* Get graph format */
		ps = avi_lookup_vstream( input, param );
		*((XMM_VideoFormat **) data) = ps->vf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT:			/* Get sound format */
		ps = avi_lookup_astream( input, param );
		memcpy( data, ps->af, sizeof( XMM_AudioFormat ));
		((XMM_AudioFormat *)data)->extraSize = 0;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		ps = avi_lookup_astream( input, param );
		*((XMM_AudioFormat **) data) = ps->af;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_CAPS:				/* Get capabilities */
		*((uint32_t *) data ) =	XMM_INPUT_CF_MODE_READ |
					XMM_INPUT_CF_AUDIO |
					XMM_INPUT_CF_VIDEO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:			/* Set graph format */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(AVI) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AVI) cmd ( 0x%x )" );
}

/*
 * Seek to position
 */
static int avi_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;
  uint32_t		frame, pts;

  /*
   * Seek video
   */
  ps = avi_lookup_vstream( input, vstream );

  /* Calculate frame */
  frame = (uint32_t) ( seek * ps->ash.dwLength );

  /* Check for EOS */
  if( frame >= ps->idx1_entries )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AVI) Invalid frame number: %i, Stream contains only %i frames", frame, ps->idx1_entries );
	return XMM_RET_INVALID_ARG;
  }

  /* Find next keyframe */
  while((( ps->idx1_chunk[frame].dwFlags & AVIIF_KEYFRAME ) == 0 ) &&
		( frame < ps->idx1_entries ))	frame++;

  /* End of stream, rewind stream */
  if( frame >= ps->idx1_entries )	return XMM_RET_EOS;

  /* Set frame number */
  ps->position = frame;

  /*
   * Seek Audio
   */
  if( priv->fi.astreams > 0 )
  {
	input->VideoPTS( input, vstream, &pts );
	input->AudioSeek( input, astream, pts );
  }

  return XMM_RET_OK;
}

/*
 * Get Information
 */
static int avi_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;
  double		time, total;

  if( ci )	memcpy( ci, &priv->ci, sizeof( XMM_ClipInfo ));  

  ps = avi_lookup_vstream( input, 0 );
  time = ((double)ps->position * ps->ash.dwScale / ps->ash.dwRate );
  total = ((double)ps->ash.dwLength * ps->ash.dwScale / ps->ash.dwRate );

  if( seekval )		*seekval = time / total;

  return XMM_RET_OK;
}

/*
 * Get audio stream number
 */
static int avi_AudioStreams( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  return priv->fi.astreams;
}

/*
 * Get audio stream information
 */
static int avi_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;

  ps = avi_lookup_astream( input, stream );

  if( ai )		memcpy( ai, &priv->fi.ai[stream], sizeof( XMM_AudioInfo ));

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int avi_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  struct priv_t			*priv = input->sys.priv;
  struct avi_index_entry_s	*idx;
  struct stream_t		*ps;
  uint32_t			demuxed = 0, proc, left;
  int				ret;
#ifdef AVI_CHECK_IDX1_ENTRIES
  uint32_t			dwName, dwSize;
#endif

  ps = avi_lookup_astream( input, stream );

  /* Get data from next audio chunks */
  while( demuxed < size )
  {
	left = ps->idx1_chunk[ps->position].dwChunkLength - ps->bposition;

	/* No more data left in chunk */
	if( left == 0 )
	{

	    /* Check for EOS */
	    if(( ps->position + 1 ) >= ps->idx1_entries )
	    {
		if( demuxed > 0 )	break;
		else	return XMM_RET_EOS;
	    }

	    /* Next chunk */
	    ps->position++;
	    ps->bposition = 0;

	    continue;	/* recalculate left */
	}

	/* Get idx1 entry */
	idx = &ps->idx1_chunk[ ps->position ];

	/* Lock IO */
#ifdef AVI_USE_MUTEX_IO
	xmmMutex_Lock( priv->mutex_io );
#endif

#ifdef AVI_CHECK_IDX1_ENTRIES

	/* Seek to position */
	priv->pIO->Seek( priv->pIO, idx->dwChunkOffset, XMM_SEEK_SET );

	/* Get chunk ID and size */
	dwName = READ32( priv->pIO );
	dwSize = READ32( priv->pIO );

	/* Check for conflicts */
	if( dwName != idx->dwChunkID )
	{
	    xmm_logging( 1, "AVI! Audio ChunkID conflict! IDX1 = %x RAW = %x\n", idx->dwChunkID, dwName );
	}

	if(( dwSize != idx->dwChunkLength ) && (( dwSize + 1 ) != idx->dwChunkLength ))
	{
	    xmm_logging( 1, "AVI! Audio ChunkSize conflict! IDX1 = %x RAW = %x\n", idx->dwChunkLength, dwSize );
	}

#endif

	/* Seek to position */
	priv->pIO->Seek( priv->pIO, idx->dwChunkOffset + 2 * sizeof( uint32_t ) + ps->bposition, XMM_SEEK_SET );

	/* Read some bytes into the buffer */
	proc = left;
	if( proc > ( size - demuxed ))	proc = size - demuxed;

	ret = priv->pIO->Read( priv->pIO, buffer, 1, proc );
	if( ret != proc )
	    xmm_logging( 1, "AVI! I/O Read didn't return the requested number of bytes. Corrupted file ?\n" );

	/* Unlock IO */
#ifdef AVI_USE_MUTEX_IO
	xmmMutex_Unlock( priv->mutex_io );
#endif

	demuxed += ret;
	buffer += ret;
	ps->bposition += ret;
  }

  return demuxed;
}

/*
 * Read audio data ( sequential )
 */
static int avi_AudioRead_raw( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  struct priv_t			*priv = input->sys.priv;
  struct stream_t		*ps, *psb;
  uint32_t			demuxed = 0, proc, nextChunk, i;
  char				*ptr, *endptr;
  int				ret;
  uint32_t			dwName, dwSize;

  /* Check for end of file */
  if( priv->pIO->Eof( priv->pIO ))
  {
	xmm_logging( 1, "AVI! End of File. Corrupted file ? Returning XMM_RET_EOS\n" );
	return XMM_RET_EOS;
  }

  ps = avi_lookup_astream( input, stream );

  /* Resize buffer if needed */
  if( buffer_realloc( &ps->rbuffer, &ps->rbuffersize, size  ) < 0 )
	return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(AVI) read buffer" );

  /* Get reamaining audio chunk bytes */
  if( xmmBQ_Size( &ps->bq ) > 0 )
  {
	proc = xmmBQ_Size( &ps->bq );
	if( proc > size )	proc = size;

	ret = xmmBQ_Read( &ps->bq, buffer, proc );
	if( ret != proc )
	{
	    xmm_logging( 1, "AVI! Unable to get requested number of bytes from buffer. Must NOT happen!!!\n" );
	}

	demuxed += ret;
	buffer += ret;
  }

  /* Get data from next audio chunks */
  while( demuxed < size )
  {
	/* Lock IO */
#ifdef AVI_USE_MUTEX_IO
	xmmMutex_Lock( priv->mutex_io );
#endif

	dwName = READ32( priv->pIO );
	dwSize = READ32( priv->pIO );

	if( priv->pIO->Eof( priv->pIO ))
	{
	    xmm_logging( 1, "AVI! End of File. Corrupted file ?\n" );
	    break;
	}

#ifdef DEBUG
	xmm_logging( 3, "AVI! Chunk ID = '%s' ( %lx ), File pos = %lx, Size = %li ( %lx )\n", xmm_FOURCC_string( dwName ), dwName, priv->pIO->Tell( priv->pIO ) - 4, dwSize, dwSize );
#endif

	/* AVI is WORD aligned, NOTE: paddding byte must NOT be used */
	nextChunk = dwSize + (( dwSize & 1 ) ? 1 : 0 );

	/* Lookup stream */
	i = strtol( xmm_FOURCC_string( dwName & 0xFFFF ), &endptr, 16 );
	if( i >= priv->nStreams )	return	XMM_RET_INVALID_ARG;
	psb = &priv->aStream[i];

	/* Desired stream */
	if( psb == ps )
	{
	    /* Read some byte into the buffer */
	    proc = dwSize;
	    if( proc > ( size - demuxed ))	proc = size - demuxed;

	    ret = priv->pIO->Read( priv->pIO, buffer, 1, proc );
	    if( ret != proc )
	    {
		xmm_logging( 1, "AVI! I/O Read didn't return the requested number of bytes. Corrupted file ?\n" );
	    }

	    demuxed += ret;
	    buffer += ret;
	    nextChunk -= ret;

	    /* Buffer remaining chunk data */
	    if( ret < dwSize )
	    {
		proc = dwSize - ret;	/* bytes to buffer */

		if(( ptr = malloc( proc )) == NULL )
		    return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(AVI) Error allocating memory..." );

		ret = priv->pIO->Read( priv->pIO, ptr, 1, proc );
		if( ret != proc )
		{
		    xmm_logging( 1, "AVI! I/O Read didn't return the requested number of bytes. Corrupted file ?\n" );
		}

		nextChunk -= ret;
		xmmBQ_Add( &priv->aStream[stream].bq, ptr, ret );
		free( ptr );
	    }
	}
	else
	{
	    ptr = xmmBQ_Add( &psb->bq, NULL, dwSize );

	    ret = priv->pIO->Read( priv->pIO, ptr, 1, dwSize );
	    if( ret != dwSize )
	    {
		xmm_logging( 1, "AVI! I/O Read didn't return the requested number of bytes ( %i of %i ). Corrupted file ?\n", ret, dwSize );
	    }

	    nextChunk -= ret;
	}

	SEEK( priv->pIO, nextChunk, priv->seekable );

	/* Unlock IO */
#ifdef AVI_USE_MUTEX_IO
	xmmMutex_Unlock( priv->mutex_io );
#endif
  }

  return demuxed;
}

/*
 * Audio timestamp.
 */
static int avi_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  struct stream_t	*ps;

  if( pts == NULL )
	return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AVI) audioPTS = NULL" );

  ps = avi_lookup_astream( input, stream );

  if( ps->ash.dwSampleSize == 0 )
  {
	*pts = (uint32_t)((double) ps->position * 1000
		* ps->ash.dwScale / ps->ash.dwRate );
  }
  else
  {
	*pts = (uint32_t)((double)
		( ps->idx1_total[ps->position] + ps->bposition ) * 1000
		/ ps->ash.dwSampleSize * ps->ash.dwScale / ps->ash.dwRate );
  }

  return XMM_RET_OK;
}

/*
 * Seek to position in audio stream
 */
static int avi_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts )
{
  struct stream_t	*ps;
  uint32_t		cpts;

  ps = avi_lookup_astream( input, stream );

  for( cpts = 0, ps->position = 0; cpts < pts; ps->position++ )
  {
	if( ps->position >= ps->idx1_entries )	return XMM_RET_EOS;

	input->AudioPTS( input, stream, &cpts );
  }

  /* Free buffered data */
  xmmBQ_Free( &ps->bq );

  /* beginning of chunk */
  ps->bposition = 0;

  return XMM_RET_OK;
}

/*
 * Seek Audio ( dummy function: no seeking supported )
 */
static int avi_AudioSeek_nop( XMM_PluginInput *input, int stream, uint32_t position )
{
  return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(AVI) RAW mode. No seeking supported.\n" );
}

/*
 * Get video stream number
 */
static int avi_VideoStreams( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  return priv->fi.vstreams;
}

/*
 * Get video stream information
 */
static int avi_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  struct priv_t		*priv = input->sys.priv;
  struct stream_t	*ps;

  ps = avi_lookup_vstream( input, stream );

  /* */
  if( vi )	memcpy( vi, &priv->fi.vi[stream], sizeof( XMM_VideoInfo ));
  if( cFrame )	*cFrame = ps->position;

  return XMM_RET_OK;
}

/*
 * Read video data
 */
static int avi_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  struct priv_t			*priv = input->sys.priv;
  struct avi_index_entry_s	*idx;
  struct stream_t		*ps;
  uint32_t			dwName, dwSize;
  int				ret;

  ps = avi_lookup_vstream( input, stream );

  /* Check for EOS */
  if(( ps->position + 1 ) >= ps->idx1_entries )	return XMM_RET_EOS;

  /* Get idx1 entry */
  idx = &ps->idx1_chunk[ ps->position++ ];

#ifdef AVI_USE_MUTEX_IO
  xmmMutex_Lock( priv->mutex_io );
#endif

  /* Seek to position */
  priv->pIO->Seek( priv->pIO, idx->dwChunkOffset, XMM_SEEK_SET );

  /* Get chunk ID and size */
  dwName = READ32( priv->pIO );
  dwSize = READ32( priv->pIO );

#ifdef AVI_CHECK_IDX1_ENTRIES

  /* Check for conflicts */
  if( dwName != idx->dwChunkID )
  {
	xmm_logging( 1, "AVI! Video ChunkID conflict! IDX1 = %x RAW = %x\n", idx->dwChunkID, dwName );
  }

  if(( dwSize != idx->dwChunkLength ) && (( dwSize + 1 ) != idx->dwChunkLength ))
  {
	xmm_logging( 1, "AVI! Video ChunkSize conflict! IDX1 = %x RAW = %x\n", idx->dwChunkLength, dwSize );
  }

#endif

  /* Read data */
  ret = priv->pIO->Read( priv->pIO, buffer[0], 1, dwSize );

#ifdef AVI_USE_MUTEX_IO
  xmmMutex_Unlock( priv->mutex_io );
#endif

  return ret;
}

/*
 * Read video data ( sequential )
 */
static int avi_VideoRead_raw( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  struct priv_t			*priv = input->sys.priv;
  struct stream_t		*ps, *psb;
  uint32_t			dwName, dwSize, dsize, nextChunk, i;
  int				ret;
  char				*ptr, *endptr;

  /* Check for end of file */
  if( priv->pIO->Eof( priv->pIO ))
  {
	xmm_logging( 1, "AVI! End of File. Corrupted file ? Returning XMM_RET_EOS\n" );
	return XMM_RET_EOS;
  }

  ps = avi_lookup_vstream( input, stream );

  /* No buffered frame */
  dsize = 0;

  /* Get buffered frame */
  if( xmmBQ_Size( &ps->bq ) > 0 )
  {
	ret = xmmBQ_HeadSize( &ps->bq );

	if( buffer_realloc( &ps->rbuffer, &ps->rbuffersize, ret ) < 0 )
		return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(AVI) read buffer" );

	dsize = xmmBQ_HeadRead( &ps->bq, ps->rbuffer, ps->rbuffersize );
	if( dsize < 0 )
	    xmm_logging( 1, "AVI! Unable to get buffer. Must NOT happen!!!\n" );
  }

  /* Get frame from next video chunks */
  while( dsize == 0 )
  {
	/* Lock IO */
#ifdef AVI_USE_MUTEX_IO
	xmmMutex_Lock( priv->mutex_io );
#endif

	dwName = READ32( priv->pIO );
	dwSize = READ32( priv->pIO );

	if( priv->pIO->Eof( priv->pIO ))
	{
	    xmm_logging( 1, "AVI! End of File. Corrupted file ?\n" );
	    break;
	}

#ifdef DEBUG
	xmm_logging( 3, "AVI! Chunk ID = '%s' ( %lx ), File pos = %lx, Size = %li ( %lx )\n", xmm_FOURCC_string( dwName ), dwName, priv->pIO->Tell( priv->pIO ) - 4, dwSize, dwSize );
#endif

	/* AVI is WORD aligned, NOTE: paddding byte must NOT be used */
	nextChunk = dwSize + (( dwSize & 1 ) ? 1 : 0 );

	/* Lookup stream */
	i = strtol( xmm_FOURCC_string( dwName & 0xFFFF ), &endptr, 16 );
	if( i >= priv->nStreams )	return	XMM_RET_INVALID_ARG;
	psb = &priv->aStream[i];

	/* Desired stream */
	if( psb == ps )
	{
	    if( buffer_realloc( &ps->rbuffer, &ps->rbuffersize, dwSize  ) < 0 )
		return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(AVI) read buffer" );

	    ret = priv->pIO->Read( priv->pIO, ps->rbuffer, 1, dwSize );
	    nextChunk -= ret;
	    dsize = ret;

	}
	else
	{
	    ptr = xmmBQ_Add( &psb->bq, NULL, dwSize );

	    ret = priv->pIO->Read( priv->pIO, ptr, 1, dwSize );
	    if( ret != dwSize )
	    {
		xmm_logging( 1, "AVI! I/O Read didn't return the requested number of bytes. Corrupted file ?\n" );
	    }

	    nextChunk -= ret;
	}

	SEEK( priv->pIO, nextChunk, priv->seekable );

	/* Unlock IO */
#ifdef AVI_USE_MUTEX_IO
	xmmMutex_Unlock( priv->mutex_io );
#endif
  }

  /* Update frame number */
  ps->position++;

  /* Return data */
  memcpy( buffer, ps->rbuffer, dsize );

  return dsize;
}

/*
 * Video PTS
 */
static int avi_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  struct stream_t	*ps;

  if( videoPTS == NULL )
	return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AVI) videoPTS = NULL" );

  ps = avi_lookup_vstream( input, stream );

  *videoPTS = (uint32_t)((double)ps->position * 1000 * ps->ash.dwScale / ps->ash.dwRate );

  return XMM_RET_OK;
}

/*
 * Seek to position in video stream
 */
static int avi_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  struct stream_t	*ps;

  ps = avi_lookup_vstream( input, stream );

  /* Check for EOS */
  if( frame >= ps->idx1_entries )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AVI) Invalid frame number: %i, Stream contains only %i frames", frame, ps->idx1_entries );
	return XMM_RET_INVALID_ARG;
  }

  /* Set frame number */
  ps->position = frame;

  return frame;
}

/*
 * Seek video ( dummy function: no seeking supported )
 */
static int avi_VideoSeek_nop( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(AVI) RAW mode. No seeking supported.\n" );
}

/*
 * Check if AVI file
 */
static int avi_CheckFile( void *xmm, char *filename )
{
  uint32_t		dwSize, dwName, dwTemp;
  XMM_PluginIO		*pIO;

  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  pIO->Read( pIO, &dwName, 4, 1 );
  pIO->Read( pIO, &dwSize, 4, 1 );
  pIO->Read( pIO, &dwTemp, 4, 1 );

  pIO->Close( pIO );

  if(( dwName == RIFF_ID_RIFF ) && ( dwTemp == RIFF_ID_AVI ))	return 1;
  else	return 0;
}

/*
 * Get file info
 */
static int avi_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"AVI",
				"Input: AVI",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				avi_Init, avi_Open, avi_Close,
				avi_Control, avi_Seek, avi_Info, 
				avi_AudioStreams, avi_AudioInfo,
				avi_AudioRead, avi_AudioPTS, avi_AudioSeek,
				avi_VideoStreams, avi_VideoInfo,
				avi_VideoRead, avi_VideoPTS, avi_VideoSeek,
				avi_CheckFile, avi_FileInfo };


/*
 * Internal
 */

/*
 * Misc functions for file I/O
 */

static uint8_t READ8( XMM_PluginIO *pIO )
{
  uint8_t tmp;

  pIO->Read( pIO, &tmp, 1, 1 );

  return tmp;
}

static uint16_t READ16( XMM_PluginIO *pIO )
{
  uint16_t tmp;

  pIO->Read( pIO, &tmp, 2, 1 );

  return tmp;
}

static uint32_t READ32( XMM_PluginIO *pIO )
{
  uint32_t tmp;

  pIO->Read( pIO, &tmp, 4, 1 );

  return tmp;
}

static int SEEK( XMM_PluginIO *pIO, long offset, int seekable )
{
  if( seekable )	return pIO->Seek( pIO, offset, XMM_SEEK_CUR );
  else
  {
	unsigned char tmp;
	int i;

	for( i = 0; i < offset; i++ )	pIO->Read( pIO, &tmp, 1, 1 );
	return i;
  }
}

/*
 * Lookup audio stream
 */
static struct stream_t *avi_lookup_astream( XMM_PluginInput *input, int stream )
{
  int			idx;
  struct priv_t		*priv = input->sys.priv;

  /* Lookup stream */
  if( stream >= priv->astreams )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AVI) Wrong stream number: %i, File has only %i audio streams.", stream, priv->astreams );
	return NULL;
  }
  idx = priv->asIDX[stream];			/* Audio */

  if( idx >= priv->nStreams )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AVI) Wrong stream number: %i, Internal error (?).", idx );
	return NULL;
  }

  return &priv->aStream[idx];
}

/*
 * Lookup video stream
 */
static struct stream_t *avi_lookup_vstream( XMM_PluginInput *input, int stream )
{
  int			idx;
  struct priv_t		*priv = input->sys.priv;

  /* Lookup stream */
  if( stream >= priv->vstreams )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AVI) Wrong stream number: %i, File has only %i video streams.", stream, priv->vstreams );
	return NULL;
  }
  idx = priv->vsIDX[stream];			/* Video */

  if( idx >= priv->nStreams )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(AVI) Wrong stream number: %i, Internal error (?).", idx );
	return NULL;
  }

  return &priv->aStream[idx];
}

/*
 * Read AVI headers
 */

/*
 * Main AVI header
 */
static int Read_avih( XMM_PluginIO *pIO, avi_header_t *avi_head, uint32_t size )
{
  if( size != 0x38 )
  {
	xmm_logging( 1, "AVI! avih: FILE ERROR: Size not 56 bytes ( is %i )\n", size );
	return 0;
  }

  avi_head->dwMicroSecPerFrame		= READ32( pIO );
  avi_head->dwMaxBytesPerSec		= READ32( pIO );
  avi_head->dwReserved1			= READ32( pIO );
  avi_head->dwFlags			= READ32( pIO );
  avi_head->dwTotalFrames		= READ32( pIO );
  avi_head->dwInitialFrames		= READ32( pIO );
  avi_head->dwStreams			= READ32( pIO );
  avi_head->dwSuggestedBufferSize	= READ32( pIO );
  avi_head->dwWidth			= READ32( pIO );
  avi_head->dwHeight			= READ32( pIO );
  avi_head->dwScale			= READ32( pIO );
  avi_head->dwRate			= READ32( pIO );
  avi_head->dwStart			= READ32( pIO );
  avi_head->dwLength			= READ32( pIO );

#ifdef VERBOSE
  xmm_logging( 2, "AVI! avih: MicroSecs/Frame = %li\n", avi_head->dwMicroSecPerFrame );
  xmm_logging( 2, "AVI! avih: MaxBytes/Sec = %li\n", avi_head->dwMaxBytesPerSec );
  xmm_logging( 2, "AVI! avih: Reserved = %li\n", avi_head->dwReserved1 );
  xmm_logging( 2, "AVI! avih: Flags = %lx\n", avi_head->dwFlags );
  xmm_logging( 2, "AVI! avih: TotalFrames = %li\n", avi_head->dwTotalFrames );
  xmm_logging( 2, "AVI! avih: InitialFrames = %li\n", avi_head->dwInitialFrames );
  xmm_logging( 2, "AVI! avih: Streams = %li\n", avi_head->dwStreams );
  xmm_logging( 2, "AVI! avih: Suggested buffer size = %li\n", avi_head->dwSuggestedBufferSize	);
  xmm_logging( 2, "AVI! avih: Width = %li\n", avi_head->dwWidth );
  xmm_logging( 2, "AVI! avih: Height = %li\n", avi_head->dwHeight );
  xmm_logging( 2, "AVI! avih: Scale = %li\n", avi_head->dwScale );
  xmm_logging( 2, "AVI! avih: Rate = %li\n", avi_head->dwRate );
  xmm_logging( 2, "AVI! avih: Start = %li\n", avi_head->dwStart );
  xmm_logging( 2, "AVI! avih: Length = %li\n", avi_head->dwLength );
#endif

  return 1;
}

/*
 * Stream header
 */
static int Read_strh( XMM_PluginIO *pIO, avi_stream_header_t *ash, uint32_t size )
{
  int tsize;

  if( size < 0x30 )
  {
	xmm_logging( 1, "AVI! avih: FILE ERROR: Size < 48 bytes ( is %i )\n", size );
	return 0;
  }

  ash->fccType			= READ32( pIO );
  ash->fccHandler		= READ32( pIO );
  ash->dwFlags			= READ32( pIO );
  ash->dwReserved1		= READ32( pIO );
  ash->dwInitialFrames		= READ32( pIO );
  ash->dwScale			= READ32( pIO );
  ash->dwRate			= READ32( pIO );
  ash->dwStart			= READ32( pIO );
  ash->dwLength			= READ32( pIO );
  ash->dwSuggestedBufferSize	= READ32( pIO );
  ash->dwQuality		= READ32( pIO );
  ash->dwSampleSize		= READ32( pIO );

#ifdef VERBOSE
  xmm_logging( 2, "AVI! strh: Type = '%s' ( %lx )\n", xmm_FOURCC_string( ash->fccType ), ash->fccType);
  xmm_logging( 2, "AVI! strh: Handler = '%s' ( %lx )\n", xmm_FOURCC_string( ash->fccHandler ), ash->fccHandler);
  xmm_logging( 2, "AVI! strh: Flags = %lx\n", ash->dwFlags );
  xmm_logging( 2, "AVI! strh: Reserved = %li\n", ash->dwReserved1 );
  xmm_logging( 2, "AVI! strh: InitialFrames = %li\n", ash->dwInitialFrames );
  xmm_logging( 2, "AVI! strh: Scale = %li\n", ash->dwScale );
  xmm_logging( 2, "AVI! strh: Rate = %li\n", ash->dwRate );
  xmm_logging( 2, "AVI! strh: Start = %li\n", ash->dwStart );
  xmm_logging( 2, "AVI! strh: Length = %li\n", ash->dwLength );
  xmm_logging( 2, "AVI! strh: Suggested buffer size = %li\n", ash->dwSuggestedBufferSize	);
  xmm_logging( 2, "AVI! strh: Quality = %li\n", ash->dwQuality );
  xmm_logging( 2, "AVI! strh: Sample size = %li\n", ash->dwSampleSize );
#endif

  if( size & 1 )	size++;
  for( tsize = 48; tsize < size; tsize++ )	READ8( pIO );

  return 1;
}

/*
 * Read stream format: video
 */
static int Read_strf_vids( XMM_PluginIO *pIO, uint32_t size, avi_stream_header_t *ash, BITMAPINFOHEADER *avi_bm_head )
{
  int tsize;

  avi_bm_head->biSize		= READ32( pIO );
  avi_bm_head->biWidth		= READ32( pIO );
  avi_bm_head->biHeight		= READ32( pIO );
  avi_bm_head->biPlanes		= READ16( pIO );
  avi_bm_head->biBitCount	= READ16( pIO );
  avi_bm_head->biCompression	= READ32( pIO );
  avi_bm_head->biSizeImage	= READ32( pIO );
  avi_bm_head->biXPelsPerMeter	= READ32( pIO );
  avi_bm_head->biYPelsPerMeter	= READ32( pIO );
  avi_bm_head->biClrUsed	= READ32( pIO );
  avi_bm_head->biClrImportant	= READ32( pIO );

#ifdef VERBOSE
  xmm_logging( 2, "AVI! strf_vid: Size = %li\n", avi_bm_head->biSize );
  xmm_logging( 2, "AVI! strf_vid: Width = %li\n", avi_bm_head->biWidth );
  xmm_logging( 2, "AVI! strf_vid: Height = %li\n", avi_bm_head->biHeight );
  xmm_logging( 2, "AVI! strf_vid: Planes = %i\n", avi_bm_head->biPlanes );
  xmm_logging( 2, "AVI! strf_vid: BitCount = %i\n", avi_bm_head->biBitCount );
  xmm_logging( 2, "AVI! strf_vid: Compression = '%s' ( %lx )\n", xmm_FOURCC_string( avi_bm_head->biCompression ), avi_bm_head->biCompression );
  xmm_logging( 2, "AVI! strf_vid: SizeImage = %li\n", avi_bm_head->biSizeImage );
  xmm_logging( 2, "AVI! strf_vid: XPels/m = %li\n", avi_bm_head->biXPelsPerMeter );
  xmm_logging( 2, "AVI! strf_vid: YPels/m = %li\n", avi_bm_head->biYPelsPerMeter );
  xmm_logging( 2, "AVI! strf_vid: ClrUsed = %li\n", avi_bm_head->biClrUsed );
  xmm_logging( 2, "AVI! strf_vid: ClrImportant = %li\n", avi_bm_head->biClrImportant );
#endif

  if( size & 1 )	size++;
  for( tsize = 40; tsize < size; tsize++ )	READ8( pIO );

  return 1;
}

/*
 * Read stream format: audio
 */
static int Read_strf_auds( XMM_PluginIO *pIO, uint32_t size, avi_stream_header_t *ash, WAVEFORMATEX *avi_wf_head, char **asf_extra  )
{
  avi_wf_head->wFormatTag	= READ16( pIO );
  avi_wf_head->nChannels	= READ16( pIO );
  avi_wf_head->nSamplesPerSec	= READ32( pIO );
  avi_wf_head->nAvgBytesPerSec	= READ32( pIO );
  avi_wf_head->nBlockAlign	= READ16( pIO );
  avi_wf_head->wBitsPerSample	= READ16( pIO );
  avi_wf_head->cbSize		= 0;
  *asf_extra			= NULL;

  if( avi_wf_head->wBitsPerSample == 0 )
	avi_wf_head->wBitsPerSample = 0x10;

  if( avi_wf_head->wFormatTag != 0x01 )
  {
	avi_wf_head->cbSize		= READ16( pIO );

	if( size > ( 0x12 + avi_wf_head->cbSize ))
		xmm_logging( 1, "AVI! fmt: WARNING: Extra size < chunk size - 18 ( fixing, diff = %li bytes )\n", size - ( 0x12 + avi_wf_head->cbSize ));

	size -= 2;

	if(( *asf_extra = malloc( size - 16 )) == NULL )
	{
	    xmm_logging( 1, "AVI! fmt: ERROR allocating memory for extra data ( %i bytes )\n", size - 16 );
	    return 0;
	}
  
	pIO->Read( pIO, *asf_extra, size - 16, 1 );
  }

#ifdef VERBOSE
  xmm_logging( 2, "AVI! wfx: wFormatTag = 0x%x\n", avi_wf_head->wFormatTag );
  xmm_logging( 2, "AVI! wfx: nChannels = %i\n", avi_wf_head->nChannels );
  xmm_logging( 2, "AVI! wfx: nSamplesPerSec = %li\n", avi_wf_head->nSamplesPerSec );
  xmm_logging( 2, "AVI! wfx: nAvgBytesPerSec = %li\n", avi_wf_head->nAvgBytesPerSec );
  xmm_logging( 2, "AVI! wfx: nBlockAlign = %i\n", avi_wf_head->nBlockAlign );
  xmm_logging( 2, "AVI! wfx: wBitsPerSample = %i\n", avi_wf_head->wBitsPerSample );
  xmm_logging( 2, "AVI! wfx: cbSize = %i\n", avi_wf_head->cbSize );
#endif

  return 1;
}

/*
 * Read stream data
 */
static int Read_strd( XMM_PluginIO *pIO, unsigned char *data, uint32_t size )
{
  pIO->Read( pIO, data, size, 1 );
  return 1;
}

/*
 * Read INFO chunk
 */
static int Read_INFO_chunk( XMM_PluginIO *pIO, XMM_ClipInfo *ci, uint32_t dwName, uint32_t dwSize )
{
  char	info_buffer[4096];

  pIO->Read( pIO, info_buffer, dwSize, 1 );

   switch( dwName )
   {
	case RIFF_ID_INAM:
		strncpy( ci->name, info_buffer, XMM_CIL_NAME - 1 );
		ci->name[XMM_CIL_NAME - 1] = '\0';
		break;

	case RIFF_ID_ISBJ:
		strncpy( ci->content, info_buffer, XMM_CIL_CONTENT - 1 );
		ci->content[XMM_CIL_CONTENT - 1] = '\0';
		break;

	case RIFF_ID_IART:
		strncpy( ci->author, info_buffer, XMM_CIL_AUTHOR - 1 );
		ci->author[XMM_CIL_AUTHOR - 1] = '\0';
		break;

	case RIFF_ID_ICOP:
		strncpy( ci->copyright, info_buffer, XMM_CIL_COPYRIGHT - 1 );
		ci->copyright[XMM_CIL_COPYRIGHT - 1] = '\0';
		break;

	case RIFF_ID_ISFT:
		strncpy( ci->software, info_buffer, XMM_CIL_SOFTWARE - 1 );
		ci->software[XMM_CIL_SOFTWARE - 1] = '\0';
		break;

	case RIFF_ID_ICMT:
		strncpy( ci->comment, info_buffer, XMM_CIL_COMMENT - 1 );
		ci->comment[XMM_CIL_COMMENT - 1] = '\0';
		break;

	case RIFF_ID_ICRD:
		strncpy( ci->date + 4, info_buffer, 4 );
		strncpy( ci->date + 2, info_buffer + 5, 2 );
		strncpy( ci->date, info_buffer + 8, 2 );
		ci->date[8] = '\0';
		break;

	default:
		xmm_logging( 2, "AVI! Ignoring INFO chunk 0x%x [%s] [%li bytes]\n", dwName, xmm_FOURCC_string(dwName), dwSize );
		break;
  }

  if( dwSize & 1 )	READ8( pIO );
  return 1;
}

/*
 * Read idx1 chunk
 */
static int Read_idx1( XMM_PluginIO *pIO, uint32_t size, avi_index_entry_t **_idx1_entry, uint32_t *_idx1_entries, uint32_t *_idx1_relative, struct priv_t *priv )
{
  uint32_t		minoff, i, stream;
  char			*ptr;
  avi_index_entry_t	*idx1_entry = NULL;
  uint32_t		idx1_entries = 0, idx1_relative = 0;

  idx1_relative = 0;
  idx1_entries = size >> 4;	/* 16 bytes / entry */
#ifdef VERBOSE
  xmm_logging( 1, "AVI! idx1: %li entries\n", idx1_entries );
#endif

  /* Allocate memory */
  if(( idx1_entry = malloc( idx1_entries * sizeof( avi_index_entry_t ))) == NULL )
  {
	xmm_SetError( pIO->sys.xmm, XMM_RET_ALLOC, "(AVI) Error allocating memory..." );
	return 0; 
  }

  /* Read idx1 chunk */
  pIO->Read( pIO, idx1_entry, idx1_entries * sizeof( avi_index_entry_t ), 1 );

  /**/
  size -= ( idx1_entries << 4 );
  while( size-- )	READ8( pIO );

  /*
   * Note: minoff is needed to check, whether offset is from beginn of file
   * or relative to the movi list. ( this is the way xanim handles it )
   */
  for( i = 0, minoff = 0xFFFFFFFF; i < idx1_entries; i++ )
  {
#ifdef AVI_DEBUG_IDX1_ENTRIES
	xmm_logging( 2, "AVI! idx1: %i: ID = '%s' ( %x )\n", i, xmm_FOURCC_string( idx1_entry[i].dwChunkID ), idx1_entry[i].dwChunkID );
	xmm_logging( 2, "AVI! idx1: %i: Flags = %x\n", i, idx1_entry[i].dwFlags );
	xmm_logging( 2, "AVI! idx1: %i: Offset = %i\n", i, idx1_entry[i].dwChunkOffset );
	xmm_logging( 2, "AVI! idx1: %i: Length = %i\n", i, idx1_entry[i].dwChunkLength );
#endif

	if( idx1_entry[i].dwChunkOffset < minoff )
			minoff = idx1_entry[i].dwChunkOffset;

	stream = strtol( xmm_FOURCC_string( idx1_entry[i].dwChunkID & 0xFFFF ), &ptr, 16 );
	if(( stream < 0 ) || ( stream >= MAX_STREAMS ))	continue;

	priv->aStream[stream].idx1_size++;
  }

  if( minoff < priv->data_offset )	idx1_relative = priv->data_offset - 4;

  *_idx1_entry = idx1_entry;
  *_idx1_entries = idx1_entries;
  *_idx1_relative = idx1_relative;

  return 1;
}

/*
 * Reconstruct idx1 chunk
 */
static int reconstruct_idx1( XMM_PluginIO *pIO, avi_index_entry_t **_idx1_entry, uint32_t *_idx1_entries, uint32_t *_idx1_relative, struct priv_t *priv )
{
  uint32_t		dwSize, dwName, stream;
  uint32_t		idx1_entries, idx1_relative, idx1_size, data_done = 0;
  avi_index_entry_t	*idx1_entry = NULL, *temp;
  char			*ptr;
#if 0
  uint32_t		db = xmmFOURCC( 0, 0, 'd','b' );
  uint32_t		dc = xmmFOURCC( 0, 0, 'd','c' );
  uint32_t		wb = xmmFOURCC( 0, 0, 'w','b' );
#endif

  xmm_logging( 2, "AVI! Reconstructing IDX1 chunk " );

  /* Seek to data start */
  pIO->Seek( pIO, priv->data_offset, XMM_SEEK_SET );

  idx1_entries = 0;
  idx1_relative = 0;
  idx1_size = AVI_IDX1_BLOCK;

  /* Allocate memory */
  if(( idx1_entry = malloc( AVI_IDX1_BLOCK * sizeof( avi_index_entry_t ))) == NULL )
	return xmm_SetError( pIO->sys.xmm, XMM_RET_ALLOC, "(AVI) Error allocating memory..." );

  /* Rebuild idx1 chunk */
  while( 1 )
  {
	dwName = READ32( pIO );
	dwSize = READ32( pIO );

	if( pIO->Eof( pIO ))	break;

	/**/
	if( isdigit(((char *)&dwName)[0]) && isdigit(((char *)&dwName)[1]))
	{
	    if( idx1_entries >= idx1_size )
	    {
		idx1_size += AVI_IDX1_BLOCK;
		temp = realloc( idx1_entry, idx1_size * sizeof( avi_index_entry_t ));
		if( temp == NULL )	return xmm_SetError( pIO->sys.xmm, XMM_RET_ALLOC, "(AVI) Error (re)allocating memory..." );
		idx1_entry = temp;
	    }

	    idx1_entry[idx1_entries].dwChunkID = dwName;
	    idx1_entry[idx1_entries].dwFlags = 0;
	    idx1_entry[idx1_entries].dwChunkOffset = pIO->Tell( pIO ) - 8;
	    idx1_entry[idx1_entries].dwChunkLength = dwSize;
#ifdef AVI_DEBUG_IDX1_ENTRIES
	    xmm_logging( 2, "AVI! idx1: %i: ID = '%s' ( %x )\n", idx1_entries, xmm_FOURCC_string( idx1_entry[idx1_entries].dwChunkID ), idx1_entry[idx1_entries].dwChunkID );
	    xmm_logging( 2, "AVI! idx1: %i: Flags = %x\n", idx1_entries, idx1_entry[idx1_entries].dwFlags );
	    xmm_logging( 2, "AVI! idx1: %i: Offset = %i\n", idx1_entries, idx1_entry[idx1_entries].dwChunkOffset );
	    xmm_logging( 2, "AVI! idx1: %i: Length = %i\n", idx1_entries, idx1_entry[idx1_entries].dwChunkLength );
#endif
	    idx1_entries++;

	    stream = strtol( xmm_FOURCC_string( dwName & 0xFFFF ), &ptr, 16 );
	    if(( stream < 0 ) || ( stream >= MAX_STREAMS ))	continue;

	    priv->aStream[stream].idx1_size++;
	}
	else	break;

	/* AVI is WORD aligned, NOTE: paddding byte must NOT be used */
	if( dwSize & 1 )	dwSize++;

	/* Skip data */
	SEEK( pIO, dwSize, priv->seekable );

	/* Output state */
	data_done += dwSize;
	if( data_done >= ( priv->data_size / 50 ))
	{
	    xmm_logging( 1, "." );
	    data_done -= ( priv->data_size / 50 );
	}
    }

  xmm_logging( 1, "\n" );
#ifdef VERBOSE
  xmm_logging( 1, "AVI! idx1: %li entries\n", idx1_entries );
#endif

  *_idx1_entry = idx1_entry;
  *_idx1_entries = idx1_entries;
  *_idx1_relative = idx1_relative;

  return 1;
}

/*
 * Convert M$ wFormatID to XMMP audio codec
 */
static uint32_t codec_ms2xmm( uint16_t ms_codec )
{
  int		xmm_fmt_idx;
  struct format_conv
  {
	int	xmm;
	int	ms;
  } format_table[] =
    {
	{ XMM_AUDIO_CODEC_PCM, WAVE_FORMAT_PCM },
	{ XMM_AUDIO_CODEC_IEEE, WAVE_FORMAT_IEEE_FLOAT },
	{ XMM_AUDIO_CODEC_ULAW | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_MULAW },
	{ XMM_AUDIO_CODEC_ALAW | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_ALAW },
	{ XMM_AUDIO_CODEC_MPEG | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_MPEGLAYER3 },
	{ XMM_AUDIO_CODEC_AC3 | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_AC3 },
	{ XMM_AUDIO_CODEC_GSM610MS | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_GSM610 },
	{ XMM_AUDIO_CODEC_G721 | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_G721_ADPCM },
	{ XMM_AUDIO_CODEC_G723 | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_G723_ADPCM },
	{ XMM_AUDIO_CODEC_G726 | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_G726_ADPCM },
	{ XMM_AUDIO_CODEC_WMA1 | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_WMA1 },
	{ XMM_AUDIO_CODEC_WMA2 | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_WMA2 },
	{ XMM_AUDIO_CODEC_ADPCM_MS | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_ADPCM },
	{ XMM_AUDIO_CODEC_ADPCM_IMA | XMM_AUDIO_MASK_SIGNED | 0x10, WAVE_FORMAT_DVI_ADPCM },
	{ XMM_AUDIO_CODEC_VOX_META | XMM_AUDIO_MASK_SIGNED | 0x10, 0x75 },
	{ 0, 0 }
    };

  /* Check format */
  for( xmm_fmt_idx = 0; format_table[xmm_fmt_idx].xmm; xmm_fmt_idx++ )
	if( ms_codec == format_table[xmm_fmt_idx].ms )	break;

  if( xmm_fmt_idx == 10 )	return 0;

  return format_table[xmm_fmt_idx].xmm;
}

/*
 * Realloc buffer
 */
static int buffer_realloc( char **buffer, int *oldsize, int newsize )
{
  /* Check, if rbuffer needs to be (re)allocated */
  if(( *oldsize < newsize ) && *buffer )
  {
	free( *buffer );
	*buffer = NULL;
  }

  /* Allocate buffer */
  if( *buffer == NULL )
  {
	*oldsize = newsize;
	if(( *buffer = malloc( *oldsize )) == NULL )
		return -1;
  }

  return *oldsize;
}
