/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * cdda.c
 * CD Digital Audio input
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/event.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/config.h>

#include "cdda.h"

/*
 * Definitions
 */

#define	DEFAULT_DEVICE		"/dev/cdrom"
#define	DEFAULT_CDINDEX_SERVER	"www.cdindex.org"

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Config
 */

static char	cdr_device[XMM_MAXPATHLEN] = DEFAULT_DEVICE;
static char 	cdindex_server[XMM_MAXPATHLEN] = DEFAULT_CDINDEX_SERVER;
static int	cdindex_enable = 0;
#ifdef XMM_CDDA_READ
static int	cdr_readmode = 1;
#endif

static XMM_ConfigBlock cdda_config[] =
{
    { cdr_device,	"device",		XMM_CFG_TYPE_STRING,	XMM_MAXPATHLEN, "CDROM device" },
    { &cdindex_enable,	"cdindex_enable",	XMM_CFG_TYPE_BOOL,	4,		"Enable CDindex" },
    { cdindex_server,	"cdindex_server",	XMM_CFG_TYPE_STRING,	XMM_MAXPATHLEN,	"CDindex Server" },
#ifdef XMM_CDDA_READ
    { &cdr_readmode,	"readmode",		XMM_CFG_TYPE_BOOL,	4,		"Read mode" },
#endif
    { NULL,		"",			0,			0,		"" }
};

/*
 * Prototypes
 */

/*
 * Init
 */
static XMM_PluginInput *cdda_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(CDDA) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  /* Read configuration */
  xmmCfg_BlockLoad( "config", "input-cdda", cdda_config );

  return input;
}

/*
 * Open
 */
static int cdda_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  struct cdrom_tochdr	th;
  struct cdrom_tocentry	te;
  char 			*ptr, device[256];
  XMM_Event_TOC		etoc;
  int			i;

#ifndef XMM_CDDA_DOUT
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) 'direct out' mode not compiled." );
#endif

#ifndef XMM_CDDA_READ
  if( flags & XMM_INPUT_CF_MODE_READ )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) 'read' mode not compiled." );
#endif

  /* Check filename string */
  if( !strncasecmp( filename, "cdda://", 7 ))	ptr = filename + 7;
  else	if( !strncasecmp( filename, "cdda:", 5 ))	ptr = filename + 5;
  else	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) No CDDA file pattern: '%s'", filename );

  if( *ptr == '\0' )
  {
	strcpy( device, cdr_device );
	priv->cTrack = -1;
  }
  else if( isdigit( *ptr ))
  {
	strcpy( device, cdr_device );
	priv->cTrack = atoi( ptr );
  }
  else
  {
	memset( device, 0, 256 );
	memccpy( device, ptr, ':', 255 );
	if( device[ strlen( device ) - 1 ] == ':' )	device[ strlen( device ) - 1 ] = '\0';
	ptr += strlen( device );
	if(( ptr[0] == ':' ) && ( ptr[1] != '\0' ))	priv->cTrack = atoi( ++ptr );
	else	priv->cTrack = -1;
  }

  /* Open cdrom device */
  if(( priv->fd = open( device, O_RDONLY | O_NONBLOCK )) == -1 )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) Error opening device %s: %s", device, strerror( errno ));

  /* Read table of content */
  if( ioctl( priv->fd, CDROMREADTOCHDR, &th ))
  {
	close( priv->fd );
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) ioctl ( CDROMREADTOCHDR ) failed" );
  }

  priv->FirstTrack = th.cdth_trk0;
  priv->LastTrack = th.cdth_trk1;

  if( priv->cTrack == -1 )	/* Get Track number */
  {
	etoc.type = XMM_EVENT_TOC;
	etoc.entries = priv->LastTrack - priv->FirstTrack + 1;

	if(( etoc.entry = malloc( etoc.entries * sizeof(char *))) == NULL )
		return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(CDDA) Unable to allocate data for entry pointers" );

	if(( ptr = malloc( 256 * etoc.entries * sizeof(char))) == NULL )
		return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(CDDA) Unable to allocate data for XMM_Event_TOC struct" );

	for( i = priv->FirstTrack; i <= priv->LastTrack; i++ )
	{
		etoc.entry[i - priv->FirstTrack] = &ptr[256 * ( i - priv->FirstTrack )];
		sprintf( etoc.entry[i - priv->FirstTrack], "Track %.2i", i - priv->FirstTrack + 1 );
	}

	i = xmm_PushEvent( input->sys.xmm, (XMM_Event *) &etoc, 0 );
	free( ptr );
	free( etoc.entry );

	if( i >= 0 )	priv->cTrack = etoc.selected + priv->FirstTrack;
	else	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) Wrong CDDA file pattern '%s' or no Track specified", filename );
  }

  xmm_logging( 3, "CDDA! Using device %s Track %i\n", device, priv->cTrack );

  for( i = priv->FirstTrack; i <= priv->LastTrack; i++ )
  {
	te.cdte_track = i;
	te.cdte_format = CDROM_MSF;
	if( ioctl( priv->fd, CDROMREADTOCENTRY, &te ))
	{
	    close( priv->fd );
	    return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) ioctl ( CDROMREADTOCENTRY ) failed" );
	}
	priv->Track[ i - priv->FirstTrack ].type = ( te.cdte_ctrl & CDROM_DATA_TRACK );
	priv->Track[ i - priv->FirstTrack ].minute = te.cdte_addr.msf.minute;
	priv->Track[ i - priv->FirstTrack ].second = te.cdte_addr.msf.second;
	priv->Track[ i - priv->FirstTrack ].frame = te.cdte_addr.msf.frame;

#ifdef VERBOSE
	xmm_logging( 4, "CDDA! TE(%2i): track = %i\n", i, te.cdte_track );
	xmm_logging( 4, "CDDA! TE(%2i): format = %i\n", i, te.cdte_format );
	xmm_logging( 4, "CDDA! TE(%2i): ctrl = %i\n", i, te.cdte_ctrl );
	xmm_logging( 4, "CDDA! TE(%2i): addr = %i ( %i:%i:%i )\n", i, te.cdte_addr.lba, te.cdte_addr.msf.minute, te.cdte_addr.msf.second, te.cdte_addr.msf.frame );
	xmm_logging( 4, "CDDA! TE(%2i): adr = %i\n", i, te.cdte_adr );
	xmm_logging( 4, "CDDA! TE(%2i): datamode = %i\n", i, te.cdte_datamode );
#endif
  }

  if(( priv->cTrack < priv->FirstTrack ) || ( priv->cTrack > priv->LastTrack ) || priv->Track[ priv->cTrack - priv->FirstTrack ].type )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) Invalid track number ( %i )", priv->cTrack );

  te.cdte_track = CDROM_LEADOUT;
  te.cdte_format = CDROM_MSF;
  if( ioctl( priv->fd, CDROMREADTOCENTRY, &te ))
  {
	close( priv->fd );
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) ioctl ( CDROMREADTOCENTRY ) failed" );
  }
  priv->Track[ i - priv->FirstTrack ].type = ( te.cdte_ctrl & CDROM_DATA_TRACK );
  priv->Track[ i - priv->FirstTrack ].minute = te.cdte_addr.msf.minute;
  priv->Track[ i - priv->FirstTrack ].second = te.cdte_addr.msf.second;
  priv->Track[ i - priv->FirstTrack ].frame = te.cdte_addr.msf.frame;

#ifdef VERBOSE
  xmm_logging( 4, "CDDA! TE(LO): track = %i\n", i, te.cdte_track );
  xmm_logging( 4, "CDDA! TE(LO): format = %i\n", i, te.cdte_format );
  xmm_logging( 4, "CDDA! TE(LO): ctrl = %i\n", i, te.cdte_ctrl );
  xmm_logging( 4, "CDDA! TE(LO): addr = %i ( %i:%i:%i )\n", i, te.cdte_addr.lba, te.cdte_addr.msf.minute, te.cdte_addr.msf.second, te.cdte_addr.msf.frame );
  xmm_logging( 4, "CDDA! TE(LO): adr = %i\n", i, te.cdte_adr );
  xmm_logging( 4, "CDDA! TE(LO): datamode = %i\n", i, te.cdte_datamode );
#endif

#ifdef VERBOSE
  for( i = priv->FirstTrack; i <= priv->LastTrack; i++ )
  {
	xmm_logging( 3, "CDDA! Track %.2i Start = %.2i:%.2i.%.2i Length = %4.2f seconds\n",
					i - priv->FirstTrack + 1,
					priv->Track[ i - priv->FirstTrack ].minute,
					priv->Track[ i - priv->FirstTrack ].second,
					priv->Track[ i - priv->FirstTrack ].frame,
	(float)( LBA( priv->Track[i - priv->FirstTrack + 1] ) - LBA( priv->Track[i - priv->FirstTrack] )) / 75 );
  }
#endif

#ifdef COMPILE_CDINDEX
  /* Get cdindex information */
  if( cdindex_enable )
  {
	priv->hasData = cdindex_get_info( input->sys.xmm, cdindex_server, priv->FirstTrack, priv->LastTrack, priv->Track, &priv->Data );
	if( priv->hasData )
	{
	    xmm_logging( 3, "CDDA! Album: %s ( %s )\n", priv->Data.title, ( priv->Data.artistType == CDDA_ARTIST_SINGLE ) ? "Single Artist" : "Multiple Artist" );
	    for( i = priv->FirstTrack; i <= priv->LastTrack; i++ )	xmm_logging( 3, "CDDA! Track %i: %s - %s\n", i, priv->Data.track[i].artist, priv->Data.track[i].title );
	}
  }

  /* Create title string */
//  if( priv->hasData == 0 )	sprintf( priv->display, "CD Audio Track %.2i", priv->cTrack );
//  else	cdda_create_display( priv->display, &priv->Data, priv->cTrack );
#endif

  /* Fill audio format struct */
  priv->ai.fmt.format = XMM_AUDIO_FMT_S16LE;
  priv->ai.fmt.samprate = 44100;
  priv->ai.fmt.channels = 2;
  priv->ai.fmt.bitrate = 44100 * 2 * 2 * 8;
  priv->ai.fmt.blockSize = 4;
  priv->ai.fmt.extraSize = 0;
  priv->ai.fmt.desc[0] = '\0';
  priv->ai.tSize = ( LBA( priv->Track[priv->cTrack - priv->FirstTrack + 1] ) -
	    LBA( priv->Track[priv->cTrack - priv->FirstTrack] )) * CD_FRAMESIZE_RAW;
  priv->ai.tSamples = priv->ai.tSize / priv->ai.fmt.blockSize;
  priv->ai.offset = 0;

  /* Clip Info */
  priv->ci.name[0] = '\0';
  priv->ci.author[0] = '\0';
  priv->ci.album[0] = '\0';
  priv->ci.content[0] = '\0';
  priv->ci.copyright[0] = '\0';
  priv->ci.software[0] = '\0';
  priv->ci.date[0] = '\0';
  priv->ci.comment[0] = '\0';
  priv->ci.size = priv->ai.tSize;
  priv->ci.playtime = (double) (
	    LBA( priv->Track[priv->cTrack - priv->FirstTrack + 1] ) -
	    LBA( priv->Track[priv->cTrack - priv->FirstTrack] )) / 75;

#ifdef XMM_CDDA_DOUT
  /* Now we are ready ( for direct output ) */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
  {
	cdda_dout_init( input );
	return XMM_RET_OK;
  }
#endif

#ifdef XMM_CDDA_READ
  if(( flags & XMM_INPUT_CF_MODE_READ ) && !cdr_readmode )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(CDDA) CDDA 'read' mode not enabled" );

  /* Read mode desired, intitialize it */
  priv->frames_spos = LBA( priv->Track[ priv->cTrack - priv->FirstTrack ] );
  priv->frames_epos = LBA( priv->Track[ priv->cTrack - priv->FirstTrack + 1 ] );
  priv->frames_cpos = priv->frames_spos;

  cdda_read_init( input );
#endif

  return XMM_RET_OK;
}

/*
 * Close
 */
static int cdda_Close( XMM_PluginInput *input )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  XMM_ConfigBlock	*cfg;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
	case XMM_CTLQUERY_AFORMAT:
	case XMM_CTLQUERY_YFLIP:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_TRUE;

	case XMM_CTLGET_GFORMAT:
	case XMM_CTLGET_AFORMAT:
	case XMM_CTLGET_VOLUME:
	case XMM_CTLSET_GFORMAT:
	case XMM_CTLSET_SCALE:
	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	/* Get capabilities */
	case XMM_CTLGET_CAPS:
		*((uint32_t *) data ) = 0 |
#ifdef XMM_CDDA_DOUT
					XMM_INPUT_CF_MODE_DOUT |
					XMM_INPUT_CF_VOLUME |
#endif
#ifdef XMM_CDDA_READ
					XMM_INPUT_CF_MODE_READ |
#endif
					XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	/* Get config */
	case XMM_CTLGET_CONFIG:
		if( data == NULL )	return XMM_RET_INVALID_ARG;

		if( input->sys.priv == NULL )
		{
			/* Read configuration */
			xmmCfg_BlockLoad( "config", "input-cdda", cdda_config );
		}

		cfg = xmmCfg_Blockdup( cdda_config );
		if( cfg == NULL )	return XMM_RET_ERROR;

		*((XMM_ConfigBlock **)data) = cfg;
		return XMM_CTLRET_ARG;

	/* Set config */
	case XMM_CTLSET_CONFIG:
		if( data == NULL )	return XMM_RET_INVALID_ARG;

		xmmCfg_BlockSave( "config", "input-cdda", (XMM_ConfigBlock *)data );
		return XMM_CTLRET_TRUE;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(CDDA) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(CDDA) cmd ( 0x%x )" );
}

static int cdda_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_AudioStreams( XMM_PluginInput *input )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * 
 */

/*
 * CDDA tracks do not contain video data
 */
static int cdda_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int cdda_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if CDDA
 */
static int cdda_Check( void *xmm, char *filename )
{
  return !( strncasecmp( filename, "cdda://", 7 ) && strncasecmp( filename, "cdda:", 5 ));
}

/*
 * Get file info
 */
static int cdda_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = { { NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"CDDA",
				"Input: CD Digital Audio",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				cdda_Init, cdda_Open, cdda_Close,
				cdda_Control, cdda_Seek, cdda_Info,
				cdda_AudioStreams, cdda_AudioInfo,
				cdda_AudioRead, cdda_AudioPTS, cdda_AudioSeek,
				cdda_VideoStreams, cdda_VideoInfo,
				cdda_VideoRead, cdda_VideoPTS, cdda_VideoSeek,
				cdda_Check, cdda_FileInfo };
