/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * cdda_dout.c
 * CD Digital Audio Plugin ( direct out mode )
 */

#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>

#include <errno.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>

#include "cdda.h"

#ifdef XMM_CDDA_DOUT

/*
 * Prototypes
 */

static int cdda_GetVolume( XMM_PluginInput *input, uint32_t *volume );
static int cdda_SetVolume( XMM_PluginInput *input, uint32_t volume );

static int cdda_dout_Close( XMM_PluginInput *input );
static int cdda_dout_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data );
static int cdda_dout_Seek( XMM_PluginInput *input, int vstream, int astream, double seek );
static int cdda_dout_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval );
static int cdda_dout_AudioStreams( XMM_PluginInput *input );
static int cdda_dout_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai );
static int cdda_dout_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags );
static int cdda_dout_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts );
static int cdda_dout_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts );

/*
 * Initialize read mode
 */
int cdda_dout_init( XMM_PluginInput *input )
{
  input->Close = cdda_dout_Close;
  input->Control = cdda_dout_Control;
  input->Seek = cdda_dout_Seek;
  input->Info = cdda_dout_Info;
  input->AudioStreams = cdda_dout_AudioStreams;
  input->AudioInfo = cdda_dout_AudioInfo;
  input->AudioRead = cdda_dout_AudioRead;
  input->AudioPTS = cdda_dout_AudioPTS;
  input->AudioSeek = cdda_dout_AudioSeek;

  return XMM_RET_OK;
}

/*
 * Close
 */
static int cdda_dout_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  /* Stop playback */
  if( ioctl( priv->fd, CDROMSTOP ))

  /* Close file descriptor */
  close( priv->fd );

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int cdda_dout_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	/* Query sound format */
	case XMM_CTLQUERY_AFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_YFLIP:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	/* Get sound format */
	case XMM_CTLGET_AFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_VOLUME:
		cdda_GetVolume( input, (uint32_t *) data );
		return XMM_CTLRET_ARG;		/* Result in arg */

	/* Get capabilities */
	case XMM_CTLGET_CAPS:
		*((uint32_t *) data ) =	XMM_INPUT_CF_MODE_DOUT |
#ifdef XMM_CDDA_READ
					XMM_INPUT_CF_MODE_READ |
#endif
					XMM_INPUT_CF_VOLUME |
					XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		cdda_SetVolume( input, param );
		return XMM_CTLRET_TRUE;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
		priv->paused = 0;
		priv->stopped = 0;
		input->Seek( input, 0, 0, 0.0 );
		return XMM_CTLRET_TRUE;

	case XMM_CTLINP_STOP:
		priv->paused = 0;
		priv->stopped = 1;
		if( ioctl( priv->fd, CDROMSTOP ))
			return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) ioctl ( CDROMSTOP ) failed\n");

		return XMM_CTLRET_TRUE;

	case XMM_CTLINP_PAUSE:
		{
		    int cmd;

		    if( priv->paused )	cmd = CDROMRESUME;
		    else	cmd = CDROMPAUSE;

		    if( ioctl( priv->fd, cmd ))
			return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) ioctl ( CDROMPAUSE ) failed\n");

		    priv->paused = !priv->paused;

		    return XMM_CTLRET_TRUE;
		}

	case XMM_CTLINP_STATUS:
		{
		    struct cdrom_subchnl subchan;

		    subchan.cdsc_format = CDROM_MSF;
		    if( ioctl( priv->fd, CDROMSUBCHNL, &subchan ))
			return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(CDDA) ioctl ( CDROMSUBCHNL ) failed\n");

		    switch( subchan.cdsc_audiostatus )
		    {
			case CDROM_AUDIO_PLAY:
				*((uint32_t *) data ) = XMM_PBS_PLAYING;
				return XMM_CTLRET_ARG;

			case CDROM_AUDIO_PAUSED:
			case CDROM_AUDIO_COMPLETED:
				if( priv->paused )
					*((uint32_t *) data ) = XMM_PBS_PAUSED;
				else	*((uint32_t *) data ) = XMM_PBS_STOPPED;
				priv->stopped = 1;
				return XMM_CTLRET_ARG;

			default:
				if( priv->stopped )
					*((uint32_t *) data ) = XMM_PBS_STOPPED;
				else	*((uint32_t *) data ) = XMM_PBS_ERROR;
				return XMM_CTLRET_ARG;
		    }
		}

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(CDDA) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(CDDA) cmd ( 0x%x )" );
}

/*
 * Seek to position
 */
static int cdda_dout_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  struct priv_t	*priv = input->sys.priv;
  double	total;

  total = ( LBA( priv->Track[priv->cTrack - priv->FirstTrack + 1] ) -
	    LBA( priv->Track[priv->cTrack - priv->FirstTrack] )) / 75 * 1000;

  return input->AudioSeek( input, astream, (uint32_t)( seek * total ));
}

/*
 * Get CDDA Track information
 */
static int cdda_dout_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  struct priv_t		*priv = input->sys.priv;
  struct cdrom_subchnl	subchan;
  int			frame;

  if( ci )	memcpy( ci, &priv->ci, sizeof( XMM_ClipInfo ));

  /* Get current playtime */
  subchan.cdsc_format = CDROM_MSF;
  if( ioctl( priv->fd, CDROMSUBCHNL, &subchan ))
  {
	xmm_logging( 1, "CDDA! ioctl ( CDROMSUBCHNL ) failed\n");
	return (double)XMM_RET_OK;
  }

  frame = LBA( subchan.cdsc_absaddr.msf ) - LBA( priv->Track[priv->cTrack - priv->FirstTrack ] );

  if( seekval )
  {
	*seekval = (double)frame / ( 
	    LBA( priv->Track[priv->cTrack - priv->FirstTrack + 1] ) -
	    LBA( priv->Track[priv->cTrack - priv->FirstTrack] ));

	if( *seekval < 0 )	*seekval = 0.0;
  }

  return XMM_RET_OK;
}

/*
 * Get audio stream number
 */
static int cdda_dout_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int cdda_dout_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int cdda_dout_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Audio timestamp.
 */
static int cdda_dout_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  struct priv_t		*priv = input->sys.priv;
  struct cdrom_subchnl	subchan;
  int			frame;

  if( pts )
  {
	/* Get current playtime */
	subchan.cdsc_format = CDROM_MSF;
	if( ioctl( priv->fd, CDROMSUBCHNL, &subchan ))
	{
	    xmm_logging( 1, "CDDA! ioctl ( CDROMSUBCHNL ) failed\n");
	    return XMM_RET_ERROR;
	}

	frame = LBA( subchan.cdsc_absaddr.msf ) - LBA( priv->Track[priv->cTrack - priv->FirstTrack ] );

	*pts = frame * 1000 / 75;
  }

  return XMM_RET_OK;
}

/*
 * Seek to position in audio stream
 */
static int cdda_dout_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts )
{
  struct priv_t		*priv = input->sys.priv;
  struct cdrom_msf	msf;
  int			frame;

  frame = (int) pts * 75 / 1000 + LBA( priv->Track[ priv->cTrack - priv->FirstTrack ] );

  msf.cdmsf_min0 = frame / 60 / 75;
  msf.cdmsf_sec0 = ( frame - msf.cdmsf_min0 * 60 * 75 ) / 75;
  msf.cdmsf_frame0 = frame % 75;
  msf.cdmsf_min1 = priv->Track[ priv->cTrack - priv->FirstTrack + 1 ].minute;
  msf.cdmsf_sec1 = priv->Track[ priv->cTrack - priv->FirstTrack + 1 ].second;
  msf.cdmsf_frame1 = priv->Track[ priv->cTrack - priv->FirstTrack + 1 ].frame;

  if( ioctl( priv->fd, CDROMPLAYMSF, &msf ))
  {
	xmm_logging( 1, "CDDA! ioctl ( CDROMPLAYMSF ) failed [%i.%i:%i,%i.%i:%i]: %s\n", msf.cdmsf_min0, msf.cdmsf_sec0, msf.cdmsf_frame0, msf.cdmsf_min1, msf.cdmsf_sec1, msf.cdmsf_frame1, strerror(errno));
	return XMM_RET_ERROR;
  }

  return XMM_RET_OK;
}

/*
 * Internal code
 */

/*
 * Set Volume
 */

static int cdda_SetVolume( XMM_PluginInput *input, uint32_t volume )
{
  struct priv_t	*priv = input->sys.priv;
  struct cdrom_volctrl vol;

  vol.channel0 = vol.channel2 = (( volume & 0x0000FFFF ) * 255 ) / 100;
  vol.channel1 = vol.channel3 = ((( volume & 0xFFFF0000 ) >> 16) * 255 ) / 100;
  if( ioctl( priv->fd, CDROMVOLCTRL, &vol ))
  {
	xmm_logging( 1, "CDDA! ioctl ( CDROMVOLCTRL ) failed\n");
	return 0;
  }
  return 1;
}

/*
 * Get Volume
 */

static int cdda_GetVolume( XMM_PluginInput *input, uint32_t *volume )
{
  struct priv_t	*priv = input->sys.priv;
  struct cdrom_volctrl vol;

  if( ioctl( priv->fd, CDROMVOLREAD, &vol ))
  {
	xmm_logging( 1, "CDDA! ioctl ( CDROMVOLREAD ) failed\n");
	return 0;
  }

  *volume = ( 100 * vol.channel0 / 255 ) | (( 100 * vol.channel1 / 255 ) << 16 );

  return 1;
}

#endif
