/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * cdda_read.c
 * CD Digital Audio Plugin ( read mode )
 */

#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>

#include "cdda.h"

#ifdef XMM_CDDA_READ

/*
 * Definitions
 */

/*
 * Prototypes
 */

static int cdda_read_Close( XMM_PluginInput *input );
static int cdda_read_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data );
static int cdda_read_Seek( XMM_PluginInput *input, int vstream, int astream, double seek );
static int cdda_read_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval );
static int cdda_read_AudioStreams( XMM_PluginInput *input );
static int cdda_read_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai );
static int cdda_read_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags );
static int cdda_read_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts );
static int cdda_read_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts );

/*
 * Initialize read mode
 */
int cdda_read_init( XMM_PluginInput *input )
{
  input->Close = cdda_read_Close;
  input->Control = cdda_read_Control;
  input->Seek = cdda_read_Seek;
  input->Info = cdda_read_Info;
  input->AudioStreams = cdda_read_AudioStreams;
  input->AudioInfo = cdda_read_AudioInfo;
  input->AudioRead = cdda_read_AudioRead;
  input->AudioPTS = cdda_read_AudioPTS;
  input->AudioSeek = cdda_read_AudioSeek;

  return XMM_RET_OK;
}

/*
 * Close
 */
static int cdda_read_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  /* Close file descriptor */
  close( priv->fd );

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int cdda_read_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_AudioFormat	*format;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_AFORMAT:			/* Query sound format */
		format = (XMM_AudioFormat *)data;
		if(( format->format == priv->ai.fmt.format ) &&
		    ( format->samprate == priv->ai.fmt.samprate ) &&
		    ( format->channels == priv->ai.fmt.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_AFORMAT:			/* Get sound format */
		memcpy( data, &priv->ai.fmt, sizeof( XMM_AudioFormat ));
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->ai.fmt;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_CAPS:				/* Get capabilities */
		*((uint32_t *) data ) =	XMM_INPUT_CF_MODE_DOUT |
					XMM_INPUT_CF_MODE_READ |
					XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(CDDA) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(CDDA) cmd ( 0x%x )" );
}

/*
 * Seek to position
 */
static int cdda_read_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  struct priv_t	*priv = input->sys.priv;
  double	total;

  total = ( priv->frames_epos - priv->frames_spos ) / 75 * 1000;

  return input->AudioSeek( input, astream, (uint32_t)( seek * total ));
}

/*
 * Get Information
 */
static int cdda_read_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )	memcpy( ci, &priv->ci, sizeof( XMM_ClipInfo ));

  return XMM_RET_OK;
}

/*
 * Get audio stream number
 */
static int cdda_read_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int cdda_read_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int cdda_read_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  struct priv_t			*priv = input->sys.priv;
  struct cdrom_read_audio	ra;
  int				frames;

  frames = size / CD_FRAMESIZE_RAW;
  if( frames > ( priv->frames_epos - priv->frames_cpos ))
  {
	frames = priv->frames_epos - priv->frames_cpos;
	if( frames == 0 )	return XMM_RET_EOS;
	xmm_logging( 2, "CDDA! Only %i frame(s) left\n", frames );
  }

  ra.addr.lba = priv->frames_cpos;
  ra.addr_format = CDROM_LBA;
  ra.nframes = frames;
  ra.buf = (void *)buffer;

  if( ioctl( priv->fd, CDROMREADAUDIO, &ra ))
  {
	xmm_logging( 1, "CDDA! ioctl ( CDROMREADAUDIO ) failed\n");
	return XMM_RET_ERROR;
  }

  priv->frames_cpos += frames;

  return frames * CD_FRAMESIZE_RAW;
}

/*
 * Audio timestamp.
 */
static int cdda_read_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  struct priv_t		*priv = input->sys.priv;

  if( pts )	*pts = ( priv->frames_cpos - priv->frames_spos ) * 1000 / 75;

  return XMM_RET_OK;
}

/*
 * Seek to position in audio stream
 */
static int cdda_read_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts )
{
  struct priv_t	*priv = input->sys.priv;

  /* Calculate frame number */
  priv->frames_cpos = priv->frames_spos + pts * 75 / 1000;

  /* Only to be sure, we are in current track */
  if( priv->frames_cpos >= priv->frames_epos )
	priv->frames_cpos = priv->frames_epos;

  return XMM_RET_OK;
}

#endif
