/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * mikmod.c
 * MikMod input
 */

#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/config.h>

#include <mikmod.h>

/*
 * Types
 */

struct priv_t
{
    XMM_AudioInfo		ai;
    XMM_ClipInfo		ci;

    char 			*filename, path[XMM_MAXPATHLEN];
    MODULE			*module;
    uint32_t			position;
    int				byterate;

    struct mikmod_reader
    {
	MREADER			sys;
	XMM_PluginIO		*pIO;
    } mreader;
};

/*
 * Global data
 */

extern XMM_PluginInput		plugin_info;
extern MIKMODAPI		MDRIVER drv_xmm;

static struct priv_t 		*mikmod_priv = NULL;

/*
 * Config
 */

static int	cfg_rate = 44100, cfg_size = 16, cfg_chan = 2;
static int	cfg_ipolate = 1, cfg_surround = 0;

static XMM_ConfigBlock mikmod_config[] =
{
    { &cfg_rate,	"samplerate",	XMM_CFG_TYPE_INT,	4, "Samplerate (44100/22050/11025)" },
    { &cfg_size,	"samplesize",	XMM_CFG_TYPE_INT,	4, "Samplesize (8/16)" },
    { &cfg_chan,	"channels",	XMM_CFG_TYPE_INT,	4, "Channels (1/2)" },
    { &cfg_ipolate,	"interpolate",	XMM_CFG_TYPE_BOOL,	4, "Interpolate" },
    { &cfg_surround,	"surround",	XMM_CFG_TYPE_BOOL,	4, "Surround" },
    { NULL,		"",		0,			0, "" }
};

/*
 * Prototypes
 */

static int wrap_Seek( struct MREADER *_mreader, long offset, int whence );
static long wrap_Tell( struct MREADER *_mreader );
static int wrap_Read( struct MREADER *_mreader, void *ptr, size_t size );
static int wrap_Get( struct MREADER *_mreader );
static int wrap_Eof( struct MREADER *_mreader );

/*
 * Initialize
 */
static XMM_PluginInput *mikmod_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  /* Check for multiple use. This plugin ( and MikMod ) may only be used once */
  if( mikmod_priv )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, "(MikMod) MikMod already in use." );
	return NULL;
  }

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(MikMod) Unable to duplicate plugin_info" );
	return NULL;
  }

  mikmod_priv = priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  /* Read configuration */
  xmmCfg_BlockLoad( "config", "input-mikmod", mikmod_config );

  return input;
}

/*
 * Open
 */
static int mikmod_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  char			*buffer, *ptr, cmd[100], *tmpfile;

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(MikMod) Only 'read' mode supported." );

  /* Initialize MikMod */
  MikMod_RegisterDriver( &drv_xmm );
  MikMod_RegisterAllLoaders();

  md_mixfreq = cfg_rate;

  md_mode = DMODE_SOFT_MUSIC | DMODE_SOFT_SNDFX;
  if( cfg_chan == 2 )	md_mode |= DMODE_STEREO;
  if( cfg_size == 16 )	md_mode |= DMODE_16BITS;
  if( cfg_surround )	md_mode |= DMODE_SURROUND;
  if( cfg_ipolate )	md_mode |= DMODE_INTERP;

  if( MikMod_Init( "" ))
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(MikMod) Cannot initialize mikmod: %s", MikMod_strerror( MikMod_errno ));

  /* Check filename */
  ptr = strrchr( filename, '.' );
  if( ptr == NULL )	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(MikMod) Invalid file type. Should never happen." );

  /* Get uncompressed tracker module format file */
  tmpfile = xmm_gethome_filename( "tmp/mikmod_zip" );

  if( !strcasecmp( ptr, ".gz" ))
  {
	sprintf( cmd, "gzip -dc %s >>%s", filename, tmpfile );
	system( cmd );
	strcpy( priv->path, tmpfile );
  }
  else if( !strcasecmp( ptr, ".bz2" ))
  {
	sprintf( cmd, "bzip2 -dc %s >>%s", filename, tmpfile );
	system( cmd );
	strcpy( priv->path, tmpfile );
  }
  else	strcpy( priv->path, filename );

  free( tmpfile );

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* Open file */
  priv->mreader.pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->mreader.pIO == NULL )	return XMM_RET_ERROR;

  priv->mreader.sys.Seek = wrap_Seek;
  priv->mreader.sys.Tell = wrap_Tell;
  priv->mreader.sys.Read = wrap_Read;
  priv->mreader.sys.Get = wrap_Get;
  priv->mreader.sys.Eof = wrap_Eof;

  /* Load Player and start it */
  priv->module = Player_LoadGeneric( &priv->mreader.sys, 64, 0 );
  if( priv->module == NULL )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(MikMod) Error loading file: %s\nFilename: %s", MikMod_strerror( MikMod_errno ), priv->filename );

  Player_Start( priv->module );

  /* Fill audio format struct */
  priv->ai.fmt.format = XMM_AUDIO_FMT_S16LE;
  priv->ai.fmt.samprate = md_mixfreq;
  priv->ai.fmt.channels = 2;
  priv->ai.fmt.bitrate = md_mixfreq * 4;
  priv->ai.fmt.blockSize = 4;
  priv->ai.fmt.extraSize = 0;
  priv->ai.fmt.desc[0] = '\0';
  priv->ai.tSamples = 0;
  priv->ai.tSize = priv->mreader.pIO->Size( priv->mreader.pIO );
  priv->ai.offset = 0;

  /* Byte rate */
  priv->byterate = priv->ai.fmt.samprate * priv->ai.fmt.blockSize;

  /* Clip Info */
  priv->ci.name[0] = '\0';
  priv->ci.author[0] = '\0';
  priv->ci.album[0] = '\0';
  strncpy( priv->ci.content, Player_LoadTitle( priv->path ), XMM_CIL_CONTENT - 1 );
  priv->ci.content[XMM_CIL_CONTENT - 1] = '\0';
  priv->ci.copyright[0] = '\0';
  priv->ci.software[0] = '\0';
  priv->ci.date[0] = '\0';
  priv->ci.comment[0] = '\0';
  priv->ci.size = priv->mreader.pIO->Size( priv->mreader.pIO );
  priv->ci.playtime = (double)0.0;

  return XMM_RET_OK;
}

/*
 * Close
 */
static int mikmod_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  Player_Stop();
  Player_Free( priv->module );
  MikMod_Exit();

  if( priv->filename )	free( priv->filename );
  priv->mreader.pIO->Close( priv->mreader.pIO );

  mikmod_priv = NULL;

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int mikmod_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_AudioFormat	*format;
  XMM_ConfigBlock	*cfg;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_AFORMAT:			/* Query sound format */
		format = (XMM_AudioFormat *)data;
		if(( format->format == priv->ai.fmt.format ) &&
		    ( format->samprate == priv->ai.fmt.samprate ) &&
		    ( format->channels == priv->ai.fmt.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_TRUE;

	case XMM_CTLGET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_AFORMAT:			/* Get sound format */
		memcpy( data, &priv->ai.fmt, sizeof( XMM_AudioFormat ));
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->ai.fmt;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_CAPS:				/* Get capabilities */
		*((uint32_t *) data ) = XMM_INPUT_CF_MODE_READ | XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	/* Get config */
	case XMM_CTLGET_CONFIG:
		if( data == NULL )	return XMM_RET_INVALID_ARG;

		if( input->sys.priv == NULL )
		{
			/* Read configuration */
			xmmCfg_BlockLoad( "config", "input-mikmod", mikmod_config );
		}

		cfg = xmmCfg_Blockdup( mikmod_config );
		if( cfg == NULL )	return XMM_RET_ERROR;

		*((XMM_ConfigBlock **)data) = cfg;
		return XMM_CTLRET_ARG;

	/* Set config */
	case XMM_CTLSET_CONFIG:
		if( data == NULL )	return XMM_RET_INVALID_ARG;

		xmmCfg_BlockSave( "config", "input-mikmod", (XMM_ConfigBlock *)data );
		return XMM_CTLRET_TRUE;

	case XMM_CTLSET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(MikMod) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(MikMod) cmd ( 0x%x )" );
}

/*
 * Seek to position
 */
static int mikmod_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Get Information
 */
static int mikmod_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )	memcpy( ci, &priv->ci, sizeof( XMM_ClipInfo ));
  if( seekval )	*seekval = (double)priv->module->sngpos / priv->module->numpos;

  return XMM_RET_OK;
}

/*
 * Get audio stream number
 */
static int mikmod_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int mikmod_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int mikmod_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  struct priv_t	*priv = input->sys.priv;
  uint32_t	read = 0;

  read = VC_WriteBytes( (SBYTE *) buffer, size );
  priv->position += read;

  return read;
}

/*
 * Audio timestamp.
 */
static int mikmod_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  struct priv_t	*priv = input->sys.priv;

  if( pts )	*pts = (uint32_t)((double)priv->position / priv->byterate * 1000 );

  return XMM_RET_OK;
}

/*
 * Seek to position in audio stream
 */
static int mikmod_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * MikMod files do not contain video data
 */
static int mikmod_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int mikmod_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mikmod_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mikmod_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mikmod_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if MikMod file
 */
static int mikmod_Check( void *xmm, char *filename )
{
  char *ptr;
  
  ptr = strrchr( filename, '.' );
  if( ptr == NULL )	return -1;

  if( !strcasecmp( ptr, ".gz" ) || !strcasecmp( ptr, ".bz2" ))
  {
	for( ptr--; ( *ptr != '.' ) && ( ptr != filename ); ptr-- );
	if( ptr == filename )	return XMM_RET_ERROR;
  }

  if( !strncasecmp( ptr, ".669", 4 ))	return 1;	/* 669 and Extended-669 (by Tran/Renaissance) */
  if( !strncasecmp( ptr, ".amf", 4 ))	return 1;	/* DMP Advanced Module Format (by Otto Chrons) */
  if( !strncasecmp( ptr, ".dsm", 4 ))	return 1;	/* DSIK internal module format */
  if( !strncasecmp( ptr, ".far", 4 ))	return 1;	/* Farandole Composer (by Daniel Potter) */
  if( !strncasecmp( ptr, ".gdm", 4 ))	return 1;	/* General DigiMusic (by Edward Schlunder) */
  if( !strncasecmp( ptr, ".it", 3 ))	return 1;	/* Impulse Tracker (by Jeffrey Lim) */
  if( !strncasecmp( ptr, ".imf", 4 ))	return 1;	/* Imago Orpheus (by Lutz Roeder) */
  if( !strncasecmp( ptr, ".med", 4 ))	return 1;	/* Amiga MED modules (by Teijo Kinnunen) */
  if( !strncasecmp( ptr, ".m15", 4 ))	return 1;	/* Soundtracker 15-instrument */
  if( !strncasecmp( ptr, ".mod", 4 ))	return 1;	/* Standard 31-instrument Module loader */
  if( !strncasecmp( ptr, ".mtm", 4 ))	return 1;	/* Multi-Tracker Module (by Renaissance) */
  if( !strncasecmp( ptr, ".okt", 4 ))	return 1;	/* Amiga Oktalyzer */
  if( !strncasecmp( ptr, ".stm", 4 ))	return 1;	/* ScreamTracker 2 (by Future Crew) */
  if( !strncasecmp( ptr, ".stx", 4 ))	return 1;	/* STMIK 0.2 (by Future Crew) */
  if( !strncasecmp( ptr, ".s3m", 4 ))	return 1;	/* ScreamTracker 3 (by Future Crew) */
  if( !strncasecmp( ptr, ".ult", 4 ))	return 1;	/* UltraTracker (by MAS) */
  if( !strncasecmp( ptr, ".uni", 4 ))	return 1;	/* MikMod and APlayer internal module format */
  if( !strncasecmp( ptr, ".xm", 3 )) 	return 1;	/* FastTracker 2 (by Triton) */

  /* I have some mod's with this name pattern */
  if( !strncasecmp( filename, "mod.", 4 ))	return 1;
  
  return 0;
}

/*
 * Get file info
 */
static int mikmod_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"MikMod",
				"Input: MikMod",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				mikmod_Init, mikmod_Open, mikmod_Close,
				mikmod_Control, mikmod_Seek, mikmod_Info, 
				mikmod_AudioStreams, mikmod_AudioInfo,
				mikmod_AudioRead, mikmod_AudioPTS,
				mikmod_AudioSeek,
				mikmod_VideoStreams, mikmod_VideoInfo,
				mikmod_VideoRead, mikmod_VideoPTS,
				mikmod_VideoSeek,
				mikmod_Check, mikmod_FileInfo };

/*
 * Internal code
 */

/*
 * MikMod driver
 */

/*
 * Functions
 */

static BOOL xmm_IsPresent( void )
{
  return 1;
}

static BOOL xmm_Reset( void )
{
  VC_Exit();
  return VC_Init();
}

static void xmm_Update( void )
{
}

/*
 * Driver data
 */

MIKMODAPI MDRIVER drv_xmm = {
	NULL,
	"xmm",
	"xmm output driver v1.0",
	0,
	255,
	"xmm",
	NULL,
	xmm_IsPresent,
	VC_SampleLoad,
	VC_SampleUnload,
	VC_SampleSpace,
	VC_SampleLength,
	VC_Init,
	VC_Exit,
	xmm_Reset,
	VC_SetNumVoices,
	VC_PlayStart,
	VC_PlayStop,
	xmm_Update,
	NULL,
	VC_VoiceSetVolume,
	VC_VoiceGetVolume,
	VC_VoiceSetFrequency,
	VC_VoiceGetFrequency,
	VC_VoiceSetPanning,
	VC_VoiceGetPanning,
	VC_VoicePlay,
	VC_VoiceStop,
	VC_VoiceStopped,
	VC_VoiceGetPosition,
	VC_VoiceRealVolume
};

/*
 * MREADER wrapper functions
 */

static int wrap_Seek( struct MREADER *_mreader, long offset, int whence )
{
  struct mikmod_reader	*mreader = (struct mikmod_reader *)_mreader;
  int			ret;

  ret = mreader->pIO->Seek( mreader->pIO, offset, whence );
  if( ret < XMM_RET_OK )	ret = -1;

  return ret;
}

static long wrap_Tell( struct MREADER *_mreader )
{
  struct mikmod_reader	*mreader = (struct mikmod_reader *)_mreader;
  long			ret;

  ret = mreader->pIO->Tell( mreader->pIO );
  if( ret < XMM_RET_OK )	ret = -1;

  return ret;
}

static int wrap_Read( struct MREADER *_mreader, void *ptr, size_t size )
{
  struct mikmod_reader	*mreader = (struct mikmod_reader *)_mreader;
  int			ret;

  ret = mreader->pIO->Read( mreader->pIO, ptr, size, 1 );
  if( ret < XMM_RET_OK )	ret = 0;

  return ret;
}

static int wrap_Get( struct MREADER *_mreader )
{
  struct mikmod_reader	*mreader = (struct mikmod_reader *)_mreader;
  char			tmp;

  if( mreader->pIO->Read( mreader->pIO, &tmp, 1, 1 ) < XMM_RET_OK )	tmp = -1;

  return (int) tmp;
}

static int wrap_Eof( struct MREADER *_mreader )
{
  struct mikmod_reader	*mreader = (struct mikmod_reader *)_mreader;
  int			ret;

  ret = mreader->pIO->Eof( mreader->pIO );
  if( ret < XMM_RET_OK )	ret = -1;

  return ret;
}
