/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * mpg123.c
 * MPEG 1.0/2.0/2.5 (Audio) Layer 1/2/3 DeMUX
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>

#include "mpg123head.h"

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/* Use libid3tag ( mad package )*/
#define	USE_libid3tag

/*
 * Types
 */

typedef int (*bq_raw_read_t)( void *bq, void *data, int size );
typedef int (*bq_raw_size_t)( void *bq );

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_AudioInfo		ai;
    XMM_ClipInfo		ci;

    double			cPTS;

    char			filepath[XMM_MAXPATHLEN];
    char 			*filename;
    int				seekable;

    uint32_t			data_offset, data_size;
    uint32_t			tFrames;

    /* Read buffer */
    char			*rbuffer;
    int				rbuffersize;

    MPG123_Info			mpg123_info;	/* MPEG header */
    VBRHeader			vbrhead;	/* VBR header */
    int				vbrhead_found;

};

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Prototypes
 */
 
int mpg123_id3tag_ci( char *filename, char *filetitle, XMM_ClipInfo *ci );

/*
 * Initialize
 */
static XMM_PluginInput *mpg123_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(MPG123) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int mpg123_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t	*priv = input->sys.priv;
  uint32_t	read, tagsize = 0;
  char		*buffer, *ptr;
  int		ret;
#ifdef VERBOSE
  char		*tab_mpeg[] = { "1", "2", "?", "2.5" };
  char		*tab_layer[] = { "?", "I", "II", "III" };
  char		*tab_flag[] = { "No", "Yes" };
  char		*tab_mode[] = { "Stereo", "Joint stereo", "Dual channel", "Mono" };
#endif
  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(MPG123) Only 'read' mode supported" );

  /* Open file */
  priv->pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->pIO == NULL )	return XMM_RET_ERROR;

  /* Seekable ? */
  if( priv->pIO->Seek( priv->pIO, 0, XMM_SEEK_CUR ) == XMM_RET_OK )
	priv->seekable = 1;
  else	priv->seekable = 0;
#ifdef VERBOSE
  xmm_logging( 2, "MPG123! Stream is %s seekable\n", priv->seekable ? "" : "NOT" );
#endif

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* Save filepath */
  strcpy( priv->filepath, filename );

  /* Get ID3 Tag */
  tagsize = mpg123_id3tag_ci( priv->filepath, priv->filename, &priv->ci );

  /* Allocate read buffer */
  priv->rbuffersize = 0;
  if(( priv->rbuffer = malloc( 4096 )) == NULL )
	return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(MPG123) rbuffer (%i bytes)", 4096 );

  priv->data_offset = 0;

  while( 1 )
  {
	/* Read data */
	read = priv->pIO->Read( priv->pIO, priv->rbuffer + priv->rbuffersize, 1, 4096 - priv->rbuffersize );
	if(( read == 0 ) && priv->pIO->Eof( priv->pIO ))
	{
		priv->pIO->Close( priv->pIO );
		return XMM_RET_EOS;
	}
	priv->rbuffersize += read;

	/* Get Info ( header data, VBR header ), find first frame */
	ret = MPG123_DecodeInfo( priv->rbuffer, priv->rbuffersize, &priv->mpg123_info, &priv->vbrhead, &priv->vbrhead_found );
	if( ret )
	{
	    priv->rbuffersize -= ABS( ret );
	    priv->data_offset += ABS( ret );
	    memmove( priv->rbuffer, priv->rbuffer + ABS( ret ), priv->rbuffersize );

#ifdef DEBUG
	    xmm_logging( 1, "MPG123! sync at %i ( ==> rbuffersize = %i )\n", ret, priv->rbuffersize );
#endif
	}

	if( ret >= 0 )	break;		/* header found */
	else	continue;
  }

#ifdef VERBOSE
  xmm_logging( 1, "MPG123! INFO: MPEG %s (audio) Layer %s, %i kbps, %i Hz, %s\n",
						tab_mpeg[priv->mpg123_info.mpeg],
						tab_layer[priv->mpg123_info.layer],
						priv->mpg123_info.bitrate,
						priv->mpg123_info.samprate,
						tab_mode[priv->mpg123_info.mode] );

  xmm_logging( 1, "MPG123! INFO: bytes / frame = %.2f (%i) time / frame = %.4fs\n",
						priv->mpg123_info.bpf,
						priv->mpg123_info.framesize,
						priv->mpg123_info.tpf );

  xmm_logging( 1, "MPG123! INFO: Emphasis = %i, CRC: %s, Copyright: %s, Original: %s\n",
					priv->mpg123_info.emphasis,
					tab_flag[ priv->mpg123_info.crc ],
					tab_flag[ priv->mpg123_info.copyright ],
					tab_flag[ priv->mpg123_info.original ] );
  
#endif

  /* VBR ( variable bitrate ) header */
  if( priv->vbrhead_found )
  {
#ifdef VERBOSE
	int i;
#endif
#ifndef VERBOSE
	xmm_logging( 2, "MPG123! VBR Header found...\n" );
#endif
#ifdef VERBOSE
	xmm_logging( 1, "MPG123! INFO(VBR): Flags = 0x%lx\n", priv->vbrhead.flags );
	xmm_logging( 1, "MPG123! INFO(VBR): Frames = %li\n", priv->vbrhead.frames );
	xmm_logging( 1, "MPG123! INFO(VBR): Bytes = %li\n", priv->vbrhead.bytes );
	xmm_logging( 1, "MPG123! INFO(VBR): Scale = %li", priv->vbrhead.scale );

	for( i = 0; i < VBR_TOC_ENTRIES; i++ )
	{
	    if(( i % 10 ) == 0 )	xmm_logging( 1, "\nMPG123! INFO(VBR): TOC (%2i): ", i );
	    xmm_logging( 1, "%3i ", priv->vbrhead.toc[i] );
	}
	xmm_logging( 1, "\n" );
#endif
  }

  /* Fill audio format struct */
  priv->ai.fmt.format = XMM_AUDIO_CODEC_MPEG | XMM_AUDIO_MASK_SIGNED | 0x10;
  priv->ai.fmt.samprate = priv->mpg123_info.samprate;
  priv->ai.fmt.channels = priv->mpg123_info.channels;
  priv->ai.fmt.blockSize = (uint16_t) priv->mpg123_info.bpf;
  priv->ai.fmt.extraSize = 0;
  priv->ai.fmt.desc[0] = '\0';
  priv->ai.tSize = priv->pIO->Size( priv->pIO ) - tagsize;
  priv->ai.offset = 0;

  if( priv->vbrhead_found )
  {
	priv->data_size = priv->vbrhead.bytes;
	priv->tFrames = priv->vbrhead.frames;
	priv->ai.fmt.bitrate = -((int32_t)(( priv->vbrhead.bytes / 
			((double) priv->tFrames * priv->mpg123_info.tpf )) *
			8 ));
  }
  else
  {
	priv->data_size = priv->ai.tSize;
	priv->tFrames = priv->ai.tSize / priv->mpg123_info.bpf;
	priv->ai.fmt.bitrate = priv->mpg123_info.bitrate * 1000;
  }

  priv->ai.tSamples = (uint32_t)((double) priv->tFrames *
			priv->mpg123_info.tpf *
			priv->ai.fmt.samprate );

  /* Clip Info */
  priv->ci.size = priv->ai.tSize;
  priv->ci.playtime = (double)priv->ai.tSamples / priv->ai.fmt.samprate;

  return XMM_RET_OK;
}

/*
 * Close
 */
static int mpg123_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  if( priv->filename )	free( priv->filename );
  if( priv->pIO )	priv->pIO->Close( priv->pIO );
  if( priv->rbuffer )	free( priv->rbuffer );

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int mpg123_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_AudioFormat	*format;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_AFORMAT:			/* Query sound format */
		format = (XMM_AudioFormat *)data;
		if(( format->format == priv->ai.fmt.format ) &&
		    ( format->samprate == priv->ai.fmt.samprate ) &&
		    ( format->channels == priv->ai.fmt.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_AFORMAT:			/* Get sound format */
		memcpy( data, &priv->ai.fmt, sizeof( XMM_AudioFormat ));
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->ai.fmt;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_CAPS:				/* Get capabilities */
		*((uint32_t *) data ) = XMM_INPUT_CF_MODE_READ | XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_CLIPINFO:
		return XMM_RET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(MPG123) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(MPG123) cmd ( 0x%x )" );
}

/*
 * Seek to position
 */
static int mpg123_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  struct priv_t	*priv = input->sys.priv;
  uint32_t	bytes, p1, sp1, sp2;
  double	sp;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(MPG123) Stream is not seekable." );

  /* Clip seek value */
  if( seek < 0.0 )	seek = 0.0;
  if( seek > 100.0 )	seek = 100.0;

  /* Calculate new position */
  if( priv->vbrhead_found )
  {
	/* Get left seek point */
	p1 = (int)( seek * 100 );
	if( p1 > 99 )	p1 = 99;
	sp1 = priv->vbrhead.toc[ p1 ];

	/* Get right seek point */
	sp2 = priv->vbrhead.toc[ ( p1 < 99 ) ? ( p1 + 1 ) : 99 ];

	/* Interpolate seek position */
	sp = sp1 + ( sp2 - sp1 ) * ( seek * 100 - p1 );

	/* Get new position in bytes */
	bytes = (uint32_t)(( sp / 256.0 ) * priv->vbrhead.bytes );

	/* Update cSample, actually we guess it... because of the VBR */
	priv->cPTS = ( seek * priv->ai.tSamples ) / priv->ai.fmt.samprate;

	priv->rbuffersize = 0;	/* Flush read buffer */

	/* I/O Seek */
	return priv->pIO->Seek( priv->pIO, priv->data_offset + bytes, XMM_SEEK_SET );
  }

  return input->AudioSeek( input, astream, (uint32_t)( seek * priv->ci.playtime * 1000 ));
}

/*
 * Get Information
 */
static int mpg123_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )	memcpy( ci, &priv->ci, sizeof( XMM_ClipInfo ));

  return XMM_RET_OK;
}

/*
 * Get audio stream number
 */
static int mpg123_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int mpg123_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int mpg123_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  struct priv_t *priv = input->sys.priv;
  uint32_t	read = 0, done = 0;
  int		ret;

  if( size == 0 )	return 0;	/* Yes... */

  while( 1 )				/* Get the data */
  {
    if( priv->rbuffersize > 0 )
    {
	ret = MPG123_DecodeInfo( priv->rbuffer, priv->rbuffersize, &priv->mpg123_info, NULL, NULL );
	if( ret != 0 )	xmm_logging( 2, "MPG123! WARNING: header at pos %i in rbuffer ( 0 expected )\n", ret );

#ifdef DEBUG
	xmm_logging( 1, "MPG123! sync at %i ( rbuffersize = %i ), framesize = %i total = %i\n", ret, priv->rbuffersize, priv->mpg123_info.framesize, size );
#endif

	/* only copy complete frames */
	if(( done + priv->mpg123_info.framesize ) > size )	break;

	/* copy frame */
	if( priv->rbuffersize >= ( 4 + priv->mpg123_info.framesize ))
	{
		read = priv->mpg123_info.framesize + 4;

		memcpy( buffer, priv->rbuffer, read );
		done += read;
		buffer += read;

		priv->rbuffersize -= read;
		memmove( priv->rbuffer, priv->rbuffer + read, priv->rbuffersize );

#ifdef DEBUG
		xmm_logging( 1, "MPG123! rbuffersize = %i ( -%i )\n", priv->rbuffersize, read );
#endif

		priv->cPTS += priv->mpg123_info.tpf;
		continue;
	}
    }

    while( 1 )
    {
#ifdef DEBUG
	xmm_logging( 1, "MPG123! rbuffersize = %i\n", priv->rbuffersize );
#endif

	/* Read data */
	read = priv->pIO->Read( priv->pIO, priv->rbuffer + priv->rbuffersize, 1, 4096 - priv->rbuffersize );
	if(( read == 0 ) && priv->pIO->Eof( priv->pIO ))
	{
	    if( done == 0 )	return XMM_RET_EOS;
	    return done;
	}

	priv->rbuffersize += read;

#ifdef DEBUG
	xmm_logging( 1, "MPG123! rbuffersize = %i ( +%i )\n", priv->rbuffersize, read );
#endif

	/* Get Info ( header data, VBR header ), find first frame */
	ret = MPG123_DecodeInfo( priv->rbuffer, priv->rbuffersize, &priv->mpg123_info, &priv->vbrhead, &priv->vbrhead_found );
	if( ret != 0 )
	{
	    xmm_logging( 2, "MPG123! WARNING: sync lost !!! header at pos %i in rbuffer ( 0 expected )\n", ret );

	    priv->rbuffersize -= ABS( ret );
	    memmove( priv->rbuffer, priv->rbuffer + ABS( ret ), priv->rbuffersize );

#ifdef DEBUG
	    xmm_logging( 1, "MPG123! sync at %i ( ==> rbuffersize = %i )\n", ret, priv->rbuffersize );
#endif
	}

	if( ret >= 0 )	break;		/* header found */
	else	continue;
    }
  }

  return done;
}

/*
 * Audio timestamp.
 */
static int mpg123_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  struct priv_t	*priv = input->sys.priv;

  if( pts )	*pts = (uint32_t)( priv->cPTS * 1000 );

  return XMM_RET_OK;
}

/*
 * Seek to position in audio stream
 */
static int mpg123_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts )
{
  struct priv_t	*priv = input->sys.priv;
  uint32_t	bytes;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(MPG123) Stream is not seekable." );

  if( priv->vbrhead_found )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(MPG123) No PTS based seeking for VBR streams." );

  /* Calculate position */
  priv->cPTS = pts / 1000;
  bytes = (uint32_t)( priv->cPTS / priv->mpg123_info.tpf * priv->mpg123_info.bpf );

  priv->rbuffersize = 0;	/* Flush read buffer */

  /* I/O Seek */
  return priv->pIO->Seek( priv->pIO, priv->data_offset + bytes, XMM_SEEK_SET );
}

/*
 * MPEG Audio Layer 1/2/3 files do not contain video data
 */
static int mpg123_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int mpg123_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mpg123_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mpg123_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mpg123_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if MPG123 file
 */
static int mpg123_Check( void *xmm, char *filename )
{
  char *ptr;

  ptr = strrchr( filename, '.' );
  if( ptr == NULL )	return XMM_RET_ERROR;

  return ( !strcasecmp( ptr, ".mp3" ) || !strcasecmp( ptr, ".mp2" ));
}

/*
 * Get file info
 */
static int mpg123_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"MPG123",
				"Input: MPEG 1.0/2.0/2.5 Layer 1/2/3",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				mpg123_Init, mpg123_Open, mpg123_Close,
				mpg123_Control, mpg123_Seek, mpg123_Info, 
				mpg123_AudioStreams, mpg123_AudioInfo,
				mpg123_AudioRead, mpg123_AudioPTS,
				mpg123_AudioSeek,
				mpg123_VideoStreams, mpg123_VideoInfo,
				mpg123_VideoRead, mpg123_VideoPTS,
				mpg123_VideoSeek,
				mpg123_Check, mpg123_FileInfo };

/*
 * Internal codec
 */

#ifdef USE_libid3tag

#include "libid3tag/id3tag.h"

int mpg123_id3tag_ci( char *filename, char *filetitle, XMM_ClipInfo *ci )
{
  int			i, j, nstrings;
  struct id3_file	*file;
  struct id3_tag	*tag;
  union id3_field	const *field;
  struct id3_frame	const *frame;
  id3_ucs4_t		const *ucs4;
  id3_latin1_t		*latin1;
  struct
  {
	char const	*id;
	char const	*name;
	char		*ptr;
	int		len;
  } const info[] =
  {
    { ID3_FRAME_TITLE,	"Title",	ci->name,	XMM_CIL_NAME },
    { ID3_FRAME_ARTIST,	"Artist",	ci->author,	XMM_CIL_AUTHOR },
    { ID3_FRAME_ALBUM,	"Album",	ci->album,	XMM_CIL_ALBUM },
    { ID3_FRAME_GENRE,	"Genre",	ci->content,	XMM_CIL_CONTENT },
    { "TCOP",		"Copyright",	ci->copyright,	XMM_CIL_COPYRIGHT },
    { "TENC",		"Encoder",	ci->software,	XMM_CIL_SOFTWARE },
    { ID3_FRAME_YEAR,	"Year",		ci->date + 4,	XMM_CIL_DATE - 4 },
    { NULL, NULL }
  };

  file = id3_file_open( filename, ID3_FILE_MODE_READONLY );
  if( file == NULL )	return 0;
  tag = id3_file_tag( file );

#if 0
  ci->flags = XMM_CIF_NAME | XMM_CIF_AUTHOR | XMM_CIF_ALBUM | XMM_CIF_CONTENT |
		XMM_CIF_DATE | XMM_CIF_COMMENT;

  if(( tag->options & ID3_TAG_OPTION_ID3V1 ) == 0 )
  {
	xmm_logging( 1, "MPG123! ID3: v1 not found\n" );
	ci->flags = ci->flags | XMM_CIF_SOFTWARE | XMM_CIF_COPYRIGHT;
  }
#endif

  strncpy( ci->name, filetitle, XMM_CIL_NAME - 1 );
  ci->name[XMM_CIL_NAME - 1] = '\0';

  for( i = 0; info[i].id; i++ )
  {
	frame = id3_tag_findframe( tag, info[i].id, 0 );
	if( frame == NULL )	continue;

	field = &frame->fields[1];
	nstrings = id3_field_getnstrings( field );

#ifdef VERBOSE
	xmm_logging( 1, "MPG123! ID3: %i string(s) for '%s'\n", nstrings, info[i].name );
#endif

	for( j = 0; j < nstrings; j++ )
	{
	    ucs4 = id3_field_getstrings( field, j );
	    if( ucs4 == NULL )	continue;

	    if( strcmp( info[i].id, ID3_FRAME_GENRE ) == 0 )
		ucs4 = id3_genre_name( ucs4 );

	    latin1 = id3_ucs4_latin1duplicate( ucs4 );
	    if( latin1 == 0 )	continue;

#ifdef VERBOSE
	    xmm_logging( 1, "\t%s\n", latin1 );
#endif	    

	    if( j == 0 )
	    {
		if( strcmp( info[i].id, ID3_FRAME_YEAR ) == 0 )
		      strcpy( ci->date, "0000" );

		strncpy( info[i].ptr, latin1, info[i].len - 1 );
		info[i].ptr[info[i].len - 1] = '\0';
	    }
	}
  }

  /* Get comment */
  for( i = 0;; i++ )
  {
	frame = id3_tag_findframe( tag, ID3_FRAME_COMMENT, i );
	if( frame == NULL )	break;

	ucs4 = id3_field_getstring( &frame->fields[2] );
	if( ucs4 == NULL )	continue;
	if( *ucs4 != '\0' )	continue;

	ucs4 = id3_field_getfullstring( &frame->fields[3] );
	if( ucs4 == NULL )	continue;

	latin1 = id3_ucs4_latin1duplicate( ucs4 );
	if( latin1 == 0 )	continue;

#ifdef VERBOSE
	xmm_logging( 1, "MPG123! ID3: Comment(%i): %s\n", i, latin1 );
#endif

	if( i == 0 )
	{
		strncpy( ci->comment, latin1, XMM_CIL_COMMENT - 1 );
		ci->comment[XMM_CIL_COMMENT - 1] = '\0';
	}
  }

  id3_file_close( file );

  return 0;
}

#else

#include "id3v1.h"

int mpg123_id3tag_ci( char *filename, char *filetitle, XMM_ClipInfo *ci )
{
  ID3v1tag	id3tag;	/* ID3 tag */
  int		tagsize = 0;

  strcpy( id3tag.Title, filetitle );

  for( id3tag.Genre = 0; tab_genre[id3tag.Genre][0] != '\0'; id3tag.Genre++ );
  id3tag.Artist[0] = '\0';
  id3tag.Album[0] = '\0';
  id3tag.Comment[0] = '\0';
  id3tag.Year[0] = '\0';
  if( id3v1tag_read( filename, &id3tag ) == 1 )	tagsize = 128;

  strncpy( ci->name, id3tag.Title, XMM_CIL_NAME - 1 );
  ci->name[XMM_CIL_NAME - 1] = '\0';
  strncpy( ci->author, id3tag.Artist, XMM_CIL_AUTHOR - 1 );
  ci->author[XMM_CIL_AUTHOR - 1] = '\0';
  strncpy( ci->album, id3tag.Album, XMM_CIL_ALBUM - 1 );
  ci->album[XMM_CIL_ALBUM - 1] = '\0';
  strncpy( ci->content, tab_genre[id3tag.Genre], XMM_CIL_CONTENT - 1 );
  ci->content[XMM_CIL_CONTENT - 1] = '\0';
  ci->copyright[0] = '\0';
  ci->software[0] = '\0';
  ci->date[0] = '\0';
  if( tagsize )
  {
	strncpy( ci->date, "0000", 4 );
	strncpy( ci->date + 4, id3tag.Year, 4 );
  }
  ci->date[XMM_CIL_DATE - 1] = '\0';
  strncpy( ci->comment, id3tag.Comment, XMM_CIL_COMMENT - 1 );
  ci->comment[XMM_CIL_COMMENT - 1] = '\0';

#if 0
  ci->flags = XMM_CIF_NAME | XMM_CIF_AUTHOR | XMM_CIF_ALBUM | XMM_CIF_CONTENT |
		XMM_CIF_DATE | XMM_CIF_COMMENT;
#endif

  return tagsize;
}

#endif
