/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include "mpg123head.h"

/*
 * Definitions
 */

#define	MPEG_MODE_MONO		3

/*
 * Types
 */

typedef struct frame_header_s
{
    int		mpeg25;
    int		lsf;
    int		layer;
    int		crc;
    int		bridx;
    int		sridx;
    int		padding;
    int		extension;
    int		mode;
    int		mode_ext;
    int 	copyright;
    int		original;
    int		emphasis;
} frame_header_t;

/*
 * Data ( Tables )
 */

static int	tab_bitrate[2][16][3] =
{
 { /* MPEG 1 */
  { 0, 0, 0 }, { 32, 32, 32 }, { 64, 48, 40 }, { 96, 56, 48 },
  { 128, 64, 56 }, { 160, 80, 64 }, { 192, 96, 80 }, { 224, 112, 96 },
  { 256, 128, 112 }, { 288, 160, 128 }, { 320, 192, 160 }, { 352, 224, 192 },
  { 384, 256, 224 }, { 416, 320, 256 }, { 448, 384, 320 }, { -1, -1, -1 }
 },
 { /* MPEG 2 */
  { 0, 0, 0 }, { 32,  8,  8 }, { 48, 16, 16 }, { 56, 24, 24 },
  { 64, 32, 32 }, { 80, 40, 40 }, { 96, 48, 48 }, { 112, 56, 56 },
  { 128, 64, 64 }, { 144, 80, 80 }, { 160, 96, 96 }, { 176,112,112 },
  { 192, 128, 128 }, { 224, 144, 144 }, { 256, 160, 160 }, { -1, -1, -1 }
 }
};

static long	tab_freq[9] = { 44100, 48000, 32000, 22050, 24000, 16000, 11025, 12000, 8000 };

static int	tab_bs[4] = { 0, 384, 1152, 1152 };

/*
 * Prototypes
 */

static int read_header( frame_header_t *frame, uint8_t *buffer, int size );
static int GetVBRHeader( frame_header_t *frame, uint8_t *buffer, int size, VBRHeader *head );

/*
 * Get MPEG Audio layer infos
 */
int MPG123_DecodeInfo( uint8_t *buffer, int size, MPG123_Info *info, VBRHeader *vbrhead, int *vbrhead_found )
{
  frame_header_t	frame;
  int			pos;

  pos = read_header( &frame, buffer, size );
  if( pos < 0 )	return pos;

  /* Fill info */
  info->bitrate = tab_bitrate[frame.lsf][frame.bridx][frame.layer - 1];
  info->samprate = tab_freq[frame.sridx];
  info->channels = ( frame.mode == MPEG_MODE_MONO ) ? 1 : 2;

  info->tpf = (double)tab_bs[frame.layer] / ( tab_freq[frame.sridx] << frame.lsf );
  switch( frame.layer )
  {
    case 1:
	info->bpf = 48000 * (double)tab_bitrate[frame.lsf][frame.bridx][0] / ( tab_freq[frame.sridx] << frame.lsf );
	break;

    case 2:
    case 3:
	info->bpf = 144000 * (double)tab_bitrate[frame.lsf][frame.bridx][frame.layer - 1] / ( tab_freq[frame.sridx] << frame.lsf );
	break;

  }

  switch( frame.layer )
  {
    case 1:
	info->framesize  = (long) tab_bitrate[frame.lsf][frame.bridx][0] * 12000;
	info->framesize /= tab_freq[frame.sridx];
	info->framesize  = ((info->framesize + frame.padding ) << 2 ) - 4;
	break;

    case 2:
	info->framesize = (long) tab_bitrate[frame.lsf][frame.bridx][1] * 144000;
        info->framesize /= tab_freq[frame.sridx];
        info->framesize += frame.padding - 4;
	break;

    case 3:
	info->framesize  = (long) tab_bitrate[frame.lsf][frame.bridx][2] * 144000;
	info->framesize /= tab_freq[frame.sridx] << ( frame.lsf );
	info->framesize += frame.padding - 4;
	break;
  }

  info->mpeg = frame.mpeg25 ? 3 : frame.lsf;
  info->layer = frame.layer;
  info->mode = frame.mode;
  info->crc = frame.crc;
  info->copyright = frame.copyright;
  info->original = frame.original;
  info->emphasis = frame.emphasis;

  /* VBR */
  if(( vbrhead_found != NULL ) && ( vbrhead != NULL ))
	*vbrhead_found = GetVBRHeader( &frame, buffer, size, vbrhead );

  return pos;
}

/*
 * Read header
 */

/*
 * Check header
 */
static int check_header( uint32_t head )
{
  if(( head & 0xffe00000 ) != 0xffe00000 )	return 0;	/* minimum */

  if( !(( head >> 17 ) & 3 ))		return 0;	/* layer = 1,2,3 */

  if((( head >> 12 ) & 0xf ) == 0xf )	return 0;	/* br index != 0xF */

  if((( head >> 12 ) & 0xf ) == 0x0 )	return 0;	/* br index != 0x0 */

  if((( head >> 10 ) & 0x3 ) == 0x3 )	return 0;	/* sr index != 0x3 */

  if((( head >> 19 ) & 1 ) == 1 &&
	(( head >> 17 ) & 3 ) == 3 &&
	(( head >> 16 ) & 1 ) == 1 )	return 0;
		// MPEG 2.0, layer 1, err_prot

  if(( head & 0xffff0000 ) == 0xfffe0000 )	return 0;

  return 1;
}

/*
 * Decode header
 */
static void decode_header( frame_header_t *fr, uint32_t head )
{
  if( head & ( 1 << 20 ))
  {
	fr->lsf = ( head & ( 1 << 19 )) ? 0x0 : 0x1;
	fr->mpeg25 = 0;
  }
  else
  {
	fr->lsf = 1;
	fr->mpeg25 = 1;
  }

  fr->layer = 4 - (( head >> 17 ) & 3 );
  fr->crc = (( head >> 16 ) & 0x1 ) ^ 0x1;
  fr->bridx = (( head >> 12 ) & 0xf );

  if( fr->mpeg25 )
	fr->sridx = 6 + ((head >> 10) & 0x3);
  else
	fr->sridx = ((head >> 10) & 0x3) + (fr->lsf * 3);

  fr->padding   = (( head >> 9 ) & 0x1 );
  fr->extension = (( head >> 8 ) & 0x1 );
  fr->mode      = (( head >> 6 ) & 0x3 );
  fr->mode_ext  = (( head >> 4 ) & 0x3 );
  fr->copyright = (( head >> 3 ) & 0x1 );
  fr->original  = (( head >> 2 ) & 0x1 );
  fr->emphasis  = head & 0x3;
}

/*
 * Read header
 */
static int read_header( frame_header_t *frame, uint8_t *data, int size )
{
  int		i = 0;
  uint32_t	head;

  head = (data[0] << 24) | (data[1] << 16) | (data[2] << 8) | data[3];
  data += 4;
  i = 4;

  while( 1 )
  {
	if( check_header( head ))	break;

	do
	{
		i++;

		head <<= 8;
		head |= *data++;
		head &= 0xffffffff;

	} while(( i < size ) && ( check_header( head ) == 0 ));
 
	if( i == size )	return -size + 4;	/* No valid header found */
  }

  decode_header( frame, head );
  return i - 4;
}

/*
 * VBR header
 */

/*
 * Global data
 */

static int	table_TagOffset[2][2] =
{
    { 17, 32 },		/* MPEG 1 ( Mono / Stereo ) */
    {  9, 17 }		/* MPEG 2 ( Mono / Stereo ) */
};

static char	VBRTagID[] = { 'X', 'i', 'n', 'g' };

/*
 * Prototypes
 */

static uint32_t get32bits( uint8_t *buffer );

/*
 * Get VBR Header
 */
static int GetVBRHeader( frame_header_t *frame, uint8_t *buffer, int size, VBRHeader *head )
{
  int	chidx;

  chidx = ( frame->mode == MPEG_MODE_MONO ) ? 0 : 1;

  /* Get offset */
  buffer += ( table_TagOffset[frame->lsf][chidx] + 4 );

  /* Has VBR Tag ? */
  if(( buffer[0] != VBRTagID[0] ) || ( buffer[1] != VBRTagID[1] ) ||
	( buffer[2] != VBRTagID[2] ) || ( buffer[3] != VBRTagID[3] ))
		return 0;
  buffer += 4;

  /* Get VBR header flags */
  head->flags = get32bits( buffer );
  buffer += 4;

  /* Get frame number */
  if( head->flags & VBR_FLAG_FRAMES )
  {
	head->frames = get32bits( buffer );
	buffer += 4;
  }

  /* Get byte number */
  if( head->flags & VBR_FLAG_BYTES )
  {
	head->bytes = get32bits( buffer ); 
	buffer += 4;
  }

  /* Get table of contents */
  if( head->flags & VBR_FLAG_TOC )
  {
	memcpy( head->toc, buffer, 100 );
	buffer += 100;
  }

  /* Get scale ( VBR quality: (0) best (100) worst ) */
  if( head->flags & VBR_FLAG_SCALE )
  {
	head->scale = get32bits( buffer );
	buffer += 4;
  }

  return 1;
}

/*
 * Private code
 */

static uint32_t get32bits( uint8_t *buffer )
{
  return (((uint32_t) buffer[0] ) << 24 ) | (((uint32_t) buffer[1] ) << 16 ) |
	(((uint32_t) buffer[2] ) << 8 ) | ((uint32_t) buffer[3] );
}
