/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * jpeg.c
 * (JFIF) JPEG handler
 */

#include <inttypes.h>
#include <stdio.h>

#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/lpgraph.h>
#include <libxmm/xmmp.h>
#include <libxmm/endian.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/*
 * Block IDs
 */

#define	JFIF_ID_SOI		0xffd8
#define	JFIF_ID_APP0		0xffe0
#define	JFIF_ID_DQT		0xffdb
#define	JFIF_ID_DHT		0xffc4
#define	JFIF_ID_SOF0		0xffc0
#define	JFIF_ID_SOF1		0xffc1
#define	JFIF_ID_SOF2		0xffc2
#define	JFIF_ID_SOS		0xffda
#define	JFIF_ID_EOI		0xffd9

/*
 * SOFn Block ( Image data definition )
 */
typedef struct __attribute__((packed))
{
    uint8_t		bPrecision;
    uint16_t		wHeight;
    uint16_t		wWidth;
    uint8_t		bComponents;

    struct __attribute__((packed))
    {
	uint8_t		bID;
	uint8_t		bFactor;
	uint8_t		bTable;
    }			comp[3];

} jpeg_sof_t;

/*
 * APP0 Block
 */
typedef struct __attribute__((packed))
{
    uint8_t			pbJFIF[5];	/* 0x4A 0x46 0x49 0x46 0x00 */
    uint16_t			wVersion;	/* version 0x01 0x01 */
    uint8_t			bResType;		/* 0x00 */
    uint16_t			wResHor;
    uint16_t			wResVer;
    uint8_t			bPHorSize;
    uint8_t			bPVerSize;
} jpeg_app0_t;

/*
 * Prototypes
 */

static uint8_t READ8( XMM_PluginIO *pIO );
static uint16_t READ16( XMM_PluginIO *pIO );
static int read_sof( XMM_PluginIO *pIO, jpeg_sof_t *sof, int size );
static int read_app0( XMM_PluginIO *pIO, jpeg_app0_t *app0, int size );

/*
 * Fill VideoFormat struct
 */
int pic_jpeg_format( void *xmm, char *filename, XMM_VideoFormat *vf )
{
  XMM_PluginIO	*pIO;
  uint16_t	wName, wSize, wName_tmp, wHeight = 0;
  int		found_sof = 0, found_app0 = 0, found_soi = 0, found_sos = 0;
  jpeg_sof_t	sof;
  jpeg_app0_t	app0;
  int		fields = 0, loop = 1, ret;

  /* Open file */
  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  /* Parse file */
  while( loop )
  {
	if( found_sos )		/* SOS - find EOI */
	{
	    wName_tmp = READ16( pIO );

	    while( wName_tmp != JFIF_ID_EOI )
	    {
		wName_tmp <<= 8;
		wName_tmp |= READ8( pIO );

		if( pIO->Eof( pIO ))	break;
	    }

	    found_sos = 0;
	    found_soi = 0;
	    if( pIO->Eof( pIO ))	break;
	}

	wName = READ16( pIO );
	if( pIO->Eof( pIO ))	break;

	/*
	 * Blocks / BlockIDs without size
	 */

	/* Note: I have a JPEG created by DC10, with 0xFFFF after JFIF_IO_EOI.
	 * Without this hack ( wName != 0xFFFF ), playback is not possible */
	if(( found_soi == 0 ) && ( wName != JFIF_ID_SOI ) && ( wName != 0xFFFF ))
		return xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, "(PIC) JPEG: no SOI marker found" );

	if( wName == JFIF_ID_SOI )	/* Start Of Image */
	{
	    found_soi = 1;
	    fields++;
	    continue;
	}

	if( wName == JFIF_ID_EOI )	/* End Of Image */
	{
	    found_soi = 0;
	    continue;
	}

	/*
	 * All other blocks
	 */

	wSize = READ16( pIO );
	if( pIO->Eof( pIO ))	break;

#ifdef VERBOSE
	xmm_logging( 1, "PIC! JPEG: block = 0x%x at 0x%x size = 0x%x\n", wName, pIO->Tell( pIO ) - 4, wSize );
#endif

	switch( wName )
	{

	    case JFIF_ID_APP0:
		    ret = read_app0( pIO, &app0, wSize );
		    found_app0 = 1;
		    break;

	    case JFIF_ID_SOF0:		/* baseline */
		    ret = read_sof( pIO, &sof, wSize );
		    wHeight += sof.wHeight;
		    found_sof = 1;
		    break;

	    case JFIF_ID_SOF1:		/* non-baseline */
		    ret = read_sof( pIO, &sof, wSize );
		    wHeight += sof.wHeight;
		    found_sof = 1;
		    break;

	    case JFIF_ID_SOF2:		/* progressive */
		    ret = read_sof( pIO, &sof, wSize );
		    wHeight += sof.wHeight;
		    found_sof = 1;
		    break;

	    case JFIF_ID_SOS:
		    ret = 0;
		    found_sos = 1;
		    break;

	    default:
		    ret = 0;
		    break;
		
	}

	pIO->Seek( pIO, wSize - ret - 2, XMM_SEEK_CUR );
  }

  pIO->Close( pIO );

   /* Fill video format struct */
  vf->codec = xmmFOURCC( 'M', 'J', 'P', 'G' );
  vf->width = sof.wWidth;
  vf->height = wHeight;
  vf->planes = 3;
  vf->bpp = 8;
  vf->imgSize = 0;
  vf->bitrate = 0;
  vf->framerate = 24;
  vf->aspect_val = 1;
  vf->aspect_div = 1;
  vf->extraSize = 0;
  vf->extraType = 0;
  vf->desc[0] = '\0';

  return XMM_RET_OK;
}

int pic_jpeg_read( void *xmm, char *filename, uint8_t *buffer )
{
  XMM_PluginIO	*pIO;
  int		ret;

  /* Open file */
  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  ret = pIO->Read( pIO, buffer, 1, pIO->Size( pIO ));
  pIO->Close( pIO );

  return ret;
}

/*
 * Internal
 */

static uint8_t READ8( XMM_PluginIO *pIO )
{
  uint8_t	tmp;

  pIO->Read( pIO, &tmp, 1, 1 );

  return tmp;
}

static uint16_t READ16( XMM_PluginIO *pIO )
{
  uint16_t	tmp;

  pIO->Read( pIO, &tmp, 2, 1 );

  return XMM_INT16_BE( tmp );
}

/*
 * Read SOFn
 */
static int read_sof( XMM_PluginIO *pIO, jpeg_sof_t *sof, int size )
{

  sof->bPrecision	= READ8( pIO );
  sof->wHeight		= READ16( pIO );
  sof->wWidth		= READ16( pIO );
  sof->bComponents	= READ8( pIO );

  sof->comp[0].bID	= READ8( pIO );
  sof->comp[0].bFactor	= READ8( pIO );
  sof->comp[0].bTable	= READ8( pIO );
  sof->comp[1].bID	= READ8( pIO );
  sof->comp[1].bFactor	= READ8( pIO );
  sof->comp[1].bTable	= READ8( pIO );
  sof->comp[2].bID	= READ8( pIO );
  sof->comp[2].bFactor	= READ8( pIO );
  sof->comp[2].bTable	= READ8( pIO );

#ifdef VERBOSE
 xmm_logging( 1, "PIC! JPEG: SOFn: bPrecision = %i\n", sof->bPrecision );
 xmm_logging( 1, "PIC! JPEG: SOFn: wHeight = %i\n", sof->wHeight );
 xmm_logging( 1, "PIC! JPEG: SOFn: wWidth = %i\n", sof->wWidth );
 xmm_logging( 1, "PIC! JPEG: SOFn: bComponents = %i\n", sof->bComponents );

 xmm_logging( 1, "PIC! JPEG: SOFn: comp[0]: bID = %x bFactor = %x btable = %i\n", sof->comp[0].bID, sof->comp[0].bFactor, sof->comp[0].bTable );
 xmm_logging( 1, "PIC! JPEG: SOFn: comp[1]: bID = %x bFactor = %x btable = %i\n", sof->comp[1].bID, sof->comp[1].bFactor, sof->comp[1].bTable );
 xmm_logging( 1, "PIC! JPEG: SOFn: comp[2]: bID = %x bFactor = %x btable = %i\n", sof->comp[2].bID, sof->comp[2].bFactor, sof->comp[2].bTable );
#endif

 return 15;
}

/*
 * Read APP0
 */
static int read_app0( XMM_PluginIO *pIO, jpeg_app0_t *app0, int size )
{
  app0->pbJFIF[0]	= READ8( pIO );
  app0->pbJFIF[1]	= READ8( pIO );
  app0->pbJFIF[2]	= READ8( pIO );
  app0->pbJFIF[3]	= READ8( pIO );
  app0->pbJFIF[4]	= READ8( pIO );

  app0->wVersion	= READ16( pIO );
  app0->bResType	= READ8( pIO );
  app0->wResHor		= READ16( pIO );
  app0->wResVer		= READ16( pIO );
  app0->bPHorSize	= READ8( pIO );
  app0->bPVerSize	= READ8( pIO );

#ifdef VERBOSE
 xmm_logging( 1, "PIC! JPEG: APP0: pbJFIF = %s\n", app0->pbJFIF );
 xmm_logging( 1, "PIC! JPEG: APP0: wVersion = %i\n", app0->wVersion );
 xmm_logging( 1, "PIC! JPEG: APP0: bResType = %i\n", app0->bResType );
 xmm_logging( 1, "PIC! JPEG: APP0: wResHor = %i\n", app0->wResHor );
 xmm_logging( 1, "PIC! JPEG: APP0: wResVer = %i\n", app0->wResVer );
 xmm_logging( 1, "PIC! JPEG: APP0: bPHorSize = %i\n", app0->bPHorSize );
 xmm_logging( 1, "PIC! JPEG: APP0: bPVerSize = %i\n", app0->bPVerSize );
#endif

 return 14;
}
