/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * list.c
 * PicList file reader
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include "list.h"

int pic_list_read( void *xmm, char *filename, pic_list_t *pl )
{
  XMM_PluginIO	*pIO;
  char		buffer[XMM_MAXPATHLEN], *ptr;
  char		tmp[XMM_MAXPATHLEN], path[XMM_MAXPATHLEN];
  int		i, line, section = 0;

  /* Open file */
  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  /* Initialize structure */
  pl->count = 0;
  pl->list = NULL;
  pl->framerate = 1;

  /* Read file */
  for( line = 1; ; line++ )
  {
	/* Read line */
	for( i = 0; i < XMM_MAXPATHLEN; i++ )
	{
		pIO->Read( pIO, &buffer[i], 1, 1 );
		if( pIO->Eof( pIO ))	break;
		if( buffer[i] == '\n' )	break;
	}

	if(( i == 0 ) && pIO->Eof( pIO ))	break;

	if( i == XMM_MAXPATHLEN )
	{
		pIO->Close( pIO );
		return xmm_SetError( xmm, XMM_RET_ERROR, "(PIC/list) line %i too long ( max %i chars )", line, XMM_MAXPATHLEN );
	}

	buffer[i] = '\0';

	if( buffer[0] == ';' )	continue;
	if( buffer[0] == '\n' )	continue;

	ptr = strchr( buffer, ';' );
	if( ptr )	*ptr = '\0';

	ptr = strchr( buffer, '\n' );
	if( ptr )	*ptr = '\0';

	if( buffer[0] == '\0' )	continue;

	if( buffer[0] == '[' )
	{
		ptr = strchr( buffer, ']' );
		if( ptr == NULL )
		{
			pIO->Close( pIO );
			return xmm_SetError( xmm, XMM_RET_ERROR, "(PIC/list) Parse ERROR in line %i, no ']' found", line );
		}
		*ptr = '\0';

		if( !strncasecmp( buffer +1, "PicList", 7 ))	section = 1;
		else if( !strncasecmp( buffer +1, "Input", 5 ))	section = 2;
		else 
		{
			pIO->Close( pIO );
			return xmm_SetError( xmm, XMM_RET_ERROR, "(PIC/list) Unknown section '%s' in line %i", buffer + 1, line );
		}
	}
	else
	{
		if( section == 0 )
		{
			pIO->Close( pIO );
			return xmm_SetError( xmm, XMM_RET_ERROR, "(PIC/list) Entry '%s' outside section in line %i", buffer, line );
		}

		if( !strncasecmp( buffer, "SourceDir=", 10 ))
		{
			strcpy( path, buffer + 10 );
		}
		else if( !strncasecmp( buffer, "Framerate=", 10 ))
		{
			pl->framerate = strtol( buffer + 10, NULL, 10 );
		}
		else
		{
			strcpy( tmp, path );
			strcat( tmp, XMM_DIR_SEPERATOR_S );
			strcat( tmp, buffer );
			pl->list = xmmList_Append( pl->list, strdup( tmp ));
		}
	}
  }

  pIO->Close( pIO );
  return XMM_RET_OK;
}
