/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * pic.c
 * Singe pic/frame input plugin
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/list.h>
#include <libxmm/util/utils.h>

#include "list.h"
#include "jpeg.h"

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Types
 */

/*
 * Private plugin data
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_VideoInfo		vi;

    /* Clip info */
    XMM_ClipInfo		ci;
    int				position;

    /* Misc stuff */
    int				list;
    pic_list_t			pl;
};

/*
 * Prototypes
 */

/*
 * Initialize
 */
static XMM_PluginInput *pic_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(PIC) plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int pic_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t	*priv = input->sys.priv;
  char		*buffer, *ptr, *filebase;
  XMM_List	*node;
  int		ret;

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(PIC) Only 'read' mode supported" );

  /* Check filename string */
  if( !strncasecmp( filename, "pic://", 6 ))	ptr = filename + 6;
  else if( !strncasecmp( filename, "pic:", 4 ))	ptr = filename + 4;
  else ptr = filename;

  /* Check for list */
  ret = pic_list_read( input->sys.xmm, ptr, &priv->pl );
  if( ret == XMM_RET_OK )
  {
	node = xmmList_GetNodePos( priv->pl.list, 1 );
	if( node )	ptr = (char *)node->data;
	else	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(PIC) Unable to get first node ( internal error ? )" );
	priv->list = 1;
  }

  /* Check for JPEG format */
  ret = pic_jpeg_format( input->sys.xmm, ptr, &priv->vi.fmt );
  if( ret != XMM_RET_OK )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(PIC) Unsupported file format '%s'", ptr );

  /* One single frame */
  if( priv->list == 0 )
  {
	priv->pl.list = xmmList_Append( NULL, strdup( filename ));
	priv->pl.count = 1;
	priv->pl.framerate = 1;
  }
  
  /* VideoInfo */
  priv->vi.fmt.framerate = priv->pl.framerate;
  priv->vi.tFrames = xmmList_Length( priv->pl.list );
  priv->vi.tSize = 0;
  priv->vi.offset = 0;

  /* Clip info */
  memset( &priv->ci, 0, sizeof( XMM_ClipInfo ));

  priv->ci.size = 0;
  priv->ci.playtime = ((double)priv->vi.tFrames / priv->vi.fmt.framerate );

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	filebase = strdup( ptr + 1 );
  else	filebase = strdup( buffer );
  free( buffer );

  strncpy( priv->ci.content, filebase, XMM_CIL_CONTENT - 1 );
  priv->ci.content[XMM_CIL_CONTENT - 1] = '\0';
  free( filebase );

  return XMM_RET_OK;
}

/*
 * Close
 */
static int pic_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;
  XMM_List	*node;

  for( node = priv->pl.list; node; node = node->next )	free( node->data );
  xmmList_Free( priv->pl.list );

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int pic_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:			/* Query graph format */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_AFORMAT:			/* Query sound format */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_YFLIP:			/* Query flip state */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_GFORMAT:			/* Get graph format */
		memcpy( data, &priv->vi.fmt, sizeof( XMM_VideoFormat ));
		((XMM_VideoFormat *)data)->extraSize = 0;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_GFORMAT_PTR:			/* Get graph format */
		*((XMM_VideoFormat **) data) = &priv->vi.fmt;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT:			/* Get sound format */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_CAPS:				/* Get capabilities */
		*((uint32_t *) data ) =	XMM_INPUT_CF_MODE_READ |
					XMM_INPUT_CF_VIDEO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:			/* Set graph format */
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(PIC) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(PIC) cmd ( 0x%x )" );
}

/*
 * Seek to position
 */
static int pic_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Get Information
 */
static int pic_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  struct priv_t		*priv = input->sys.priv;

  if( ci )	memcpy( ci, &priv->ci, sizeof( XMM_ClipInfo ));  

  if( seekval )		*seekval = priv->position / priv->vi.tFrames;

  return XMM_RET_OK;
}


/*
 * Get audio stream number
 */
static int pic_AudioStreams( XMM_PluginInput *input )
{
  return 0;
}

/*
 * Get audio stream information
 */
static int pic_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Read audio data
 */
static int pic_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Audio timestamp.
 */
static int pic_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Seek to position in audio stream
 */
static int pic_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Get video stream number
 */
static int pic_VideoStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get video stream information
 */
static int pic_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  struct priv_t		*priv = input->sys.priv;

  /* */
  if( vi )	memcpy( vi, &priv->vi, sizeof( XMM_VideoInfo ));
  if( cFrame )	*cFrame = priv->position;

  return XMM_RET_OK;
}

/*
 * Read video data
 */
static int pic_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  struct priv_t	*priv = input->sys.priv;
  XMM_List	*node;
  int		ret;

  /* Check for EOS */
  if(( priv->position + 1 ) >= priv->vi.tFrames )	return XMM_RET_EOS;

  node = xmmList_GetNodePos( priv->pl.list, priv->position + 1 );
  if( node == NULL )	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(PIC) Unable to get node %i ( internal error ? )", priv->position + 1 );

  /* Read file */
  ret = pic_jpeg_read( input->sys.xmm, (char *)node->data, buffer[0] );

  priv->position++;
  return ret;
}

/*
 * Video PTS
 */
static int pic_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  struct priv_t		*priv = input->sys.priv;

  if( videoPTS == NULL )
	return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(PIC) videoPTS = NULL" );

  *videoPTS = (uint32_t)((double)priv->position * 1000 / priv->vi.fmt.framerate );

  return XMM_RET_OK;
}

/*
 * Seek to position in video stream
 */
static int pic_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  struct priv_t		*priv = input->sys.priv;

  /* Check for EOS */
  if( frame >= priv->vi.tFrames )
  {
	xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(PIC) Invalid frame number: %i, Stream contains only %i frames", frame, priv->vi.tFrames );
	return XMM_RET_INVALID_ARG;
  }

  priv->position = frame;
  return frame;
}

/*
 * Check if picture
 */
static int pic_CheckFile( void *xmm, char *filename )
{
  char *ptr;

  if( !( strncasecmp( filename, "pic://", 6 )))	return 1;
  if( !( strncasecmp( filename, "pic:", 4 )))	return 1;

  ptr = strrchr( filename, '.' );
  if( ptr == NULL )	return XMM_RET_ERROR;

  return ( !strcasecmp( ptr, ".jpg" ) || !strcasecmp( ptr, ".jpeg" ));
}

/*
 * Get file info
 */
static int pic_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"PIC",
				"Input: Single Picture/Frame",
				"Copyright (c) 2002 Arthur Kleer",
				NULL, NULL },
				pic_Init, pic_Open, pic_Close,
				pic_Control, pic_Seek, pic_Info,
				pic_AudioStreams, pic_AudioInfo,
				pic_AudioRead, pic_AudioPTS, pic_AudioSeek,
				pic_VideoStreams, pic_VideoInfo,
				pic_VideoRead, pic_VideoPTS, pic_VideoSeek,
				pic_CheckFile, pic_FileInfo };
