/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * voc.c
 * VOC DeMUX
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/endian.h>
#include <libxmm/util/utils.h>
#include "voc.h"

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#define	VERBOSE
#endif
 
/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_AudioInfo		ai;

    /* Clip info */
    XMM_ClipInfo		ci;

    char 			*filename;
    int				seekable;
    uint32_t			data_offset, data_size, data_type;
    int				byterate;

    /* VOC headers */
    voc_header			head;
    voc_snd_info		sndinfo;
    voc_snd_info2		sndinfo2;
    voc_silence			silence;
    voc_marker			marker;
    voc_repeat			repeat;
    voc_extended		extended;
};


/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Prototypes
 */

static uint32_t	READn( XMM_PluginIO *pIO, void *data, uint32_t size );
static uint32_t	READ32( XMM_PluginIO *pIO );
static uint16_t	READ16( XMM_PluginIO *pIO );
static uint8_t	READ8( XMM_PluginIO *pIO );
static uint32_t READ_IEEE_EXT( XMM_PluginIO *pIO );
static int	SEEK( XMM_PluginIO *pIO, long offset, int seekable );

/*
 * Initialize
 */
static XMM_PluginInput *voc_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(VOC) plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int voc_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  char			*buffer, *ptr;
  uint8_t		type, loop = 1;
  uint32_t		size;
  char			*type_tab[] =
  {
	"Terminator", "Sound data", "Sound continue", "Silence", "Marker",
	"ASCII", "Repeat", "End Repeat", "Extended", "Type 9",
	"Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown"
  };

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(VOC) Only 'read' mode supported" );

  /* Open file */
  priv->pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->pIO == NULL )	return XMM_RET_ERROR;

  /* Seekable ? */
  if( priv->pIO->Seek( priv->pIO, 0, XMM_SEEK_CUR ) == XMM_RET_OK )
	priv->seekable = 1;
  else	priv->seekable = 0;
#ifdef VERBOSE
  xmm_logging( 2, "VOC! Stream is %s seekable\n", priv->seekable ? "" : "NOT" );
#endif

  /* Clip info */
  memset( &priv->ci, 0, sizeof( XMM_ClipInfo ));

  /* main VOC header */
  READn( priv->pIO, priv->head.pcID, 19 );
  priv->head.cID = READ8( priv->pIO );
  priv->head.wOffset = READ16( priv->pIO );
  priv->head.wVersion = READ16( priv->pIO );
  priv->head.wVersion2 = READ16( priv->pIO );

  if( strncmp( priv->head.pcID, VOC_FORMAT_ID, strlen( VOC_FORMAT_ID )))
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(VOC) No VOC file ID found" );

  if(( priv->head.wVersion != VOC_VERSION10 ) && ( priv->head.wVersion != VOC_VERSION20 ))
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(VOC) Unknown VOC file version found" );

#ifdef VERBOSE
  xmm_logging( 1, "VOC! INFO: wOffset = 0x%x [%i]\n", priv->head.wOffset, priv->head.wOffset );
  xmm_logging( 1, "VOC! INFO: wVersion = 0x%x [%i]\n", priv->head.wVersion, priv->head.wVersion );
  xmm_logging( 1, "VOC! INFO: wVersion2 = 0x%x [%i]\n", priv->head.wVersion2, priv->head.wVersion2 );
#endif

  /* Parse file */
  while( loop )
  {
	type = READ8( priv->pIO );
	if( priv->pIO->Eof( priv->pIO ))	break;

	if( type == VOC_TYPE_TERMINATOR )
	{
#ifdef VERBOSE
		xmm_logging( 1, "VOC! Block %x ( %s ) at %x\n", type, type_tab[type], priv->pIO->Tell( priv->pIO ) - 1 );
#endif
		break;
	}

	size = READ8( priv->pIO );	/* Big endian */
	size <<= 8;
	size |= READ8( priv->pIO );
	size <<= 8;
	size |= READ8( priv->pIO );
	size <<= 8;

#if XMM_BYTE_ORDER == XMM_LITTLE_ENDIAN
	size = XMM_INT32_BE( size );
#endif

#ifdef VERBOSE
	xmm_logging( 1, "VOC! Block %x ( %s ) size %x (%li) at %x\n", type, type_tab[type], size, size, priv->pIO->Tell( priv->pIO ) - 4 );
#endif

	switch( type )
	{
	    case VOC_TYPE_SOUND_DATA:
		    priv->sndinfo.sr = READ8( priv->pIO );
		    priv->sndinfo.df = READ8( priv->pIO );
		    if( priv->sndinfo.df != 0 )
			return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(VOC) Sound Data: Unsupported data type (%i)", priv->sndinfo.df );

#ifdef VERBOSE
		    xmm_logging( 1, "VOC! INFO: Sound Data: sr = 0x%x [%i] %i Hz\n", priv->sndinfo.sr, priv->sndinfo.sr, VOC_SRB2RATE(priv->sndinfo.sr));
		    xmm_logging( 1, "VOC! INFO: Sound Data: df = 0x%x [%i]\n", priv->sndinfo.df, priv->sndinfo.df );
#endif
		    priv->data_offset = priv->pIO->Tell( priv->pIO );
		    priv->data_size = size;
		    priv->data_type = type;

		    if( priv->seekable )	priv->pIO->Seek( priv->pIO, priv->data_size, XMM_SEEK_CUR );
		    else	loop = 0;

		    xmm_logging( 2, "VOC! data_offset = %li (%lx) data_size = %li (%lx)\n", priv->data_offset, priv->data_offset, priv->data_size, priv->data_size );
		    break;


	    case VOC_TYPE_SOUND_CONT:
		    SEEK( priv->pIO, size, priv->seekable );
		    break;

	    case VOC_TYPE_SILENCE:
		    priv->silence.wLength = READ16( priv->pIO );
		    priv->sndinfo.sr = READ8( priv->pIO );
#ifdef VERBOSE
		    xmm_logging( 1, "VOC! INFO: Silence: wLength = 0x%x [%i]\n", priv->silence.wLength, priv->silence.wLength );
		    xmm_logging( 1, "VOC! INFO: Silence: sr = 0x%x [%i] %i Hz\n", priv->silence.sr, priv->silence.sr, VOC_SRB2RATE(priv->silence.sr));
#endif
		    break;

	    case VOC_TYPE_MARKER:
		    priv->marker.value = READ16( priv->pIO );
#ifdef VERBOSE
		    xmm_logging( 1, "VOC! INFO: Marker: value = 0x%x [%i]\n", priv->silence.wLength, priv->silence.wLength );
#endif
		    break;

	    case VOC_TYPE_ASCII:
		    if(( ptr = malloc( size )) == NULL )
		    {
			    xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(VOC) ASCII: %i bytes", size );
			    break;
		    }
		    READn( priv->pIO, ptr, size );
#ifdef VERBOSE
		    xmm_logging( 1, "VOC! INFO: ASCII: %s\n", ptr );
#endif
		    free( ptr );
		    break;

	    case VOC_TYPE_REPEAT:
		    priv->repeat.value = READ16( priv->pIO );
#ifdef VERBOSE
		    xmm_logging( 1, "VOC! INFO: Repeat: value = 0x%x [%i] (%s)\n", priv->repeat.value, priv->repeat.value, ( priv->repeat.value == 0xFFFF ) ? "endless" : "" );
#endif
		    break;

	    case VOC_TYPE_REPEAT_END:
		    break;

	    case VOC_TYPE_EXTENDED:
		    priv->extended.wTimeConstant = READ16( priv->pIO );
		    priv->extended.bPack = READ8( priv->pIO );
		    priv->extended.bMode = READ8( priv->pIO );
#ifdef VERBOSE
		    xmm_logging( 1, "VOC! INFO: Extended: wTimeConstant = 0x%x [%i] ( %i Hz )\n", priv->extended.wTimeConstant, priv->extended.wTimeConstant, ( 256000000 >> priv->extended.bMode ) / ( 65536 - priv->extended.wTimeConstant ));
#endif
		    break;

	    case VOC_TYPE_SOUND_DATA_NEW:
		    priv->sndinfo2.dwSampleRate = READ32( priv->pIO );
		    priv->sndinfo2.bSampleSize = READ8( priv->pIO );
		    priv->sndinfo2.bChannels = READ8( priv->pIO );
		    priv->sndinfo2.bUnknown1 = READ8( priv->pIO );
		    READn( priv->pIO, priv->sndinfo2.pbUnknown2, 5 );
#ifdef VERBOSE
		    xmm_logging( 1, "VOC! INFO: Sound Data(New): dwSampleRate = 0x%x [%i]\n", priv->sndinfo2.dwSampleRate, priv->sndinfo2.dwSampleRate );
		    xmm_logging( 1, "VOC! INFO: Sound Data(New): bSampleSize = 0x%x [%i]\n", priv->sndinfo2.bSampleSize, priv->sndinfo2.bSampleSize );
		    xmm_logging( 1, "VOC! INFO: Sound Data(New): bChannels = 0x%x [%i]\n", priv->sndinfo2.bChannels, priv->sndinfo2.bChannels );
		    xmm_logging( 1, "VOC! INFO: Sound Data(New): bUnknown1 = 0x%x [%i]\n", priv->sndinfo2.bUnknown1, priv->sndinfo2.bUnknown1 );
#endif

		    priv->data_offset = priv->pIO->Tell( priv->pIO );
		    priv->data_size = size;
		    priv->data_type = type;

		    if( priv->seekable )	priv->pIO->Seek( priv->pIO, priv->data_size, XMM_SEEK_CUR );
		    else	loop = 0;

		    xmm_logging( 2, "VOC! data_offset = %li (%lx) data_size = %li (%lx)\n", priv->data_offset, priv->data_offset, priv->data_size, priv->data_size );
		    break;

	    default:
		xmm_logging( 2, "VOC! Ignoring unknown data block type (%x)\n", type );
		SEEK( priv->pIO, size, priv->seekable );
		break;
	}	
  }

  if(( priv->data_type != VOC_TYPE_SOUND_DATA ) && ( priv->data_type != VOC_TYPE_SOUND_DATA_NEW ))
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(VOC) Sound data block (%x) unknown or not present.", priv->data_type );

  /* Seek to the beginning of data */
  if( priv->seekable )	priv->pIO->Seek( priv->pIO, priv->data_offset, XMM_SEEK_SET );

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* AudioInfo struct */
  if( priv->data_type == VOC_TYPE_SOUND_DATA )
  {
    priv->ai.fmt.format = XMM_AUDIO_FMT_U8;
    priv->ai.fmt.samprate = VOC_SRB2RATE(priv->sndinfo.sr);
    priv->ai.fmt.channels = 1;
  }
  else
  {
    priv->ai.fmt.format = XMM_AUDIO_CODEC_PCM | priv->sndinfo2.bSampleSize;
    if( priv->sndinfo2.bSampleSize != 8 )
	priv->ai.fmt.format |= XMM_AUDIO_MASK_SIGNED;
    priv->ai.fmt.samprate = priv->sndinfo2.dwSampleRate;
    priv->ai.fmt.channels = priv->sndinfo2.bChannels;
  }

  priv->ai.fmt.bitrate = priv->ai.fmt.channels * priv->ai.fmt.samprate *
			( priv->ai.fmt.format & XMM_AUDIO_MASK_SIZE );
  priv->ai.fmt.blockSize = priv->ai.fmt.channels * ( priv->ai.fmt.format & XMM_AUDIO_MASK_SIZE ) / 8;
  priv->ai.fmt.extraSize = 0;
  priv->ai.fmt.desc[0] = '\0';

  priv->ai.tSamples = (uint32_t)(((double)priv->data_size / ( priv->ai.fmt.bitrate / 8 )) * priv->ai.fmt.samprate );
  priv->ai.tSize = priv->data_size;
  priv->ai.offset = 0;

  /* Byte rate */
  priv->byterate = priv->ai.fmt.samprate * priv->ai.fmt.blockSize;

  /* Clip info */
  strncpy( priv->ci.name, priv->filename, XMM_CIL_NAME - 1 );
  priv->ci.name[XMM_CIL_NAME - 1] = '\0';
  priv->ci.author[0] = '\0';
  priv->ci.album[0] = '\0';
  priv->ci.content[0] = '\0';
  priv->ci.copyright[0] = '\0';
  priv->ci.software[0] = '\0';
  priv->ci.date[0] = '\0';
  priv->ci.comment[0] = '\0';
  priv->ci.size = priv->pIO->Size( priv->pIO );
  priv->ci.playtime = (double)priv->ai.tSamples / priv->ai.fmt.samprate;

  return XMM_RET_OK;
}

/*
 * Close
 */
static int voc_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  if( priv->filename )	free( priv->filename );
  if( priv->pIO )	priv->pIO->Close( priv->pIO );
  free( input );

  return XMM_RET_OK;
}

/*
 * Control
 */
static int voc_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_AudioFormat	*format;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_AFORMAT:			/* Query sound format */
		format = (XMM_AudioFormat *)data;
		if(( format->format == priv->ai.fmt.format ) &&
		    ( format->samprate == priv->ai.fmt.samprate ) &&
		    ( format->channels == priv->ai.fmt.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_AFORMAT:			/* Get sound format */
		memcpy( data, &priv->ai.fmt, sizeof( XMM_AudioFormat ));
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->ai.fmt;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_CAPS:				/* Get capabilities */
		*((uint32_t *) data ) = XMM_INPUT_CF_MODE_READ | XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(VOC) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(VOC) cmd ( 0x%x )" );
}

/*
 * Seek to position
 */
static int voc_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  struct priv_t	*priv = input->sys.priv;

  return input->AudioSeek( input, astream, (uint32_t)( seek * priv->ai.tSize / priv->byterate * 1000 ));
}

/*
 * Get Information
 */
static int voc_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )	memcpy( ci, &priv->ci, sizeof( XMM_ClipInfo ));

  return XMM_RET_OK;
}

/*
 * Get audio stream number
 */
static int voc_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int voc_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int voc_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  struct priv_t *priv = input->sys.priv;
  uint32_t	read;

  if(( priv->pIO->Tell( priv->pIO ) + size ) >= ( priv->data_offset + priv->data_size ))
  {
    size = priv->data_offset + priv->data_size - priv->pIO->Tell( priv->pIO );
    if( size <= 0 )	return XMM_RET_EOS;
  }

  read = priv->pIO->Read( priv->pIO, buffer, 1, size );
  if(( read == 0 ) && priv->pIO->Tell( priv->pIO ))	return XMM_RET_EOS;

  return read;
}

/*
 * Audio timestamp.
 */
static int voc_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  struct priv_t	*priv = input->sys.priv;

  if( pts )	*pts = (uint32_t)((double)( priv->pIO->Tell( priv->pIO ) -
				priv->data_offset ) / priv->byterate * 1000 );

  return XMM_RET_OK;
}

/*
 * Seek to position in audio stream
 */
static int voc_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts )
{
  struct priv_t	*priv = input->sys.priv;
  double	bytes;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(VOC) Stream is not seekable." );

  bytes = (double)pts * priv->byterate / 1000;

  /* I/O Seek */
  return priv->pIO->Seek( priv->pIO, priv->data_offset + (uint32_t)bytes, XMM_SEEK_SET );
}

/*
 * AU files do not contain video data
 */
static int voc_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int voc_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int voc_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  return XMM_RET_NOTSUPPORTED;
}

static int voc_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int voc_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if VOC file
 */
static int voc_Check( void *xmm, char *filename )
{
  uint8_t		pcID[19];
  XMM_PluginIO		*pIO;

  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  pIO->Read( pIO, pcID, strlen( VOC_FORMAT_ID ), 1 );
  pIO->Close( pIO );

  if( !strncmp( pcID, VOC_FORMAT_ID, strlen( VOC_FORMAT_ID )))	return 1;
  return 0;
}

/*
 * Get file info
 */
static int voc_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */
XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"VOC",
				"Input: VOC",
				"Copyright (c) 2002 Arthur Kleer",
				NULL, NULL },
				voc_Init, voc_Open, voc_Close,
				voc_Control, voc_Seek, voc_Info, 
				voc_AudioStreams, voc_AudioInfo,
				voc_AudioRead, voc_AudioPTS, voc_AudioSeek,
				voc_VideoStreams, voc_VideoInfo,
				voc_VideoRead, voc_VideoPTS, voc_VideoSeek,
				voc_Check, voc_FileInfo };



/*
 * Misc functions for file i/o
 */

static uint32_t READn( XMM_PluginIO *pIO, void *data, uint32_t size )
{
  return pIO->Read( pIO, data, size, 1 );
}

static uint32_t READ32( XMM_PluginIO *pIO )
{
  uint32_t tmp;

  pIO->Read( pIO, &tmp, 4, 1 );

  return XMM_INT32_LE( tmp );
}

static uint16_t READ16( XMM_PluginIO *pIO )
{
  uint16_t tmp;

  pIO->Read( pIO, &tmp, 2, 1 );

  return XMM_INT16_LE( tmp );
}

static uint8_t READ8( XMM_PluginIO *pIO )
{
  uint8_t tmp;

  pIO->Read( pIO, &tmp, 1, 1 );

  return tmp;
}

static uint32_t READ_IEEE_EXT( XMM_PluginIO *pIO )
{
  uint32_t	tmp;
  uint8_t	buffer[10];

  pIO->Read( pIO, buffer, 1, 10 );

  /* 1 bit: sign, 15 bit: exponent, 64 bit: mantissa */
  tmp = ( buffer[2] << 24 ) | ( buffer[3] << 16 ) | ( buffer[4] << 8 ) | buffer[0];
  tmp >>= ( 30 - buffer[1] );
	
  return tmp;
}

static int SEEK( XMM_PluginIO *pIO, long offset, int seekable )
{
  if( seekable )	return pIO->Seek( pIO, offset, XMM_SEEK_CUR );
  else
  {
	uint8_t tmp;
	int i;

	for( i = 0; i < offset; i++ )	pIO->Read( pIO, &tmp, 1, 1 );
	return i;
  }
}
