/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wave.c
 * WAVE file format DeMUX
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include "wave.h"

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#define	VERBOSE
#endif
 
#define alignUP( a,b )	((((a) + (b) - 1 ) / (b)) * (b))

/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_AudioInfo		ai;
    XMM_AudioFormat		*af;

    /* Clip info */
    XMM_ClipInfo		ci;

    char 			*filename;
    int				seekable;
    uint32_t			data_offset, data_size;
    int				byterate;
};

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Prototypes
 */

static uint32_t	READn( XMM_PluginIO *wave, void *data, uint32_t size );
static uint32_t	READ32( XMM_PluginIO *wave );
static uint16_t	READ16( XMM_PluginIO *wave );
static uint8_t	READ8( XMM_PluginIO *wave );
static int	SEEK( XMM_PluginIO *wave, long offset, int seekable );

static int	Read_fmt( XMM_PluginIO *wave, WAVEFORMATEX *wave_fmt_head, uint32_t size, unsigned char **wave_fmt_extra );
static int	Read_INFO_chunk( XMM_PluginIO *pIO, XMM_ClipInfo *ci, uint32_t dwName, uint32_t dwSize );

static int	wave_AudioSeek( XMM_PluginInput *input, int stream, uint32_t sample );
uint32_t	codec_ms2xmm( uint16_t ms_codec );

/*
 * Initialize
 */
static XMM_PluginInput *wave_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(WAVE) plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int wave_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  uint32_t		dwSize, dwName, dwTemp;
  char			*buffer, *ptr;
  int			loop = 1;
  WAVEFORMATEX		wave_fmt_head;
  unsigned char		*wave_fmt_extra;

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(WAVE) Only 'read' mode supported" );

  /* Open file */
  priv->pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->pIO == NULL )	return XMM_RET_ERROR;

  /* Seekable ? */
  if( priv->pIO->Seek( priv->pIO, 0, XMM_SEEK_CUR ) == XMM_RET_OK )
	priv->seekable = 1;
  else	priv->seekable = 0;
#ifdef VERBOSE
  xmm_logging( 2, "WAVE! Stream is %sseekable\n", priv->seekable ? "" : "NOT " );
#endif

  /* Clip info */
  memset( &priv->ci, 0, sizeof( XMM_ClipInfo ));

  /* Init data */
  priv->data_offset = 0;
  priv->data_size = 0;

  /* Parse file */
  while( loop )
  {
	dwName = READ32( priv->pIO );
	if( priv->pIO->Eof( priv->pIO ))	break;

#ifdef VERBOSE
	xmm_logging( 2, "WAVE! Chunk ID = '%s' ( %lx )\n", xmm_FOURCC_string( dwName ), dwName );
#endif

	/*
	 * Handle list chunks
	 */
	switch( dwName )
	{
		case RIFF_ID_INFO:
			dwName = 0;
			break;

		/* Non-list chunk: read size */
		default:
			dwSize = READ32( priv->pIO );
#ifdef VERBOSE
			xmm_logging( 2, "WAVE! ID = %x [%s] Chunk Size = %li ( %lx )\n", dwName, xmm_FOURCC_string(dwName), dwSize, dwSize );
#endif
			break;
	}

	if( dwName == 0 )	continue;
	
	switch( dwName )
	{
		case RIFF_ID_RIFF:
			dwTemp = READ32( priv->pIO );
			if( dwTemp != RIFF_ID_WAVE )
			{
				priv->pIO->Close( priv->pIO );
				return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(WAVE) No WAVE RIFF form ( '%s' )", xmm_FOURCC_string( dwTemp ));
			}
			xmm_logging( 2, "WAVE! RIFF Type = '%s'\n", xmm_FOURCC_string( dwTemp ));
			break;

		case RIFF_ID_LIST:
			break;

		case RIFF_ID_fmt:
			Read_fmt( priv->pIO, &wave_fmt_head, dwSize, &wave_fmt_extra );
			break;

		case RIFF_ID_data:
			priv->data_offset = priv->pIO->Tell( priv->pIO );
			priv->data_size = dwSize;
			xmm_logging( 2, "WAVE! data: offset = %li ( %lx ) size = %li ( %lx )\n", priv->data_offset, priv->data_offset, priv->data_size, priv->data_size );
			if( dwSize & 1 )	dwSize++;
			if( priv->seekable )	SEEK( priv->pIO, dwSize, priv->seekable );
			else	loop = 0;
			break;

		/* INFO chunks */
		case RIFF_ID_INAM:
		case RIFF_ID_ISBJ:
		case RIFF_ID_IART:
		case RIFF_ID_ICOP:
		case RIFF_ID_ISFT:
		case RIFF_ID_ICMT:
			Read_INFO_chunk( priv->pIO, &priv->ci, dwName, dwSize );
			break;

		default:
			xmm_logging( 2, "WAVE! Ignoring chunk '%s' (%lx)\n", xmm_FOURCC_string(dwName), dwName );
			if( dwSize & 1 )	dwSize++;
			SEEK( priv->pIO, dwSize, priv->seekable );
			break;
	}
  }

  if( priv->data_offset == 0 )
  {
	priv->pIO->Close( priv->pIO );
	return xmm_SetError( input->sys.xmm, XMM_RET_ERROR, "(WAVE) No data found" );
  }

  /* Seek to the beginning of data */
  if( priv->seekable )	priv->pIO->Seek( priv->pIO, priv->data_offset, XMM_SEEK_SET );

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* Allocate memory */
  if(( priv->af = malloc( sizeof( XMM_AudioFormat ) + wave_fmt_head.cbSize )) == NULL )
	return xmm_SetError( input->sys.xmm, XMM_RET_ALLOC, "(WAVE) XMM_AudioFormat" );

  /* Fill audio format struct */
  priv->af->format = codec_ms2xmm( wave_fmt_head.wFormatTag );
  priv->af->format |= ( wave_fmt_head.wBitsPerSample & XMM_AUDIO_MASK_SIZE );

  if(( wave_fmt_head.wFormatTag == WAVE_FORMAT_IEEE_FLOAT ) ||
	(( wave_fmt_head.wBitsPerSample != 8 )))
		priv->af->format |= XMM_AUDIO_MASK_SIGNED;

  priv->af->samprate = wave_fmt_head.nSamplesPerSec;
  priv->af->channels = wave_fmt_head.nChannels;
  priv->af->bitrate = wave_fmt_head.nAvgBytesPerSec * 8;
  priv->af->blockSize = wave_fmt_head.nBlockAlign;
  priv->af->extraSize = wave_fmt_head.cbSize;
  priv->af->extraType = XMM_AUDIO_EXT_WAVE;
  priv->af->desc[0] = '\0';

  if( priv->af->extraSize )
	memcpy( &priv->af[1], wave_fmt_extra, priv->af->extraSize );

  /* AudioInfo struct */
  memcpy( &priv->ai.fmt, priv->af, sizeof( XMM_AudioFormat ));
  priv->ai.fmt.extraSize = 0;
  priv->ai.tSamples = (uint32_t)(((double)priv->data_size /
			    ( priv->af->bitrate / 8 )) * priv->af->samprate );
  priv->ai.tSize = priv->data_size;
  priv->ai.offset = 0;

  /* Byte rate */
  priv->byterate = priv->ai.fmt.bitrate / 8;

  /* Clip info */
  priv->ci.size = priv->pIO->Size( priv->pIO );
  priv->ci.playtime = (double)priv->ai.tSamples / priv->ai.fmt.samprate;

  if( priv->ci.content[0] == '\0' )
  {
	strncpy( priv->ci.content, priv->filename, XMM_CIL_CONTENT - 1 );
	priv->ci.content[XMM_CIL_CONTENT - 1] = '\0';
  }

  return XMM_RET_OK;
}

/*
 * Close
 */
static int wave_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  if( priv->filename )	free( priv->filename );
  if( priv->pIO )	priv->pIO->Close( priv->pIO );
  if( priv->af )	free( priv->af );
  free( input );

  return XMM_RET_OK;
}

/*
 * Control
 */
static int wave_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_AudioFormat	*format;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_AFORMAT:			/* Query sound format */
		format = (XMM_AudioFormat *)data;
		if(( format->format == priv->ai.fmt.format ) &&
		    ( format->samprate == priv->ai.fmt.samprate ) &&
		    ( format->channels == priv->ai.fmt.channels ))
			return XMM_CTLRET_TRUE;

		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_AFORMAT:			/* Get sound format */
		memcpy( data, priv->af, sizeof( XMM_AudioFormat ));
		((XMM_AudioFormat *)data)->extraSize = 0;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = priv->af;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_CAPS:				/* Get capabilities */
		*((uint32_t *) data ) = XMM_INPUT_CF_MODE_READ |
					XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_RET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(WAVE) cmd = 0x%x" );

  return xmm_SetError( input->sys.xmm, XMM_RET_INVALID_ARG, "(WAVE) cmd ( 0x%x )" );
}

/*
 * Seek to position
 */
static int wave_Seek( XMM_PluginInput *input, int vstream, int astream, double seek )
{
  struct priv_t	*priv = input->sys.priv;

  return input->AudioSeek( input, astream, (uint32_t)( seek * priv->ai.tSize / priv->byterate * 1000 ));
}

/*
 * Get Information
 */
static int wave_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )	memcpy( ci, &priv->ci, sizeof( XMM_ClipInfo ));

  return XMM_RET_OK;
}

/*
 * Get audio stream number
 */
static int wave_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int wave_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int wave_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags )
{
  struct priv_t *priv = input->sys.priv;
  uint32_t	read;

  if(( priv->pIO->Tell( priv->pIO ) + size ) >= ( priv->data_offset + priv->data_size ))
  {
    size = priv->data_offset + priv->data_size - priv->pIO->Tell( priv->pIO );
    if( size <= 0 )	return XMM_RET_EOS;
  }

  read = priv->pIO->Read( priv->pIO, buffer, 1, size );
  if(( read == 0 ) && priv->pIO->Eof( priv->pIO ))	return XMM_RET_EOS;

  return read;
}

/*
 * Audio timestamp.
 */
static int wave_AudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts )
{
  struct priv_t	*priv = input->sys.priv;

  if( pts )	*pts = (uint32_t)((double)( priv->pIO->Tell( priv->pIO ) -
				priv->data_offset ) / priv->byterate * 1000 );

  return XMM_RET_OK;
}

/*
 * Seek to position in audio stream
 */
static int wave_AudioSeek( XMM_PluginInput *input, int stream, uint32_t pts )
{
  struct priv_t	*priv = input->sys.priv;
  double	bytes;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_RET_NOTSUPPORTED, "(WAVE) Stream is not seekable." );

  bytes = (double)pts * priv->byterate / 1000;

  /* I/O Seek */
  return priv->pIO->Seek( priv->pIO, priv->data_offset + (uint32_t)bytes, XMM_SEEK_SET );
}

/*
 * WAVE files do not contain video data
 */
static int wave_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int wave_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int wave_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags )
{
  return XMM_RET_NOTSUPPORTED;
}

static int wave_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int wave_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if WAV file
 */
static int wave_Check( void *xmm, char *filename )
{
  uint32_t		dwSize, dwName, dwTemp;
  XMM_PluginIO		*pIO;

  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  dwName = READ32( pIO );
  dwSize = READ32( pIO );
  dwTemp = READ32( pIO );

  pIO->Close( pIO );

  if(( dwName == RIFF_ID_RIFF ) && ( dwTemp == RIFF_ID_WAVE ))	return 1;
  else	return 0;
}

/*
 * Get file info
 */
static int wave_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */
XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"WAVE",
				"Input: WAVE",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				wave_Init, wave_Open, wave_Close,
				wave_Control, wave_Seek, wave_Info, 
				wave_AudioStreams, wave_AudioInfo,
				wave_AudioRead, wave_AudioPTS, wave_AudioSeek,
				wave_VideoStreams, wave_VideoInfo,
				wave_VideoRead, wave_VideoPTS, wave_VideoSeek,
				wave_Check, wave_FileInfo };

/*
 * Internal code
 */

/*
 * Misc functions for file i/o
 */

static uint32_t READn( XMM_PluginIO *pIO, void *data, uint32_t size )
{
  return pIO->Read( pIO, data, size, 1 );
}

static uint32_t READ32( XMM_PluginIO *pIO )
{
  uint32_t tmp;

  pIO->Read( pIO, &tmp, 4, 1 );

  return tmp;
}

static uint16_t READ16( XMM_PluginIO *pIO )
{
  uint16_t tmp;

  pIO->Read( pIO, &tmp, 2, 1 );

  return tmp;
}

static uint8_t READ8( XMM_PluginIO *pIO )
{
  uint8_t tmp;

  pIO->Read( pIO, &tmp, 1, 1 );

  return tmp;
}

static int SEEK( XMM_PluginIO *pIO, long offset, int seekable )
{
  if( seekable )	return pIO->Seek( pIO, offset, XMM_SEEK_CUR );
  else
  {
	uint8_t tmp;
	int i;

	for( i = 0; i < offset; i++ )	pIO->Read( pIO, &tmp, 1, 1 );
	return i;
  }
}

/*
 * Read WAVE format header
 */
static int Read_fmt( XMM_PluginIO *pIO, WAVEFORMATEX *wave_fmt_head, uint32_t size, unsigned char **wave_fmt_extra )
{
  if( size < 0x10 )
  {
	xmm_logging( 1, "WAVE! fmt: FILE ERROR: Size < 16 bytes ( is %li )\n", size );
	return 0;
  }

  wave_fmt_head->wFormatTag		= READ16( pIO );
  wave_fmt_head->nChannels		= READ16( pIO );
  wave_fmt_head->nSamplesPerSec		= READ32( pIO );
  wave_fmt_head->nAvgBytesPerSec	= READ32( pIO );
  wave_fmt_head->nBlockAlign		= READ16( pIO );
  wave_fmt_head->wBitsPerSample		= READ16( pIO );
  wave_fmt_head->cbSize			= 0;

  if( wave_fmt_head->wBitsPerSample == 0 )
	wave_fmt_head->wBitsPerSample = 0x10;

  if( wave_fmt_head->wFormatTag != 0x01 )
  {
	if( size < 0x12 )
	{
		xmm_logging( 1, "WAVE! fmt: FILE ERROR: Format not 0x01 and no extradata found\n" );
		return 0;
	}

	wave_fmt_head->cbSize		= READ16( pIO );

	if( size > ( 0x12 + wave_fmt_head->cbSize ))
		xmm_logging( 1, "WAVE! fmt: WARNING: Extra size < chunk size - 18 ( fixing, diff = %li bytes )\n", size - ( 0x12 + wave_fmt_head->cbSize ));

	size -= 2;

	*wave_fmt_extra = malloc( size - 16 );
	if( *wave_fmt_extra == NULL )
	{
	    xmm_logging( 1, "WAVE! fmt: ERROR allocating memory for extra data ( %li bytes )\n", size - 16 );
	    return 0;
	}
  
	READn( pIO, *wave_fmt_extra, size - 16 );
  }
  else
  {
	if( size != 16 )
	{
		xmm_logging( 1, "WAVE! fmt: ERROR: Header size for PCM format not 16 ( is %li )\n", size );
		return 0;
	}
  }

#ifdef VERBOSE
  xmm_logging( 2, "WAVE! wfx: wFormatTag = 0x%x\n", wave_fmt_head->wFormatTag );
  xmm_logging( 2, "WAVE! wfx: nChannels = %i\n", wave_fmt_head->nChannels );
  xmm_logging( 2, "WAVE! wfx: nSamplesPerSec = %li\n", wave_fmt_head->nSamplesPerSec );
  xmm_logging( 2, "WAVE! wfx: nAvgBytesPerSec = %li\n", wave_fmt_head->nAvgBytesPerSec );
  xmm_logging( 2, "WAVE! wfx: nBlockAlign = %i\n", wave_fmt_head->nBlockAlign );
  xmm_logging( 2, "WAVE! wfx: wBitsPerSample = %i\n", wave_fmt_head->wBitsPerSample );
  xmm_logging( 2, "WAVE! wfx: cbSize = %i\n", wave_fmt_head->cbSize );
#endif

  return 1;
}

/*
 * Read INFO chunk
 */
static int Read_INFO_chunk( XMM_PluginIO *pIO, XMM_ClipInfo *ci, uint32_t dwName, uint32_t dwSize )
{
  char	info_buffer[4096];

  pIO->Read( pIO, info_buffer, dwSize, 1 );

   switch( dwName )
   {
	case RIFF_ID_INAM:
		strncpy( ci->name, info_buffer, XMM_CIL_NAME - 1 );
		ci->name[XMM_CIL_NAME - 1] = '\0';
		break;

	case RIFF_ID_ISBJ:
		strncpy( ci->content, info_buffer, XMM_CIL_CONTENT - 1 );
		ci->content[XMM_CIL_CONTENT - 1] = '\0';
		break;

	case RIFF_ID_IART:
		strncpy( ci->author, info_buffer, XMM_CIL_AUTHOR - 1 );
		ci->author[XMM_CIL_AUTHOR - 1] = '\0';
		break;

	case RIFF_ID_ICOP:
		strncpy( ci->copyright, info_buffer, XMM_CIL_COPYRIGHT - 1 );
		ci->copyright[XMM_CIL_COPYRIGHT - 1] = '\0';
		break;

	case RIFF_ID_ISFT:
		strncpy( ci->software, info_buffer, XMM_CIL_SOFTWARE - 1 );
		ci->software[XMM_CIL_SOFTWARE - 1] = '\0';
		break;

	case RIFF_ID_ICMT:
		strncpy( ci->comment, info_buffer, XMM_CIL_COMMENT - 1 );
		ci->comment[XMM_CIL_COMMENT - 1] = '\0';
		break;

	case RIFF_ID_ICRD:
		strncpy( ci->date + 4, info_buffer, 4 );
		strncpy( ci->date + 2, info_buffer + 5, 2 );
		strncpy( ci->date, info_buffer + 8, 2 );
		ci->date[XMM_CIL_DATE - 1] = '\0';
		break;

	default:
		xmm_logging( 2, "AVI! Ignoring INFO chunk 0x%x [%s] [%li bytes]\n", dwName, xmm_FOURCC_string(dwName), dwSize );
		break;
  }

  if( dwSize & 1 )	READ8( pIO );
  return 1;
}

/*
 * Misc
 */

uint32_t codec_ms2xmm( uint16_t ms_codec )
{
  int		xmm_fmt_idx;
  struct format_conv
  {
	int	xmm;
	int	ms;
  } format_table[] =
    {
	{ XMM_AUDIO_CODEC_PCM, WAVE_FORMAT_PCM },
	{ XMM_AUDIO_CODEC_IEEE, WAVE_FORMAT_IEEE_FLOAT },
	{ XMM_AUDIO_CODEC_ULAW, WAVE_FORMAT_MULAW },
	{ XMM_AUDIO_CODEC_ALAW, WAVE_FORMAT_ALAW },
	{ XMM_AUDIO_CODEC_MPEG, WAVE_FORMAT_MPEGLAYER3 },
	{ XMM_AUDIO_CODEC_AC3, WAVE_FORMAT_AC3 },
	{ XMM_AUDIO_CODEC_GSM610MS, WAVE_FORMAT_GSM610 },
	{ XMM_AUDIO_CODEC_G721, WAVE_FORMAT_G721_ADPCM },
	{ XMM_AUDIO_CODEC_G723, WAVE_FORMAT_G723_ADPCM },
	{ XMM_AUDIO_CODEC_G726, WAVE_FORMAT_G726_ADPCM },
	{ XMM_AUDIO_CODEC_WMA1, WAVE_FORMAT_WMA1 },
	{ XMM_AUDIO_CODEC_WMA2, WAVE_FORMAT_WMA2 },
	{ XMM_AUDIO_CODEC_ADPCM_MS, WAVE_FORMAT_ADPCM },
	{ XMM_AUDIO_CODEC_ADPCM_IMA, WAVE_FORMAT_DVI_ADPCM },
	{ XMM_AUDIO_CODEC_VOX_META, 0x75 },
	{ 0, 0 }
    };

  /* Check format */
  for( xmm_fmt_idx = 0; format_table[xmm_fmt_idx].xmm; xmm_fmt_idx++ )
	if( ms_codec == format_table[xmm_fmt_idx].ms )	break;

  return format_table[xmm_fmt_idx].xmm;
}
