/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * vcd.c
 * VCD ( VideoCD ) I/O
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>

#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>

#include <libxmm/lpio.h>
#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>

/*
 * Definitions
 */

#define	DEFAULT_DEVICE		"/dev/cdrom"

/*
 * Types
 */

typedef struct cdrom_frame_raw_s
{
    char		sync[CD_SYNC_SIZE];		/* sync ( 12 bytes ) */
    char		header[CD_HEAD_SIZE];		/* header (address) */
    char		subheader[CD_SUBHEAD_SIZE];	/* subheader */

#define	CD_FRAMESIZE_VIDEO	2324

    char		data[CD_FRAMESIZE_VIDEO];	/* data */
    char		unknown[4];			/* EDC ??? */
} cdrom_frame_raw_t;

struct priv_t
{
    int				fd;

    struct cdrom_tochdr		th;

    int				cTrackIDX;
    int				fTrackIDX;
    int				lTrackIDX;

    cdrom_frame_raw_t		vcd_frame;
    uint32_t			vcd_frame_data_pos;

    uint32_t			cFrame;

    struct
    {
	uint32_t		lba;
    } aTrack[100];
};

/*
 * Global data
 */

extern XMM_PluginIO	plugin_info;

/*
 * Open file
 */
static XMM_PluginIO *vcd_Open( void *xmm, const char *url, int mode )
{
  XMM_PluginIO			*io;
  struct priv_t			*priv;
  struct cdrom_tocentry		te;
  char 				device[256] = DEFAULT_DEVICE, *ptr;
  int				cTrack = -1, i;

  /* Check filename */
  if( strncmp( url, "vcd://", 6 ))	return NULL;
  else	url += 6;

  /* Parse device/track string */
  if( isdigit( *url ))	cTrack = atoi( url );
  else if( *url == '/' )
  {
	ptr = strchr( url, ':' );
	if( ptr )
	{
	    *ptr++ = '\0';
	    cTrack = atoi( ptr );
	}

	strcpy( device, url );
  }

  /* Check mode */
  if( mode != XMM_IO_READ )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, "(VCD) Wrong mode ( %i )", mode );
	return NULL;
  }

  /* Initialize plugin */
  if(( io = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginIO ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(VCD) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &io[1];
  io->sys.priv = (void *) priv;
  io->sys.xmm = xmm;

  xmm_logging( 2, "VCD! Using device %s\n", device );

  /* Open device */
  if(( priv->fd = open( device, O_RDONLY )) < 0 )
  {
	free( io );
	xmm_SetError( xmm, XMM_RET_ERROR, "(VCD) Unable to open device '%s': %s", device, strerror( errno ));
	return NULL;
  }

  /* Read table of content */
  if( ioctl( priv->fd, CDROMREADTOCHDR, &priv->th ))
  {
	close( priv->fd );
	xmm_SetError( xmm, XMM_RET_ERROR, "(VCD) ioctl ( CDROMREADTOCHDR ) failed" );
	return NULL;
  }

  /* Set play range */
  if( cTrack == -1 )
  {
	priv->fTrackIDX = priv->th.cdth_trk0;
	priv->lTrackIDX = priv->th.cdth_trk1;
	cTrack = priv->th.cdth_trk0;
  }
  else
  {
	if(( cTrack < priv->th.cdth_trk0 ) || ( cTrack > ( priv->th.cdth_trk1 - 1 )))
	{
	    xmm_SetError( xmm, XMM_RET_ERROR, "(VCD) Invalid track number ( %i )", cTrack );
	    return NULL;
	}

	priv->fTrackIDX = priv->th.cdth_trk0 + cTrack - 1;
	priv->lTrackIDX = priv->th.cdth_trk0 + cTrack;
	cTrack = priv->th.cdth_trk0 + cTrack - 1;
  }

  for( i = priv->th.cdth_trk0; i <= priv->th.cdth_trk1; i++ )
  {
	te.cdte_track = i;
	te.cdte_format = CDROM_MSF;
	if( ioctl( priv->fd, CDROMREADTOCENTRY, &te ))
	{
	    close( priv->fd );
	    xmm_SetError( xmm, XMM_RET_ERROR, "(VCD) ioctl ( CDROMREADTOCENTRY ) failed" );
	    return NULL;
	}

	priv->aTrack[ i - priv->th.cdth_trk0 ].lba =
			((( te.cdte_addr.msf.minute * 60 ) +
			te.cdte_addr.msf.second ) * 75 ) +
			te.cdte_addr.msf.frame;

	xmm_logging( 3, "VCD! TE(%2i): track = %i\n", i, te.cdte_track );
	xmm_logging( 3, "VCD! TE(%2i): format = %i\n", i, te.cdte_format );
	xmm_logging( 3, "VCD! TE(%2i): adr = %i\n", i, te.cdte_adr );
	xmm_logging( 3, "VCD! TE(%2i): ctrl = %i\n", i, te.cdte_ctrl );
	xmm_logging( 3, "VCD! TE(%2i): addr = lba %i ( msf %i:%i:%i = lba %i )\n", i, te.cdte_addr.lba, te.cdte_addr.msf.minute, te.cdte_addr.msf.second, te.cdte_addr.msf.frame, (( te.cdte_addr.msf.minute * 60 ) + te.cdte_addr.msf.second ) * 75 + te.cdte_addr.msf.frame );
	xmm_logging( 3, "VCD! TE(%2i): datamode = %i\n", i, te.cdte_datamode );
  }


  te.cdte_track = CDROM_LEADOUT;
  te.cdte_format = CDROM_MSF;
  if( ioctl( priv->fd, CDROMREADTOCENTRY, &te ))
  {
	close( priv->fd );
	xmm_SetError( xmm, XMM_RET_ERROR, "(VCD) ioctl ( CDROMREADTOCENTRY ) failed" );
	return NULL;
  }

  priv->aTrack[ i - priv->th.cdth_trk0 ].lba =
		((( te.cdte_addr.msf.minute * 60 ) +
		te.cdte_addr.msf.second ) * 75 ) +
		te.cdte_addr.msf.frame;

  xmm_logging( 3, "VCD! TE(LO): track = %i\n", te.cdte_track );
  xmm_logging( 3, "VCD! TE(LO): format = %i\n", te.cdte_format );
  xmm_logging( 3, "VCD! TE(LO): adr = %i\n", te.cdte_adr );
  xmm_logging( 3, "VCD! TE(LO): ctrl = %i\n", te.cdte_ctrl );
  xmm_logging( 3, "VCD! TE(LO): addr = lba %i ( msf %i:%i:%i = lba %i )\n", te.cdte_addr.lba, te.cdte_addr.msf.minute, te.cdte_addr.msf.second, te.cdte_addr.msf.frame, (( te.cdte_addr.msf.minute * 60 ) + te.cdte_addr.msf.second ) * 75 + te.cdte_addr.msf.frame );
  xmm_logging( 3, "VCD! TE(LO): datamode = %i\n", te.cdte_datamode );

  /**/
  priv->vcd_frame_data_pos = CD_FRAMESIZE_VIDEO;
  priv->cFrame = 0;
  priv->cTrackIDX = cTrack;

  return io;
}

/*
 * Close
 */
static int vcd_Close( XMM_PluginIO *io )
{
  struct priv_t	*priv = io->sys.priv;
  int ret;

  ret = ( close( priv->fd ) == 0 ) ? XMM_RET_OK : XMM_RET_ERROR;
  free( io );

  return ret;
}

/*
 * Read
 */
static int vcd_Read( XMM_PluginIO *io, void *ptr, int size, int nmemb )
{
  struct priv_t		*priv = io->sys.priv;
  struct cdrom_msf	*msf;
  uint32_t		read, done, copy, offset;

  if(( size == 0 ) || ( nmemb == 0 ))	return 0;

  /**/
  read = nmemb * size;
  done = 0;

  /**/
  if( priv->vcd_frame_data_pos < CD_FRAMESIZE_VIDEO )
  {
	copy = ( CD_FRAMESIZE_VIDEO - priv->vcd_frame_data_pos );
	if( copy > read )	copy = read;

	memcpy( ptr, priv->vcd_frame.data + priv->vcd_frame_data_pos, copy );

	ptr += copy;
	read -= copy;
	done += copy;
	priv->vcd_frame_data_pos += copy;
  }

  while( read > 0 )
  {
	/* Check for EOF */
	if( io->Eof( io ))	break;

	msf = (struct cdrom_msf *) &priv->vcd_frame;

	/* Read RAW data */
	offset = priv->aTrack[ priv->cTrackIDX ].lba + priv->cFrame;
	msf->cdmsf_min0 = offset / ( CD_SECS * CD_FRAMES );
	msf->cdmsf_sec0 = ( offset / CD_FRAMES ) % CD_SECS; 
	msf->cdmsf_frame0 = offset % CD_FRAMES;

	if( ioctl( priv->fd, CDROMREADRAW, msf ))
	{
	    xmm_logging( 1, "VCD! ioctl ( CDROMREADRAW ) failed at %i:%i.%i\n", msf->cdmsf_min0, msf->cdmsf_sec0, msf->cdmsf_frame0 );
	    return XMM_RET_ERROR;
	}

	/* Next frame */
	priv->cFrame++;

	copy = CD_FRAMESIZE_VIDEO;
	if( read < copy )	copy = read;

	priv->vcd_frame_data_pos = copy;
	memcpy( ptr, priv->vcd_frame.data, copy );

	ptr += copy;
	read -= copy;
	done += copy;
  }

  return done;
}

/*
 * Write
 */
static int vcd_Write( XMM_PluginIO *io, void *ptr, int size, int nmemb )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Tell
 */
static long vcd_Tell( XMM_PluginIO *io )
{
  struct priv_t		*priv = io->sys.priv;

  return priv->cFrame * CD_FRAMESIZE_VIDEO - ( CD_FRAMESIZE_VIDEO - priv->vcd_frame_data_pos );
}

/*
 * Seek
 */ 
static int vcd_Seek( XMM_PluginIO *io, long offset, int whence )
{
  struct priv_t	*priv = io->sys.priv;
  uint32_t	pos;
  uint8_t	vcd_frame[CD_FRAMESIZE_RAW];
  

  /* Get seek base */
  switch( whence )
  {
	case XMM_SEEK_SET:
		pos = 0;
		break;

	case XMM_SEEK_CUR:
		pos = io->Tell( io );
		break;

	case XMM_SEEK_END:
		pos = io->Size( io );
		break;

	default:
		return xmm_SetError( io->sys.xmm, XMM_RET_NOTSUPPORTED, "(VCD) Unknown seek mode (%i)", whence );
  }

  /* Add offset */
  pos += offset;

  /* Calculate new frame and offset */
  priv->cFrame = pos / CD_FRAMESIZE_VIDEO;

  if( pos % CD_FRAMESIZE_VIDEO )
  {
	io->Read( io, vcd_frame, pos % CD_FRAMESIZE_VIDEO, 1 );
  }
  else	priv->vcd_frame_data_pos = CD_FRAMESIZE_VIDEO;

  return XMM_RET_OK;
}

/*
 * End of file check
 */
static int vcd_Eof( XMM_PluginIO *io )
{
  struct priv_t	*priv = io->sys.priv;

  if(( priv->aTrack[ priv->cTrackIDX ].lba + priv->cFrame ) >=
		priv->aTrack[ priv->lTrackIDX ].lba )	return 1;

  return 0;
}

/*
 * File size
 */
static long vcd_Size( XMM_PluginIO *io )
{
  struct priv_t	*priv = io->sys.priv;

  return ( priv->aTrack[ priv->lTrackIDX - priv->th.cdth_trk0 + 1 ].lba -
		priv->aTrack[ priv->fTrackIDX ].lba ) * CD_FRAMESIZE_VIDEO;
}

/*
 * Plugin data
 */
XMM_PluginIO	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_IO,
				0,
				XMM_VERSION_NUM,
				"",
				"VCD",
				"I/O: VCD",
				"Copyright (c) 2001 Arthur Kleer",
				NULL, NULL },
				vcd_Open, vcd_Close,
				vcd_Read, vcd_Write,
				vcd_Tell, vcd_Seek,
				vcd_Eof, vcd_Size };
