/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * xmmplay.c
 * Very simple example of how to use the LinuX MultiMedia Library
 * However, it is complex enough for a command line player.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <libxmm/libxmm.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/system.h>
#include <libxmmplay/xmmplay.h>

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/*
 * Global data
 */

/*
 * Prototypes
 */

static int EventCB( void *priv, XMM_Event *event );

/*
 * Main()
 */
int main( int argc, char *argv[] )
{
  char			*file = argv[1];
  double		seekval;
  XMM_Play		*xmmplay;
  XMM_FileInfo		info;
  XMM_ClipInfo		ci;
  XMM_ControlScale	scale;
  XMM_Event		event;
  XMM_FilterAudioInfo	*ofai;
  int			i, factor = XMM_GRAPH_RESIZE_ORIG, loop = 1, ret;
  char			*graph_plugin = NULL, *sound_plugin = NULL;
  char			*output_plugin = NULL, *gconv_plugin = NULL;
  const char		*ptr;
  double		seek_value = 0.0;
  int			afoidx = -1;
  uint32_t		flags = 0, ctime;
  int			count = 0;

  printf( "\n * xmmplay %s\n", XMM_VERSION_STRING );
  printf( " * LinuX MultiMedia Project - command line player\n");
  printf( " * www.frozenproductions.com\n\n");

  if( argc < 2 )
  {
	printf( "Usage: xmmplay [options] file\n\n" );
	printf( "Options:\n" );
	printf( "  -g plugin\t\tuse 'plugin' for graph output\n" );
	printf( "  -s plugin\t\tuse 'plugin' for sound output\n" );
	printf( "  -o plugin\t\tuse 'plugin' for (file) output\n" );
	printf( "  -c plugin\t\tuse 'plugin' for graph conversion\n" );
	printf( "  -2       \t\tdouble size mode\n" );
	printf( "  -3       \t\tfullscreen mode\n" );
	printf( "  -audio-fmt id\t\taudio codec ID (for output), use '0' for list\n" );
	printf( "  -audio-dsc\t\taudio direct stream copy\n" );
	printf( "  -audio-off\t\tdon't play audio\n" );
	printf( "  -video-off\t\tdon't play audio\n" );
	printf( "\n" );
	exit(1);
  }

  for( i = 1; i < argc; i++ )
  {
	if( !strcmp( argv[i], "-g" ))	graph_plugin = argv[++i];
	else if( !strcmp( argv[i], "-s" ))	sound_plugin = argv[++i];
	else if( !strcmp( argv[i], "-o" ))	output_plugin = argv[++i];
	else if( !strcmp( argv[i], "-c" ))	gconv_plugin = argv[++i];
	else if( !strcmp( argv[i], "-2" ))	factor = XMM_GRAPH_RESIZE_DOUBLE;
	else if( !strcmp( argv[i], "-3" ))	factor = XMM_GRAPH_RESIZE_FULLSCREEN;
	else if( !strcmp( argv[i], "-audio-fmt" ))	afoidx = strtol( argv[++i], NULL, 0 );
	else if( !strcmp( argv[i], "-audio-dsc" ))	flags |= XMMPLAY_OFLAG_AUDIO_DSC;
	else if( !strcmp( argv[i], "-audio-off" ))	flags |= XMMPLAY_OFLAG_AUDIO_OFF;
	else if( !strcmp( argv[i], "-video-off" ))	flags |= XMMPLAY_OFLAG_VIDEO_OFF;
	else if( !strcmp( argv[i], "-seek" ))	seek_value = strtod( argv[++i], NULL );
	else file = argv[i];
  }

  printf( "Playing '%s'...\n", file );

  if(( xmmplay = xmmplay_Init()) == NULL )
  {
	printf( "\nError initializing xmmplay library: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
  }

  if( output_plugin )
  {
    ptr = xmm_AcOutputPluginName( XMMPLAY_2XMM(xmmplay), output_plugin );
    if( ptr == NULL )
    {
	printf( "\nError setting output plugins: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
    }
    printf( "Using output plugin: %s\n\n", ptr );
  }

  if( graph_plugin )
  {
    ptr = xmm_AcGraphPluginName( XMMPLAY_2XMM(xmmplay), graph_plugin );
    if( ptr == NULL )
    {
	printf( "\nError setting output plugins: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
    }
    printf( "Using graph plugin: %s\n", ptr );
  }

  if( sound_plugin )
  {
    ptr = xmm_AcSoundPluginName( XMMPLAY_2XMM(xmmplay), sound_plugin );
    if( ptr == NULL )
    {
	printf( "\nError setting output plugins: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
    }
    printf( "Using sound plugin: %s\n", ptr );
  }

  if( gconv_plugin )
  {
    ptr = xmm_AcGConvPluginName( XMMPLAY_2XMM(xmmplay), gconv_plugin );
    if( ptr == NULL )
    {
	printf( "\nError setting converter plugins: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
    }
    printf( "Using gconv plugin: %s\n", ptr );
  }

  if( xmm_SetEventCB( XMMPLAY_2XMM(xmmplay), EventCB, (void *)xmmplay, 1 ) < XMM_RET_OK )
  {
	printf( "\nError setting event callback: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
  }

  ofai = xmm_FilterAudioInfo( XMMPLAY_2XMM(xmmplay), (char *)xmm_AcSoundPluginName( XMMPLAY_2XMM(xmmplay), NULL ));
  if( ofai == NULL )	printf( "\nWARNING: Unable to get audio output filter info\n" );
  else
  {
	/* Check for autoselection support */
	if((( ofai->caps & XMM_FILTER_ACF_AUTOSELECT ) == 0 ) && ( afoidx == -1 ))
	{
		printf( "\nThe selected audio output filter does NOT support autoselect.\n" );
		printf( "Select one, using the -afoidx argument.\n\n" );

		for( i = 0; i < ofai->nfmt; i++ )
		    printf( "%.2i. 0x%x\t%i Hz, %i Channel(s), %i kbit/s. %s\n", i, ofai->fmt[i].format, ofai->fmt[i].samprate, ofai->fmt[i].channels, ofai->fmt[i].bitrate / 1000, ofai->fmt[i].desc );

		printf( "\n" );

		free( ofai );
		exit(1);
	}

	xmmplay_SetAudioFormat( xmmplay, ofai, afoidx );
	free( ofai );
  }

  if( xmmplay_Open( xmmplay, file, flags | XMM_INPUT_CF_MODE_DOUT | XMM_INPUT_CF_MODE_READ ) < XMM_RET_OK )
  {
	printf( "\nError opening file %s: %s\n", file, xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	xmmplay_Exit( xmmplay );
	exit(1);
  }

  sleep(1);

  if( factor != XMM_GRAPH_RESIZE_ORIG )
  {
	xmmplay_Scale( xmmplay, 0, 0, factor );
	printf( "Scaling value: %i\n", -factor );
  }

  xmmplay_Seek( xmmplay, seek_value );

  xmmplay_Info( xmmplay, &info, &ci, NULL );

  if( ci.name[0] )	printf( "INFO(clip): name = %s\n", ci.name );
  if( ci.author[0] )	printf( "INFO(clip): author = %s\n", ci.author );
  if( ci.album[0] )	printf( "INFO(clip): album = %s\n", ci.album );
  if( ci.content[0] )	printf( "INFO(clip): content = %s\n", ci.content );
  if( ci.copyright[0] )	printf( "INFO(clip): copyright = %s\n", ci.copyright );
  if( ci.software[0] )	printf( "INFO(clip): software = %s\n", ci.software );
  if( ci.date[0] )	printf( "INFO(clip): date = %s\n", ci.date );
  if( ci.comment[0] )	printf( "INFO(clip): comment = %s\n", ci.comment );
  if( ci.size )		printf( "INFO(clip): size = %i\n", ci.size );
  if( ci.playtime )	printf( "INFO(clip): playtime: %.2i:%.2i:%.2i\n",
					XMMPLAY_TIME_H( ci.playtime ),
					XMMPLAY_TIME_M( ci.playtime ),
					XMMPLAY_TIME_S( ci.playtime ));

  printf( "\n" );

  if( info.vstreams )
  {
	printf( "Video: %ix%i, %.2f FPS, %i kbps, Aspect %.2f:%.2f ( %i Frames, %ib, %i ms )\n",
				info.vi[0].fmt.width, info.vi[0].fmt.height,
				info.vi[0].fmt.framerate, info.vi[0].fmt.bitrate,
				info.vi[0].fmt.aspect_val, info.vi[0].fmt.aspect_div,
				info.vi[0].tFrames, info.vi[0].tSize, info.vi[0].offset );

	printf( "Video: codec %x [%s], %s\n",
				info.vi[0].fmt.codec,
				xmm_FOURCC_string( info.vi[0].fmt.codec ),
				info.vi[0].fmt.desc );
  }

  if( info.astreams )
  {
	printf( "Audio: %i Hz, %i kbps%s, %i Channels ( %i Samples, %ib, %i ms )\n",
				info.ai[0].fmt.samprate,
				abs( info.ai[0].fmt.bitrate ) / 1000,
				info.ai[0].fmt.bitrate < 0 ? " (VBR)" : "",
				info.ai[0].fmt.channels,
				info.ai[0].tSamples, info.ai[0].tSize, info.ai[0].offset );

	printf( "Audio: formatID %x, %s\n",
				info.ai[0].fmt.format,
				info.ai[0].fmt.desc );
  }

  printf( "\n" );

  if( xmmplay_Play( xmmplay ) < XMM_RET_OK )
  {
	printf( "\nError starting playback: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	xmmplay_Close( xmmplay );
	xmmplay_Exit( xmmplay );
	exit(1);
  }

  /* Status can never be XMM_PAUSED, as we do not pause the input */
  while( !( xmmplay_Status( xmmplay ) & XMM_PBS_STOPPED ) && ( loop == 1 ))
  {
	ctime = xmmplay_Info( xmmplay, NULL, NULL, &seekval );
	printf( "%.2i:%.2i:%.2i ( %3.2f%% )  \r", XMMPLAY_PTS_H( ctime ), XMMPLAY_PTS_M( ctime ), XMMPLAY_PTS_S( ctime ), seekval * 100 );
	fflush( stdout );

	/* Sleep some time */
	xmmSYS_usleep( 500000 );

	count++;
	if( count == 10 )
	{
#if 0
		xmm_logging( 1, "xmmplay: Seeking...\n" );
		xmmplay_Seek( xmmplay, 0.55 );
#endif
	}

	/* Check for event */
	if( xmm_PollEvent( XMMPLAY_2XMM(xmmplay), 0, &event ) <= 0 )	continue;

	/* We have an event */
	switch( event.type )
	{
	    case XMM_EVENT_KEY:
		{
		    XMM_Event_Key *key = (XMM_Event_Key *) &event;

#ifdef DEBUG
		    printf( "Key event: scancode %i ASCII %i ( '%c' ) state %i mode = %x\n", key->scode, key->sym, key->sym, key->state, key->mod );
#endif

		    if(( key->state == XMM_KEY_RELEASED ) &&
			    (( key->sym >= '1' ) && ( key->sym <= '4' )))
		    {
			xmm_logging( 1, "xmmplay: Scaling...\n" );

			/* fill XMM_ControlScale structure */
			scale.width = 0;
			scale.height = 0;
			scale.flags = 0;

			if( key->sym == '1' )	scale.flags = XMM_GRAPH_RESIZE_ORIG;
			if( key->sym == '2' )	scale.flags = XMM_GRAPH_RESIZE_DOUBLE;
			if( key->sym == '3' )	scale.flags = XMM_GRAPH_RESIZE_FULLSCREEN;
			if( key->sym == '4' )
			{
			    scale.width = info.vi[0].fmt.width;
			    scale.height = info.vi[0].fmt.height;
			    scale.flags = XMM_GRAPH_RESIZE_FULLSCREEN;
			}

			ret = xmmplay_Scale( xmmplay, scale.width, scale.height, scale.flags );
			if( ret < 0 )
			{
			    fprintf( stderr, "Error resizing output: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
			}
		    }

		    if(( key->state == XMM_KEY_RELEASED ) && ( key->sym == 'q' ))
		    {
			xmm_logging( 1, "xmmplay: Quitting...\n" );
			loop = 0;
		    }

		    if(( key->state == XMM_KEY_RELEASED ) && ( key->sym == 'p' ))
		    {
			xmm_logging( 1, "xmmplay: Pause toggle...\n" );
			xmmplay_Pause( xmmplay );
		    }

		    if(( key->state == XMM_KEY_RELEASED ) && ( key->sym == 's' ))
		    {
			xmm_logging( 1, "xmmplay: Seeking...\n" );
			xmmplay_Seek( xmmplay, 0.55 );
		    }

		    break;
		}

	/* Unknown Event: return error */
	default:
		fprintf( stderr, "ERROR: Unable to handler event of type %i\n", event.type );
		break;  
    }
  }

  xmmplay_Stop( xmmplay );
  xmmplay_Close( xmmplay );
  xmmplay_Exit( xmmplay );

  return 0;
}

/*
 * Event handler ( for not queueable events )
 */

static int EventCB( void *priv, XMM_Event *event )
{

  switch( event->type )
  {
	/* Select from Table of Contents */
	case XMM_EVENT_TOC:
		{
		    int i;
		    XMM_Event_TOC *etoc = (XMM_Event_TOC *) event;

		    fprintf( stdout, "\nTable of Contents:\n" );
		    for( i = 0; i < etoc->entries; i++ )	fprintf( stdout, "\t%2i: %s\n", i, etoc->entry[i] );

		    fprintf( stdout, "Please make your selection: " );
		    scanf( "%i", &etoc->selected );
		    return 0;
		}

	/* Authorization requiered */
	case XMM_EVENT_AUTH:
		{
		    char *passwd;
		    XMM_Event_Auth *eauth = (XMM_Event_Auth *) event;

		    fprintf( stdout, "\nAuthorization required:\n" );
		    fprintf( stdout, "User: " );
		    scanf( "%s", eauth->user );
		    passwd = getpass( "Password:" );
		    if( passwd == NULL )	break;
		    strcpy( eauth->passwd, passwd );
		    return 0;
		}

	/* Unknown Event: return error */
	default:
		fprintf( stderr, "ERROR: Unable to handler event of type %i\n", event->type );
		break;  
  }

  return XMM_RET_ERROR;
}
