/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <libmplayer/mplayer.h>

//
// libmplayer API code
//

void mplayer_ShowWindow( mptTheme *theme )
{
  gtk_widget_show( theme->window );
  theme->visible = 1;
  mplayer_WidgetDrawAll( theme );
}

void mplayer_HideWindow( mptTheme *theme )
{
  gtk_widget_hide( theme->window );
  theme->visible = 0;
}

//
// libmplayer internal
//

//
// Prototypes
//

static void on_pl_focus_in_event( GtkWidget * widget, GdkEvent * event, gpointer callback_data );
static void on_pl_focus_out_event( GtkWidget * widget, GdkEvent * event, gpointer callback_data );
static gint on_pl_expose_event( GtkWidget *widget, GdkEventButton *event, gpointer callback_data );
static gint on_pl_button_press_event( GtkWidget *widget, GdkEventButton *event, gpointer callback_data );
static gint on_pl_button_release_event( GtkWidget *widget, GdkEventButton *event, gpointer callback_data );
static gint on_pl_motion_notify_event( GtkWidget *widget, GdkEventButton *event, gpointer callback_data );

static int widget_inside( gint x, gint y, mptWidget *w );

//
// Create Gtk window
//

GtkWidget *mpint_CreateGtkWindow( mptTheme *theme, int x, int y, int width, int height, char *title )
{
  GtkWidget *window;

  // * Create window
  window = gtk_window_new( GTK_WINDOW_TOPLEVEL );

  // * Don't allow shrink and grow, auto_shrink
  gtk_window_set_policy( GTK_WINDOW( window ), FALSE, FALSE, TRUE );
  // * Window Size
  gtk_widget_set_usize( GTK_WIDGET( window ), width, height );
  // * Window Position
  gtk_widget_set_uposition( GTK_WIDGET( window ), x, y );
  // * Window Name
  gtk_window_set_title( GTK_WINDOW( window ), title );
  // * We want to be notified about...
  gtk_widget_set_events( window,	GDK_FOCUS_CHANGE_MASK |
					GDK_EXPOSURE_MASK |
					GDK_BUTTON_PRESS_MASK |
					GDK_BUTTON_RELEASE_MASK |
					GDK_POINTER_MOTION_MASK );

  //
  //  Connect signals
  //

  gtk_signal_connect( GTK_OBJECT( window ), "focus_in_event",
			    GTK_SIGNAL_FUNC( on_pl_focus_in_event ), theme );
  gtk_signal_connect( GTK_OBJECT( window ), "focus_out_event",
			    GTK_SIGNAL_FUNC( on_pl_focus_out_event ), theme );

  gtk_signal_connect( GTK_OBJECT( window ), "expose_event",
			    GTK_SIGNAL_FUNC( on_pl_expose_event ), theme );

  gtk_signal_connect( GTK_OBJECT( window ), "button_press_event",
			    GTK_SIGNAL_FUNC( on_pl_button_press_event ), theme );
  gtk_signal_connect( GTK_OBJECT( window ), "button_release_event",
			    GTK_SIGNAL_FUNC( on_pl_button_release_event ), theme );
  gtk_signal_connect( GTK_OBJECT( window ), "motion_notify_event",
			    GTK_SIGNAL_FUNC( on_pl_motion_notify_event ), theme );

  //
  // Final settings
  //

  // * Gdk operation need this, ( called automatic by gtk_widget_show )
  gtk_widget_realize( window );

  // * None of the GdkWMDecorations: Title, Minimize, Maximize... [gdktypes.h]
  gdk_window_set_decorations( window->window, (GdkWMDecoration) 0 );

  // * Allow painting in widget
  gtk_widget_set_app_paintable( window, TRUE );

  return window;
}
//
// Window event callbacks
//

static void on_pl_focus_in_event( GtkWidget * widget, GdkEvent * event, gpointer callback_data )
{
  mptEvent	mevent;
  mptTheme	*theme = (mptTheme *)callback_data;

  mevent.type = MPLAYER_EVENT_FOCUS_IN;
  theme->cbMPlayerEvent( theme, NULL, &mevent );
}

static void on_pl_focus_out_event( GtkWidget * widget, GdkEvent * event, gpointer callback_data )
{
  mptEvent	mevent;
  mptTheme	*theme = (mptTheme *)callback_data;

  mevent.type = MPLAYER_EVENT_FOCUS_OUT;
  theme->cbMPlayerEvent( theme, NULL, &mevent );
}

static gint on_pl_expose_event( GtkWidget *widget, GdkEventButton *event, gpointer callback_data )
{
  mplayer_WidgetDrawAll((mptTheme *)callback_data );
  return TRUE;
}

static gint on_pl_button_press_event( GtkWidget *widget, GdkEventButton *event, gpointer callback_data )
{
  mptTheme	*theme = (mptTheme *)callback_data;
  mptEvent	mevent;
  mptWidget	*w;
  GList		*le;

  if( theme->doubleState && theme->enableDouble )
  {
	event->x /= 2;
	event->y /= 2;
  }

  for( le = theme->wlist, theme->mb_pressed_w = NULL; le != NULL; le = le->next )
  {
	w = ((mptWidget *)le->data);
	if( widget_inside( event->x, event->y, w ))	theme->mb_pressed_w = w;
  }

  if( theme->mb_pressed_w )
	if( theme->mb_pressed_w->cbButtonPress && theme->mb_pressed_w->visible )
		theme->mb_pressed_w->cbButtonPress( theme->window, event, theme->mb_pressed_w );

  mevent.type = MPLAYER_EVENT_BUTTON_PRESS;
  mevent.data.mouse.x = event->x;
  mevent.data.mouse.y = event->y;
  mevent.data.mouse.button = event->button;
  theme->cbMPlayerEvent( theme, theme->mb_pressed_w, &mevent );

  if( theme->doubleState && theme->enableDouble )
  {
	event->x *= 2;
	event->y *= 2;
  }

  if(((mptWidget *)theme->mb_pressed_w == *theme->wtable[1].pWidget ) &&
			( event->button == MPLAYER_BUTTON_LEFT ))
  {
	theme->mb_pressed = 1;
	theme->mb_pressed_x = event->x;
	theme->mb_pressed_y = event->y;
  }

  return TRUE;
}

static gint on_pl_button_release_event( GtkWidget *widget, GdkEventButton *event, gpointer callback_data )
{
  mptTheme	*theme = (mptTheme *)callback_data;
  mptEvent	mevent;

  if( theme->doubleState && theme->enableDouble )
  {
	event->x /= 2;
	event->y /= 2;
  }

  if( theme->mb_pressed_w )
	if( theme->mb_pressed_w->cbButtonRelease && theme->mb_pressed_w->visible  )
		theme->mb_pressed_w->cbButtonRelease( theme->window, event, theme->mb_pressed_w );

  mevent.type = MPLAYER_EVENT_BUTTON_RELEASE;
  mevent.data.mouse.x = event->x;
  mevent.data.mouse.y = event->y;
  mevent.data.mouse.button = event->button;
  theme->cbMPlayerEvent( theme, theme->mb_pressed_w, &mevent );

  if( theme->doubleState && theme->enableDouble )
  {
	event->x *= 2;
	event->y *= 2;
  }

  if((mptWidget *)theme->mb_pressed_w == *theme->wtable[1].pWidget )	theme->mb_pressed = 0;

  return TRUE;
}

static gint on_pl_motion_notify_event( GtkWidget *widget, GdkEventButton *event, gpointer callback_data )
{
  mptTheme	*theme = (mptTheme *)callback_data;
  mptEvent	mevent;
  mptWidget	*w;
  GList		*le;

  if( theme->mb_pressed )	gtk_widget_set_uposition( widget, (int)event->x_root - theme->mb_pressed_x, (int)event->y_root - theme->mb_pressed_y );

  if( theme->doubleState && theme->enableDouble )
  {
	event->x /= 2;
	event->y /= 2;
  }

  for( le = theme->wlist; le != NULL; le = le->next )
  {
	w = ((mptWidget *)le->data);
	if( w->cbMotion && w->visible )
	    if( widget_inside( event->x, event->y, w ))
		w->cbMotion( theme->window, event, w );
  }

  mevent.type = MPLAYER_EVENT_MOTION;
  mevent.data.mouse.x = event->x;
  mevent.data.mouse.y = event->y;
  theme->cbMPlayerEvent( theme, NULL, &mevent );

  if( theme->doubleState && theme->enableDouble )
  {
	event->x *= 2;
	event->y *= 2;
  }

  return TRUE;
}

//
// Private code
//

static int widget_inside( gint x, gint y, mptWidget *w )
{
  if(	( x >= w->x ) &&  ( x < w->x + w->width ) &&
	( y >= w->y ) &&  ( y < w->y + w->height ))	return 1;

  return 0;
}

//
// Internal to libmplayer
//

#include <gdk/gdkx.h>

//
// Prototypes
//

static GdkBitmap *MaskDouble( GdkWindow *window, GdkBitmap *src, int width, int height );
static void	DrawDouble( GdkDrawable *drawable, GdkGC *gc, GdkDrawable *src,
			    int xsrc, int ysrc, int xdest, int ydest,
			    int	width, int height );

//
// Put buffer into window
//

void mpint_RefreshTheme( mptTheme *theme )
{
  mptWidget *w = *(theme->wtable[0].pWidget);

  if( theme->doubleState && theme->enableDouble )
  {
	DrawDouble( theme->window->window, 
		    theme->window->style->fg_gc[ GTK_WIDGET_STATE( theme->window )],
		    theme->pixmap, 0, 0, 0, 0, w->width, w->height );
  }
  else

  gdk_draw_pixmap( theme->window->window,
	theme->window->style->fg_gc[ GTK_WIDGET_STATE( theme->window )],
	theme->pixmap, 0, 0, 0, 0, w->width, w->height );
}

//
// Initialize theme window
//

void mpint_InitTheme( mptTheme *tdesc, int state )
{
  mptWidget	*bw;

  if( state == tdesc->doubleState )	return;
  tdesc->doubleState = state;

  if( !tdesc->enableDouble && tdesc->doubleState )	return;

  bw = *(tdesc->wtable[0].pWidget);

  // Create Mask for double size pixmap

  if( tdesc->mask_ds == NULL )
	tdesc->mask_ds = MaskDouble( tdesc->window->window, bw->mask, bw->width, bw->height );

  // Resize window

  if( tdesc->doubleState )
  {
	gdk_window_set_hints( tdesc->window->window, 0, 0, bw->width, bw->height, bw->width << 1, bw->height << 1, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE );
	gdk_window_resize( tdesc->window->window, bw->width << 1, bw->height << 1 );
	gdk_window_set_hints( tdesc->window->window, 0, 0, bw->width << 1, bw->height << 1, bw->width << 1, bw->height << 1, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE );
	gtk_widget_shape_combine_mask( tdesc->window, tdesc->mask_ds, 0, 0 );
  }
  else
  {
	gdk_window_set_hints( tdesc->window->window, 0, 0, bw->width, bw->height, bw->width << 1, bw->height << 1, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE );
	gdk_window_resize( tdesc->window->window, bw->width, bw->height );
	gdk_window_set_hints( tdesc->window->window, 0, 0, bw->width, bw->height, bw->width, bw->height, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE );
	gtk_widget_shape_combine_mask( tdesc->window, bw->mask, 0, 0 );
  }

  gdk_flush();
}

//
// Private code ( Create double sized pixmap and mask )
//

static void DrawDouble( GdkDrawable *drawable, GdkGC *gc, GdkDrawable *src,
			    int xsrc, int ysrc, int xdest, int ydest,
			    int	width, int height )
{
  GdkImage	*image, *image_dbl;

  image = gdk_image_get( src, xsrc, ysrc, width, height );
  image_dbl = gdk_image_new( GDK_IMAGE_NORMAL, gdk_visual_get_best(), width << 1, height << 1 );

  if( image_dbl->bpp == 1 )		// Not tested
  {
	register char	*src, *row1, *row2, val;
	register int	x, y;

	src = (char *) ((GdkImagePrivate *) image)->ximage->data;
	row1 = (char *) ((GdkImagePrivate *) image_dbl)->ximage->data;
	row2 = row1 + image_dbl->bpl;

	for( y = 0; y < image->height; y++ )
	{
	    for( x = 0; x < image->width; x++ )
	    {
		val = *src++;
		*row1++ = val;
		*row1++ = val;
		*row2++ = val;
		*row2++ = val;
	    }

	    src += image->bpl - image->width;	// <bpl != width * bpp>
	    row1 += ( image_dbl->bpl << 1 ) - image_dbl->width;
	    row2 += ( image_dbl->bpl << 1 ) - image_dbl->width;
	}
  }

  if( image_dbl->bpp == 2 )
  {
	register short	*src, *row1, *row2, val;
	register int	x, y, sdiff, ddiff;

	src = (short *) ((GdkImagePrivate *) image)->ximage->data;
	row1 = (short *) ((GdkImagePrivate *) image_dbl)->ximage->data;
	row2 = row1 + ( image_dbl->bpl >> 1 );

	sdiff = ( image->bpl >> 1 ) - image->width;
	ddiff = image_dbl->bpl - image_dbl->width;

	for( y = 0; y < image->height; y++ )
	{
	    for( x = 0; x < image->width; x++ )
	    {
		val = *src++;
		*row1++ = val;
		*row1++ = val;
		*row2++ = val;
		*row2++ = val;
	    }

	    src += sdiff;	// <bpl != width * bpp>
	    row1 += ddiff;
	    row2 += ddiff;
	}
  }

  if( image_dbl->bpp == 3 )		// Not tested
  {
	register char	*src, *row1, *row2, val1, val2, val3;
	register int	x, y, sdiff, ddiff;

	src = (char *) ((GdkImagePrivate *) image)->ximage->data;
	row1 = (char *) ((GdkImagePrivate *) image_dbl)->ximage->data;
	row2 = row1 + image_dbl->bpl;

	sdiff = image->bpl - image->width * 3;
	ddiff = ( image_dbl->bpl << 1 ) - image_dbl->width * 3;

	for( y = 0; y < image->height; y++ )
	{
	    for( x = 0; x < image->width; x++ )
	    {
		val1 = *src++;
		val2 = *src++;
		val3 = *src++;

		*row1++ = val1;
		*row1++ = val2;
		*row1++ = val3;
		*row1++ = val1;
		*row1++ = val2;
		*row1++ = val3;

		*row2++ = val1;
		*row2++ = val2;
		*row2++ = val3;
		*row2++ = val1;
		*row2++ = val2;
		*row2++ = val3;
	    }

	    src += sdiff;	// <bpl != width * bpp>
	    row1 += ddiff;
	    row2 += ddiff;
	}
  }

  if( image_dbl->bpp == 4 )		// Not tested
  {
	register long	*src, *row1, *row2, val;
	register int	x, y, sdiff, ddiff;

	src = (long *) ((GdkImagePrivate *) image)->ximage->data;
	row1 = (long *) ((GdkImagePrivate *) image_dbl)->ximage->data;
	row2 = row1 + ( image_dbl->bpl >> 2 );

	sdiff = ( image->bpl >> 2 ) - image->width;
	ddiff = ( image_dbl->bpl >> 1 ) - image_dbl->width;

	for( y = 0; y < image->height; y++ )
	{
	    for( x = 0; x < image->width; x++ )
	    {
		val = *src++;
		*row1++ = val;
		*row1++ = val;
		*row2++ = val;
		*row2++ = val;
	    }

	    src += sdiff;	// <bpl != width * bpp>
	    row1 += ddiff;
	    row2 += ddiff;
	}
  }

  gdk_draw_image( drawable, gc, image_dbl, 0, 0, xdest * 2, ydest * 2, width * 2, height * 2 );

  gdk_image_destroy( image_dbl );
  gdk_image_destroy( image );
}

// This is slow. But needs only to be called one time for each window

static GdkBitmap *MaskDouble( GdkWindow *window, GdkBitmap *src, int width, int height )
{
  GdkImage	*image, *image_dbl;
  GdkBitmap	*bitmap;
  GdkGC		*gc;
  GdkColor	color;

// Get image ( from original mask )
  image = gdk_image_get( src, 0, 0, width, height );

// Create double size bitmap
  bitmap = gdk_pixmap_new( window, width << 1, height << 1, 1 );
  image_dbl = gdk_image_get( bitmap, 0, 0, width << 1, height << 1 );
  gc = gdk_gc_new( bitmap );

  color.pixel = 1;
  gdk_gc_set_foreground( gc, &color );
  gdk_draw_rectangle( bitmap, gc, TRUE, 0, 0, width << 1, height << 1 );

  if( 1 )
  {
	int			x, y, i, ddiff, sdiff;
	unsigned short		*row1, *row2;
	unsigned char		*src;
	register unsigned short dval, val;

	src = (unsigned char *) ((GdkImagePrivate *) image)->ximage->data;
	row1 = (unsigned short *) ((GdkImagePrivate *) image_dbl)->ximage->data;
	row2 = row1 + ( image_dbl->bpl >> 1 );

	ddiff = ((( image_dbl->bpl << 3 ) - image_dbl->width ) >> 4 ) + ( image_dbl->bpl >> 1 );
	sdiff = (( image->bpl << 3 ) - image->width ) >> 3;

	for( y = 0; y < image->height; y++ )
	{
	    for( x = 0; x < image->width; )
	    {
		val = *src++;
		dval = 0;

		for( i = 7; ( i >= 0 ) /*&& ( x < image->width )*/; i--, x++ )
		{
		    dval |= ((( val >> i ) & 1 ) << ( i * 2 ));
		    dval |= ((( val >> i ) & 1 ) << ( i * 2 + 1 ));
		}

		*row1++ = dval;
		*row2++ = dval;
	    }

	    src += sdiff;	// <bpl != width * bpp>
	    row1 += ddiff;
	    row2 += ddiff;
	}
  }

  gdk_draw_image( bitmap, gc, image_dbl, 0, 0, 0, 0, width << 1, height << 1 );

  gdk_image_destroy( image_dbl );
  gdk_image_destroy( image );
  gdk_gc_destroy( gc );

  return bitmap;
}
