/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

//
// List
//

#include <gtk/gtk.h>
#include <libmplayer/mplayerwidgets.h>
#include "gtkwin.h"

//
// Draw callback
//

static void wListDraw( mptWidget *widget )
{
  mptList *list = (mptList *) widget;
  GdkFont *font = NULL;
  GList *le;
  GdkGC *gc;
  int y, i;

  if( !widget->visible )	return;

  if(( font = gdk_font_load( "-adobe-helvetica-medium-r-normal--*-100-*-*-*-*-*-*" )) == NULL )	return;

  gc = list->sys.gc,
  list->fheight = font->ascent + font->descent + 1;

  gdk_gc_set_foreground( gc, &list->bgcolor );
  gdk_draw_rectangle(	list->sys.window, gc, TRUE,
			list->sys.x, list->sys.y,
			list->sys.width, list->sys.height );

  for( le = list->elist, y = 0, i = 0; le && y < list->sys.height; le = le->next, i++ )
  {
	if( i < list->start )	continue;

	if(((mptListEntry *)le->data)->selected )
	{
		gdk_gc_set_foreground( gc, &list->bgcolor_h );
		gdk_draw_rectangle( list->sys.window, gc, TRUE,
		list->sys.x, list->sys.y + y, list->sys.width, list->fheight );
	}

	if( i == list->active )	gdk_gc_set_foreground( gc, &list->color_h );
	else	gdk_gc_set_foreground( gc, &list->color );

	gdk_draw_text(	list->sys.window, font,	gc,
			list->sys.x, list->sys.y + y + font->ascent,
			((mptListEntry *)le->data)->text,
			strlen(((mptListEntry *)le->data)->text ));

	y += list->fheight;
  }
}

//
// Button press callback
//

static int wList_button_pressed( GtkWidget *gtkwidget, GdkEventButton *event, mptWidget *widget )
{
  mptList *list = (mptList *) widget;
  GList *le, *les;
  int len, selected;

  len = g_list_length( list->elist );
  if( len == 0 )	return 1;

  selected = list->start + ( event->y - list->sys.y ) / list->fheight;
  if( selected > len )	selected = len - 1;

  les = g_list_nth( list->elist, selected );

  if( event->state & GDK_CONTROL_MASK )
	((mptListEntry *)les->data)->selected = (((mptListEntry *)les->data)->selected + 1 ) & 1;
  else
  {
	for( le = list->elist; le; le = le->next )	((mptListEntry *)le->data)->selected = 0;
	((mptListEntry *)les->data)->selected = 1;
  }

  if( event->type == GDK_2BUTTON_PRESS )
  {
	list->active = selected;
	if( list->active > len )	list->active = len - 1;
	if( list->sys.ReleaseCallback )	list->sys.ReleaseCallback( list->active | MPLAYER_LIST_DOUBLE_CLICK );
  }
  else	if( list->sys.ReleaseCallback )	list->sys.ReleaseCallback( selected );

  list->sys.cbDraw( (mptWidget *)list );
  mpint_RefreshTheme( list->sys.theme );
  list->selected = selected;
  list->pressed = 1;

  return 1;
}

//
// Button release callback
//

static int wList_button_released( GtkWidget *gtkwidget, GdkEventButton *event, mptWidget *widget )
{
  mptList *list = (mptList *) widget;

  list->pressed = 0;

  return 1;
}

//
// Motion callback
//

static int wList_motion( GtkWidget *gtkwidget, GdkEventButton *event, mptWidget *widget )
{
  mptList *list = (mptList *) widget;
  GList *les;
  int len, selected;
  
  if( !list->pressed )	return 1;

  len = g_list_length( list->elist );

  selected = list->start + ( event->y - list->sys.y ) / list->fheight;
  if( selected >= len )	selected = len - 1;

  if( selected == list->selected )	return 1;

  les = g_list_nth( list->elist, list->selected );
  list->elist = g_list_remove_link( list->elist, les );
  list->elist = g_list_insert( list->elist, les->data, selected );

  list->sys.cbDraw( (mptWidget *)list );
  mpint_RefreshTheme( list->sys.theme );
  list->selected = selected;

  if( list->sys.ChangeCallback )	list->sys.ChangeCallback( list->selected );

  return 1;
}

//
// Parse and create List
//

#include <string.h>
#include "parser.h"

static char		name[256];
static mptList		*list;

mptWidget *wListParse( mptTheme *theme, int mode, char *option, char *value )
{
  int color, color_h;

  if( mode == WIDGET_START )
  {
	list = (mptList *) g_malloc( sizeof( mptList ));

	list->sys.cbDraw = wListDraw;
	list->sys.cbButtonPress = wList_button_pressed;
	list->sys.cbButtonRelease = wList_button_released;
	list->sys.cbMotion = wList_motion;
	list->sys.Free = NULL;
	list->elist = NULL;
	list->start = 0;
	list->active = 0;
	list->pressed = 0;
	list->selected = 0;

	if( !ReadString( option, name ))	strcpy( name, option );
  }

  if( mode == WIDGET_STOP )
  {
	if( !LookupWidget( theme, name, NULL, (mptWidget *)list ))	return NULL;

	list->sys.gc = gdk_gc_new( list->sys.window );

	return (mptWidget *) list;
  }
  if( mode == WIDGET_OPTION )
  {
	if( !strcasecmp( option, "Position" ))	ReadCoord( value, &list->sys.x, &list->sys.y );
	if( !strcasecmp( option, "Size" ))	ReadCoord( value, &list->sys.width, &list->sys.height );
	if( !strcasecmp( option, "Color" ))
	{
		ReadCoord( value, &color, &color_h );

		list->color.red = (( color & 0xFF0000 ) >> 8 ) | 0xFF;
		list->color.green = ( color & 0xFF00 ) | 0xFF;
		list->color.blue = (( color & 0xFF ) << 8 ) | 0xFF;
		gdk_color_alloc( gdk_colormap_get_system(), &list->color );
		list->color_h.red = (( color_h & 0xFF0000 ) >> 8 ) | 0xFF;
		list->color_h.green = ( color_h & 0xFF00 ) | 0xFF;
		list->color_h.blue = (( color_h & 0xFF ) << 8 ) | 0xFF;
		gdk_color_alloc( gdk_colormap_get_system(), &list->color_h );
	}
	if( !strcasecmp( option, "bgColor" ))
	{
		ReadCoord( value, &color, &color_h );

		list->bgcolor.red = (( color & 0xFF0000 ) >> 8 ) | 0xFF;
		list->bgcolor.green = ( color & 0xFF00 ) | 0xFF;
		list->bgcolor.blue = (( color & 0xFF ) << 8 ) | 0xFF;
		gdk_color_alloc( gdk_colormap_get_system(), &list->bgcolor );
		list->bgcolor_h.red = (( color_h & 0xFF0000 ) >> 8 ) | 0xFF;
		list->bgcolor_h.green = ( color_h & 0xFF00 ) | 0xFF;
		list->bgcolor_h.blue = (( color_h & 0xFF ) << 8 ) | 0xFF;
		gdk_color_alloc( gdk_colormap_get_system(), &list->bgcolor_h );
	}
  }

  return NULL;
}

int mpwListAddEntry( mptList *list, char *text )
{
  mptListEntry *entry;

  entry = g_malloc( sizeof( mptListEntry ));
  entry->selected = 0;
  entry->text = text;

  list->elist = g_list_append( list->elist, entry );
  list->sys.cbDraw( (mptWidget *)list );
  mpint_RefreshTheme( list->sys.theme );
  return 1;
}

void mpwListClear( mptList *list )
{
  GList *le;

  for( le = list->elist; le; le = le->next )	g_free( le->data );
  g_list_free( list->elist );
  list->elist = NULL;

  list->sys.cbDraw( (mptWidget *)list );
  mpint_RefreshTheme( list->sys.theme );
}

void mpwListSetActive( mptList *list, int active )
{
  list->active = active;
  list->sys.cbDraw( (mptWidget *)list );
  mpint_RefreshTheme( list->sys.theme );
}

void mpwListRemove( mptList *list, int num )
{
  GList *le;

  le = g_list_nth( list->elist, num );
  g_free( le->data );
  list->elist = g_list_remove_link( list->elist, le );

  list->sys.cbDraw( (mptWidget *)list );
  mpint_RefreshTheme( list->sys.theme );
}

int mpwListSelectionGet( mptList *list, int nth )
{
  GList		*le;

  le = g_list_nth( list->elist, nth );
  if( le == NULL )	return -1;

  return ((mptListEntry *)le->data)->selected;
}

int mpwListSelectionSet( mptList *list, int nth, int flag )
{
  GList		*le;

  le = g_list_nth( list->elist, nth );
  if( le == NULL )	return -1;

  ((mptListEntry *)le->data)->selected = flag;
  list->sys.cbDraw( (mptWidget *)list );
  mpint_RefreshTheme( list->sys.theme );
  return 1;
}
