/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Main player window
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <sys/stat.h>
  
#include <gtk/gtk.h>
#include "interface.h"

#include <libxmm/libxmm.h>
#include <libxmm/util/utils.h>
#include <libxmmplay/xmmplay.h>
#include <libmplayer/mplayer.h>

#include "main.h"
#include "deftheme.h"

#include <gdk/gdkx.h>

/*
 * Definitions
 */

#define	MAINWIN_WIDGETS		27	/* Widgets */

#define MAINMENU_ITEMS		24	/* Popup menus */

enum				/* Codes for Menucallback */
{
    MAINMENU_ABOUT, MAINMENU_LOAD, MAINMENU_CLOSE, MAINMENU_FILEINFO,
    MAINMENU_VIEW_ORIG, MAINMENU_VIEW_DOUBLE, MAINMENU_VIEW_FULL,
    MAINMENU_PLAY_PLAY, MAINMENU_PLAY_PAUSE, MAINMENU_PLAY_STOP,
    MAINMENU_PLAY_REPEAT, MAINMENU_PLAY_RANDOM,
    MAINMENU_OPTIONS_PREF, MAINMENU_OPTIONS_SKIN, MAINMENU_PLAYLIST,
    MAINMENU_EXIT
};


/*
 * Prototypes ( needed by global data )
 */

static void mainwin_menucallback( GtkWidget *widget, gpointer data );
static void mplayer_callback( mptTheme *theme, mptWidget *widget, mptEvent *event );

static void on_menu_release( int value );
static void on_icon_release( int value );
static void on_exit_release( int value );
static void on_mini_release( int value );
static void on_eject_release( int value );

static void on_volume_change( int value );
static void on_volume_release( int value );
static void on_balance_change( int value );
static void on_balance_release( int value );
static void on_position_change( int value );
static void on_position_release( int value );

static void on_repeat_release( int value );
static void on_random_release( int value );

static void on_menubar_change( int value );
static void on_menubar_release( int value );

/*
 * Global data
 */

static mptPushButton	*mainwin_menu;
static mptPushButton	*mainwin_icon;
static mptPushButton	*mainwin_mini;
static mptPushButton	*mainwin_exit;
static mptMenu		*mainwin_menubar;
static mptPushButton	*mainwin_rew;
static mptPushButton	*mainwin_play;
static mptPushButton	*mainwin_pause;
static mptPushButton	*mainwin_stop;
static mptPushButton	*mainwin_fwd;
static mptPushButton	*mainwin_eject;
static mptToggleButton	*mainwin_repeat;
static mptToggleButton	*mainwin_random;
static mptSlider	*mainwin_volume;
static mptSlider	*mainwin_balance;
static mptSlider	*mainwin_position;
static mptTextBox	*mainwin_info;
static mptTextBox	*mainwin_info_rate;
static mptTextBox	*mainwin_info_freq;
static mptPixmap	*mainwin_back;
static mptNumber	*mainwin_time_h;
static mptNumber	*mainwin_time_m;
static mptNumber	*mainwin_time_s;
static mptStatus	*mainwin_chmode;
static mptStatus	*mainwin_pmode;
static mptStatus	*mainwin_titlebar;
static mptDrawArea	*mainwin_vis;

/* Popup menus */

static GtkItemFactory	*mainwin_mainmenu;

/* Other stuff */

static gint		timer = 0;
//static KList		*iAudioFilterList;
//extern XMM_PluginAudioFilter	main_vis_plugin_info;
static int		playing;

/*
 * Widget set
 */

static mptWidgetEntry mainwin_widgets[MAINWIN_WIDGETS] =
{
 { "back",	(mptWidget **)&mainwin_back,		deftheme_main_xpm },
 { "titlebar",	(mptWidget **)&mainwin_titlebar,	deftheme_titlebar_xpm },

 { "info",	(mptWidget **)&mainwin_info,		deftheme_font_xpm },
 { "info_freq",	(mptWidget **)&mainwin_info_freq,	deftheme_font_xpm },
 { "info_rate",	(mptWidget **)&mainwin_info_rate,	deftheme_font_xpm },

 { "menubar",	(mptWidget **)&mainwin_menubar,		deftheme_titlebar_xpm, on_menubar_release, on_menubar_change },

 { "menu",	(mptWidget **)&mainwin_menu,		deftheme_titlebar_xpm, on_menu_release },
 { "icon",	(mptWidget **)&mainwin_icon,		deftheme_titlebar_xpm, on_icon_release },
 { "mini",	(mptWidget **)&mainwin_mini,		deftheme_titlebar_xpm, on_mini_release },
 { "exit",	(mptWidget **)&mainwin_exit,		deftheme_titlebar_xpm, on_exit_release },

 { "rew",	(mptWidget **)&mainwin_rew,		deftheme_cbuttons_xpm, mp_on_rew_release },
 { "play",	(mptWidget **)&mainwin_play,		deftheme_cbuttons_xpm, mp_on_play_release },
 { "pause",	(mptWidget **)&mainwin_pause,		deftheme_cbuttons_xpm, mp_on_pause_release },
 { "stop",	(mptWidget **)&mainwin_stop,		deftheme_cbuttons_xpm, mp_on_stop_release },
 { "fwd",	(mptWidget **)&mainwin_fwd,		deftheme_cbuttons_xpm, mp_on_fwd_release },
 { "eject",	(mptWidget **)&mainwin_eject,		deftheme_cbuttons_xpm, on_eject_release },

 { "chmode",	(mptWidget **)&mainwin_chmode,		deftheme_monoster_xpm },
 { "pmode",	(mptWidget **)&mainwin_pmode,		deftheme_playpaus_xpm },

 { "volume",	(mptWidget **)&mainwin_volume,		deftheme_volbal_xpm, on_volume_release, on_volume_change },
 { "balance",	(mptWidget **)&mainwin_balance,		deftheme_volbal_xpm, on_balance_release, on_balance_change },
 { "position",	(mptWidget **)&mainwin_position,	deftheme_posbar_xpm, on_position_release, on_position_change },

 { "time_h",	(mptWidget **)&mainwin_time_h,		deftheme_digits_xpm },
 { "time_min",	(mptWidget **)&mainwin_time_m,		deftheme_digits_xpm },
 { "time_sec",	(mptWidget **)&mainwin_time_s,		deftheme_digits_xpm },

 { "shuffle",	(mptWidget **)&mainwin_random,		deftheme_shufrep_xpm, on_random_release },
 { "repeat",	(mptWidget **)&mainwin_repeat,		deftheme_shufrep_xpm, on_repeat_release },

 { "vis",	(mptWidget **)&mainwin_vis,		NULL, NULL, NULL },
};


/*
 * Theme
 */

static mptTheme mainwin_theme =
{
	mainwin_widgets, MAINWIN_WIDGETS, "MainWin", &deftheme_desc, 1,
	"LinuX Media Player", mplayer_callback
};

/*
 * Popup menus
 */

#define MENU_STR_PLAYLIST	"/Playlist"
#define MENU_STR_REPEAT		"/Play Control/Repeat"
#define MENU_STR_RANDOM		"/Play Control/Random"

static GtkItemFactoryEntry mainwin_mainmenu_items[] = {
    { "/About XMMP",		NULL,		mainwin_menucallback, MAINMENU_ABOUT, NULL },
    { "/sep1",			NULL,   	NULL, 0, "<Separator>" },
    { "/Load File...",		NULL,		mainwin_menucallback, MAINMENU_LOAD, NULL },
    { "/Close File",		NULL,		mainwin_menucallback, MAINMENU_CLOSE, NULL },
    { "/File Info",		NULL,		mainwin_menucallback, MAINMENU_FILEINFO, NULL },
    { "/sep2",			NULL,		NULL, 0, "<Separator>" },
    { MENU_STR_PLAYLIST,	NULL,		mainwin_menucallback, MAINMENU_PLAYLIST, "<ToggleItem>" },
    { "/sep3",			NULL,		NULL, 0, "<Separator>" },
    { "/View",			NULL,		NULL, 0, "<Branch>" },
    { "/View/Original",		"<alt>1",	mainwin_menucallback, MAINMENU_VIEW_ORIG, NULL },
    { "/View/Double",		"<alt>2",	mainwin_menucallback, MAINMENU_VIEW_DOUBLE, NULL },
    { "/View/Fullscreen",	"<alt>3",	mainwin_menucallback, MAINMENU_VIEW_FULL, NULL },
    { "/Play Control",		NULL,		NULL, 0, "<Branch>" },
    { "/Play Control/Play",	NULL,		mainwin_menucallback, MAINMENU_PLAY_PLAY, NULL },
    { "/Play Control/Pause",	NULL,		mainwin_menucallback, MAINMENU_PLAY_PAUSE, NULL },
    { "/Play Control/Stop",	NULL,		mainwin_menucallback, MAINMENU_PLAY_STOP, NULL },
    { "/Play Control/sep3",	NULL,		NULL, 0, "<Separator>" },
    { MENU_STR_REPEAT,		NULL,		mainwin_menucallback, MAINMENU_PLAY_REPEAT, "<ToggleItem>" },
    { MENU_STR_RANDOM,		NULL,		mainwin_menucallback, MAINMENU_PLAY_RANDOM, "<ToggleItem>" },
    { "/Options",		NULL,		NULL, 0, "<Branch>" },
    { "/Options/Preferences",	"<control>S",	mainwin_menucallback, MAINMENU_OPTIONS_PREF, NULL },
    { "/Options/Skin Browser",	"<control>T",	mainwin_menucallback, MAINMENU_OPTIONS_SKIN, NULL },
    { "/sep4",			NULL,		NULL, 0, "<Separator>" },
    { "/Exit",			"<control>Q",	mainwin_menucallback, MAINMENU_EXIT, NULL },
};

/*
 * Prototypes
 */

static void	skinwin_restore_info( void );
static gint	timer_proc( gpointer data );
static int	FileOpen( char *filename );
static void	FileClose( void );

/*
 * Public code
 */

int mainwin_Init( void )
{
  int l, r, vol, bal;
  GtkAccelGroup *mainwin_accel;

  /* Create popup menu */
  mainwin_accel = gtk_accel_group_new();
  mainwin_mainmenu = gtk_item_factory_new( GTK_TYPE_MENU, "<Main>", mainwin_accel );
  gtk_item_factory_create_items( mainwin_mainmenu, MAINMENU_ITEMS, mainwin_mainmenu_items, NULL );

// Load theme
  mplayer_LoadTheme( cfg_ThemeDir, cfg_SkinName, &mainwin_theme );
  mplayer_InitTheme( &mainwin_theme, cfg_DoubleSize );
  mplayer_ShowWindow( &mainwin_theme );

//  if( xmm_PluginRegisterFromData((XMM_Plugin *)&main_vis_plugin_info ))
//	iAudioFilterList = KListAppend( iAudioFilterList, &main_vis_plugin_info );

// Initialize widgets
  skinwin_restore_info();
  mplayer_WidgetVisible( mainwin_time_h, 0 );
  mplayer_WidgetVisible( mainwin_time_m, 0 );
  mplayer_WidgetVisible( mainwin_time_s, 0 );
  mplayer_WidgetVisible( mainwin_position, 0 );
  mplayer_WidgetVisible( mainwin_pmode, 0 );
  mplayer_WidgetVisible( mainwin_chmode, 0 );
  mplayer_WidgetVisible( mainwin_info_freq, 0 );
  mplayer_WidgetVisible( mainwin_info_rate, 0 );

// Initial volume/balance

  if( xmmplay_GetVolume( xmmplay, &l, &r ) > -1 )
  {
	if( l > r )
	{
	    bal = ( r * 50 ) / l;
	    vol = l;
	}
	else	if( l < r )
	{
	    bal = 100 - ( l * 50 / r );
	    vol = r;
	}
	else
	{
	    bal = 50;
	    vol = r;
	}
  }
  else
  {
	vol = 100;
	bal = 50;
  }

  mpwSliderSetValue( mainwin_volume, vol );
  mpwSliderSetValue( mainwin_balance, bal );

  mpwToggleButtonState( mainwin_repeat, cfg_Random );
  mpwToggleButtonState( mainwin_repeat, cfg_Repeat );

// Draw widgets

  mplayer_WidgetDrawAll( &mainwin_theme );

// Initialize state of gtk widgets

  GTK_CHECK_MENU_ITEM( gtk_item_factory_get_widget( mainwin_mainmenu,
			    MENU_STR_PLAYLIST ))->active = cfg_ShowPlaylist;

  GTK_CHECK_MENU_ITEM( gtk_item_factory_get_widget( mainwin_mainmenu,
			    MENU_STR_REPEAT ))->active = cfg_Repeat;

  GTK_CHECK_MENU_ITEM( gtk_item_factory_get_widget( mainwin_mainmenu,
			    MENU_STR_RANDOM ))->active = cfg_Random;

  return 1;
}

//
// File select callbacks
//
// Also used in win_pl.c
//

void mp_FileSelectCallback( char *filename, int mode )
{
  char *ptr, *ptr2;

  if( mode == MPLAYER_FILESEL_OK )	playlist_Clear();

  ptr = g_basename( filename );
  ptr2 = g_dirname( filename );
  playlist_AddFile( ptr2, ptr );
  g_free( ptr2 );
}

//
// Play controls
//
// Also used in win_pl.c
//

void mp_on_play_release( int value )
{
  char *ptr;

  if( xmmplay_Status( xmmplay ) < 0 )
  {
	ptr = playlist_Current();

	if( ptr == NULL )
	{
		mplayer_FileSelect( 1, cfg_CurrentDir, mp_FileSelectCallback );
		ptr = playlist_Current();
		if( ptr == NULL )	return;
	}

	FileOpen( ptr );
	if( xmmplay_Status( xmmplay ) < 0 )	return;
  }

  xmmplay_Play( xmmplay );
  mpwStatusSetStatus( mainwin_pmode, 0 );
  playing = 1;
}

void mp_on_pause_release( int value )
{
  mpwStatusSetStatus( mainwin_pmode, xmmplay_Pause( xmmplay ));
}

void mp_on_stop_release( int value )
{
  playing = 0;
  xmmplay_Stop( xmmplay );
  mpwStatusSetStatus( mainwin_pmode, 2 );
}

void mp_on_rew_release( int value )
{
  char *ptr;
  int play = 0;

  ptr = playlist_Prev();
  if( ptr == NULL )	return;

  if( xmmplay_Status( xmmplay ) & XMM_PBS_PLAYING )	play = 1;
  if( xmmplay_Status( xmmplay ) & XMM_PBS_PAUSED )	play = 1;

  FileClose();
  FileOpen( ptr );

  if( xmmplay_Status( xmmplay ) < 0 )	return;

  if( play )
  {
	xmmplay_Play( xmmplay );
	mpwStatusSetStatus( mainwin_pmode, 0 );
	playing = 1;
  }
}

void mp_on_fwd_release( int value )
{
  char *ptr;
  int play = 0;

  ptr = playlist_Next();
  if( ptr == NULL )
  {
	if( cfg_Repeat == 0 )	return;
	ptr = playlist_nth( 0 );
	if( ptr == NULL )	return;
  }

  if(( xmmplay_Status( xmmplay ) & XMM_PBS_PLAYING ) || value )	play = 1;
  if( xmmplay_Status( xmmplay ) & XMM_PBS_PAUSED )	play = 1;

  FileClose();
  FileOpen( ptr );

  if( xmmplay_Status( xmmplay ) < 0 )	return;

  if( play )
  {
	xmmplay_Play( xmmplay );
	mpwStatusSetStatus( mainwin_pmode, 0 );
	playing = 1;
  }
}

//
// Show / Hide playlist
//

void mp_on_playlist_release( int value )
{
  if( !cfg_ShowPlaylist )
  {
	cfg_ShowPlaylist = 1;
	playlist_show();
  }
  else
  {
	cfg_ShowPlaylist = 0;
	playlist_hide();
  }

  GTK_CHECK_MENU_ITEM( gtk_item_factory_get_widget( mainwin_mainmenu,
			    MENU_STR_PLAYLIST ))->active = cfg_ShowPlaylist;
}

//
// Private data
//

static void mplayer_callback( mptTheme *theme, mptWidget *widget, mptEvent *event )
{
    switch( event->type )
    {
	case	MPLAYER_EVENT_FOCUS_IN:
		    mpwStatusSetStatus( mainwin_titlebar, 0 );
		    break;

	case	MPLAYER_EVENT_FOCUS_OUT:
		    mpwStatusSetStatus( mainwin_titlebar, 1 );
		    break;

	case	MPLAYER_EVENT_BUTTON_PRESS:
		    if( event->data.mouse.button == MPLAYER_BUTTON_RIGHT )
			    gtk_menu_popup( GTK_MENU( mainwin_mainmenu->widget ), NULL, NULL, NULL, NULL, 2, GDK_CURRENT_TIME );
		    else if( widget == (mptWidget *)mainwin_titlebar )
				gdk_window_raise( mainwin_theme.window->window );
		    break;

	case	MPLAYER_EVENT_BUTTON_RELEASE:
		    break;

	case	MPLAYER_EVENT_MOTION:
		    break;

	case	MPLAYER_EVENT_THEMECHANGE:
		    strcpy( cfg_ThemeDir, event->data.theme.theme );
		    strcpy( cfg_SkinName, event->data.theme.skin );
		    break;
    }
}

//
// Menu and callback
//

static void mainwin_menucallback( GtkWidget *widget, gpointer data )
{
  switch( (int) data )
  {
	case MAINMENU_ABOUT:		mp_AboutWin();
					break;

	case MAINMENU_LOAD:		on_eject_release( 0 );
					break;

	case MAINMENU_CLOSE:		FileClose();
					mplayer_WidgetVisible( mainwin_time_h, 0 );
					mplayer_WidgetVisible( mainwin_time_m, 0 );
					mplayer_WidgetVisible( mainwin_time_s, 0 );
					mplayer_WidgetVisible( mainwin_position, 0 );
					mplayer_WidgetVisible( mainwin_pmode, 0 );
					mplayer_WidgetVisible( mainwin_chmode, 0 );
					mplayer_WidgetVisible( mainwin_info_rate, 0 );
					mplayer_WidgetVisible( mainwin_info_freq, 0 );
					if( timer > -1 )
					{
						gtk_timeout_remove( timer );
						timer = -1;
					}
					mplayer_WidgetDrawAll( &mainwin_theme );
					break;

	case MAINMENU_FILEINFO:		xmmplay_InputDialogue( xmmplay, XMM_GTKDLG_INFO );
					break;

	case MAINMENU_VIEW_ORIG:	xmmplay_Scale( xmmplay, 0, 0, XMM_GRAPH_RESIZE_ORIG );
					break;

	case MAINMENU_VIEW_DOUBLE:	xmmplay_Scale( xmmplay, 0, 0, XMM_GRAPH_RESIZE_DOUBLE );
					break;

	case MAINMENU_VIEW_FULL:	xmmplay_Scale( xmmplay, 0, 0, XMM_GRAPH_RESIZE_FULLSCREEN );
					break;

	case MAINMENU_PLAY_PLAY:	mp_on_play_release( 0 );
					break;

	case MAINMENU_PLAY_PAUSE:	mp_on_pause_release( 0 );
					break;

	case MAINMENU_PLAY_STOP:	mp_on_stop_release( 0 );
					break;

	case MAINMENU_PLAY_REPEAT:	on_repeat_release( 0 );
					break;

	case MAINMENU_PLAY_RANDOM:	on_random_release( 0 );
					break;

	case MAINMENU_OPTIONS_PREF:	mp_PrefWin( xmmplay );
					break;

	case MAINMENU_OPTIONS_SKIN:	mplayer_SkinBrowser();
					break;

	case MAINMENU_PLAYLIST:		mp_on_playlist_release( 0 );
					break;

	case MAINMENU_EXIT:		FileClose();
					gtk_main_quit();
					break;

	default:	g_print( "MainMenu item %i selected\n", (int) data );
  }
}

//
// Widget callbacks
//

static void on_menu_release( int value )
{
  gtk_menu_popup( GTK_MENU( mainwin_mainmenu->widget ), NULL, NULL, NULL, NULL, 1, GDK_CURRENT_TIME );
}

static void on_icon_release( int value )
{
  XIconifyWindow( GDK_DISPLAY(), GDK_WINDOW_XWINDOW( mainwin_theme.window->window ), DefaultScreen( GDK_DISPLAY()));
  playlist_icon();
}

static void on_exit_release( int value )
{
  FileClose();
  gtk_main_quit();
}

static void on_mini_release( int value )
{
  cfg_DoubleSize = !cfg_DoubleSize;

  mplayer_InitTheme( NULL, cfg_DoubleSize );
  mplayer_WidgetDrawAll( &mainwin_theme );
}

static void on_eject_release( int value )
{
  mplayer_FileSelect( 0, cfg_CurrentDir, mp_FileSelectCallback );
}

// Volume control

static int balance = 0;

static void on_volume_change( int value )
{
  gchar *text;

  text = g_strdup_printf( "VOLUME: %i %%", value );
  mpwTextBoxSetText( mainwin_info, text );

  if( balance > 0 )	xmmplay_SetVolume( xmmplay, ( value * ( 100 - balance )) / 100, value );
  else	xmmplay_SetVolume( xmmplay, value, ( value * ( 100 + balance )) / 100 );
}

static void on_volume_release( int value )
{
  skinwin_restore_info();
}

// Balance control

static void on_balance_change( int value )
{
  gchar *text;
  int left, right, vol;

  balance = ( value - 50 ) << 1;
  xmmplay_GetVolume( xmmplay, &left, &right );
  vol = (( left > right ) ? left : right );

  if( balance < 0 )
  {
	text = g_strdup_printf( "BALANCE: %d%% LEFT", -balance );
	xmmplay_SetVolume( xmmplay, vol, ( vol * ( 100 + balance )) / 100 );
  }
  else if( balance > 0 )
  {
	text = g_strdup_printf( "BALANCE: %d%% RIGHT", balance );
	xmmplay_SetVolume( xmmplay, ( vol * ( 100 - balance )) / 100, vol );
  }
  else
  {
	text = g_strdup_printf( "BALANCE: CENTER" );
	xmmplay_SetVolume( xmmplay, vol, vol );
 }

  mpwTextBoxSetText( mainwin_info, text );
}

static void on_balance_release( int value )
{
  skinwin_restore_info();
}

// Position control

static void on_position_change( int value )
{
  XMM_ClipInfo	ci;
  double	position;
  gchar		*text;

  xmmplay_Info( xmmplay, NULL, &ci, NULL, NULL );
  position = ( value * ci.playtime ) / 100;

  text = g_strdup_printf( "SEEK: %i:%.2i:%.2i/%i:%.2i:%.2i ( %i%% )",
						XMMPLAY_TIME_H( position ),
						XMMPLAY_TIME_M( position ),
						XMMPLAY_TIME_S( position ),
						XMMPLAY_TIME_H( ci.playtime ),
						XMMPLAY_TIME_M( ci.playtime ),
						XMMPLAY_TIME_S( ci.playtime ),
						value );

  mpwTextBoxSetText( mainwin_info, text );
}

static void on_position_release( int value )
{
  xmmplay_Seek( xmmplay, (double)value / 100 );
  skinwin_restore_info();
}

// Repeat

static void on_repeat_release( int value )
{
  if( cfg_Repeat == 0 )	cfg_Repeat = 1;
  else	cfg_Repeat = 0;

  mpwToggleButtonState( mainwin_repeat, cfg_Repeat );
  GTK_CHECK_MENU_ITEM( gtk_item_factory_get_widget( mainwin_mainmenu,
			    MENU_STR_REPEAT ))->active = cfg_Repeat;
}

// Random

static void on_random_release( int value )
{
  if( cfg_Random == 0 )	cfg_Random = 1;
  else	cfg_Random = 0;

  mpwToggleButtonState( mainwin_random, cfg_Random );
  GTK_CHECK_MENU_ITEM( gtk_item_factory_get_widget( mainwin_mainmenu,
			    MENU_STR_RANDOM ))->active = cfg_Random;
}

// Menubar

static void on_menubar_change( int value )
{
  gchar *text;

  text = g_strdup_printf( "Menuitem %i", value );
  mpwTextBoxSetText( mainwin_info, text );
}

static void on_menubar_release( int value )
{
  skinwin_restore_info();
}

//
// Update info text
//

static void skinwin_restore_info( void )
{
  XMM_ClipInfo	ci;
  char		*ptr;

  if( xmmplay_Status( xmmplay ) > -1 )
  {
	xmmplay_Info( xmmplay, NULL, &ci, NULL, NULL );
	ptr = g_strdup_printf( "%s ( %i:%.2i )", ci.name, XMMPLAY_TIME_M( ci.playtime ), XMMPLAY_TIME_S( ci.playtime ));
	mpwTextBoxSetText( mainwin_info, ptr );
	g_free( ptr );
  }
  else	mpwTextBoxSetText( mainwin_info, "LinuX Media Player " XMM_VERSION_STRING );
}

//
// Timer proc, for updating position bar and playtime
//

static gint timer_proc( gpointer data )
{
  XMM_ControlScale	scale;
  XMM_Event		event;
  double		ctime, seekval;
  int			ret;

  if(( playing == 1 ) && ( xmmplay_Status( xmmplay ) & XMM_PBS_PLAYING ))
	playing = 3;

  if( playing != 3 )	return TRUE;

  if( xmmplay_Status( xmmplay ) & XMM_PBS_STOPPED )
  {
	if( playing )
	{
		mp_on_stop_release(0);
		mp_on_fwd_release(1);
	}
	return TRUE;
  }

  ctime = xmmplay_Info( xmmplay, NULL, NULL, NULL, &seekval );
  mpwSliderSetValue( mainwin_position, (int)( seekval * 100 ));

  mpwNumberSetValue( mainwin_time_h, XMMPLAY_TIME_H( ctime ));
  mpwNumberSetValue( mainwin_time_m, XMMPLAY_TIME_M( ctime ));
  mpwNumberSetValue( mainwin_time_s, XMMPLAY_TIME_S( ctime ));
  playlist_UpdateDisplay( XMMPLAY_TIME_H( ctime ), XMMPLAY_TIME_M( ctime ), XMMPLAY_TIME_S( ctime ));

  /* Event handling */
  if( xmm_PollEvent( XMMPLAY_2XMM(xmmplay), 0, &event ) <= 0 )	return TRUE;

  /* We have an event */
  switch( event.type )
  {
	case XMM_EVENT_KEY:
		{
		    XMM_Event_Key *key = (XMM_Event_Key *) &event;

#ifdef DEBUG
		    printf( "Key event: scancode %i ASCII %i ( '%c' ) state %i mode = %x\n", key->scode, key->sym, key->sym, key->state, key->mod );
#endif

		    if(( key->state == XMM_KEY_RELEASED ) &&
			    (( key->sym >= '1' ) && ( key->sym <= '3' )))
		    {
			xmm_logging( 1, "xmmplay: Scaling...\n" );

			/* fill XMM_ControlScale structure */
#if 0
			scale.width = width;
			scale.height = height;
#else
			scale.width = 0;
			scale.height = 0;
#endif
			scale.flags = 0;

			if( key->sym == '1' )	scale.flags = XMM_GRAPH_RESIZE_ORIG;
			if( key->sym == '2' )	scale.flags = XMM_GRAPH_RESIZE_DOUBLE;
			if( key->sym == '3' )	scale.flags = XMM_GRAPH_RESIZE_FULLSCREEN;

			ret = xmmplay_Scale( xmmplay, scale.width, scale.height, scale.flags );
			if( ret < 0 )
			{
			    fprintf( stderr, "Error resizing output: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
			}
		    }

		    break;
		}

	/* Unknown Event: return error */
	default:
		fprintf( stderr, "ERROR: Unable to handler event of type %i\n", event.type );
		break;  
  }

  return TRUE;	// Continue calling
}

//
// Open, Close file
//

//static KList	*tAudioFilterList = NULL;

static int FileOpen( char *filename )
{
  XMM_ClipInfo	ci;
  XMM_FileInfo	xmm_info;
  char		buffer[10];

  playing = 0;

  if( xmmplay_Open( xmmplay, filename, XMM_INPUT_CF_MODE_READ | XMM_INPUT_CF_MODE_DOUT ) > -1 )
  {

// Add AudioFilter
//	xmm_AudioFilterAddList( &xmmplay.xmm, iAudioFilterList );
//	tAudioFilterList = xmm_PluginRegisterDir( LIBDIR "Plugins/Visualization",  XMM_PLUGIN_TYPE_AFILTER );
//	xmm_AudioFilterAddList( &xmmplay.xmm, tAudioFilterList );
// Prepare widgets
	mplayer_WidgetVisible( mainwin_time_h, 1 );
	mplayer_WidgetVisible( mainwin_time_m, 1 );
	mplayer_WidgetVisible( mainwin_time_s, 1 );
	mplayer_WidgetVisible( mainwin_position, 1 );
	mplayer_WidgetVisible( mainwin_pmode, 1 );
	mplayer_WidgetVisible( mainwin_chmode, 1 );
	mplayer_WidgetVisible( mainwin_info_rate, 1 );
	mplayer_WidgetVisible( mainwin_info_freq, 1 );

	mpwNumberSetValue( mainwin_time_h, 0 );
	mpwNumberSetValue( mainwin_time_m, 0 );
	mpwNumberSetValue( mainwin_time_s, 0 );
	mpwSliderSetValue( mainwin_position, 0 );
	mpwStatusSetStatus( mainwin_pmode, 2 );

	xmmplay_Info( xmmplay, &xmm_info, &ci, NULL, NULL );	// Check for playtime
	if( ci.playtime == -1 )	mplayer_WidgetVisible( mainwin_position, 0 );

	sprintf( buffer, "%i", abs( xmm_info.ai[0].bitrate ) / 1000 );
	mpwTextBoxSetText( mainwin_info_rate, buffer );

	sprintf( buffer, "%i", xmm_info.ai[0].format.samprate / 1000 );
	mpwTextBoxSetText( mainwin_info_freq, buffer );

	mpwStatusSetStatus( mainwin_chmode, xmm_info.ai[0].format.channels & 1 );

	mplayer_WidgetDrawAll( &mainwin_theme );
	skinwin_restore_info();

	sleep( 1 );	// BUG-FIX
	timer = gtk_timeout_add( 100, timer_proc, mainwin_theme.window );

	return 1;
  }
  else	mplayer_Message( NULL, (char *)xmm_ErrorString( XMMPLAY_2XMM(xmmplay)));

  return 0;
}


static void FileClose( void )
{
  playing = 0;
  xmmplay_Stop( xmmplay );
  mpwStatusSetStatus( mainwin_pmode, 2 );
  xmmplay_Close( xmmplay );
//  xmm_AudioFilterRemoveAll( &xmmplay.xmm );
//  if( tAudioFilterList )
//  {
//	KListFree( tAudioFilterList );
//	tAudioFilterList = NULL;
//  }
  skinwin_restore_info();
}
