/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Dir selection dialog
 */

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include <gtk/gtk.h>
#include <libmplayer/mplayer.h>

#include "interface.h"
#include "support.h"

//
// Prototypes
//

static int has_subdirs( char *path );
static void destroy_notify( gpointer data );
static void DoCallback( char *path, int mode );

//
// Global data
//

static GtkWidget	*dir_window;
static void		(*LoadDialogCallback)( char *path, int mode );
static int		dirsel_mode;
//
// Types
//

struct DirEntry
{
    char	*path;
    int		read;
};

//
// Create dialog
//

void mplayer_DirSelect( int mode, void (*_LoadDialogCallback)( char *path, int mode ))
{
  GtkWidget		*ctree;
  GtkCTreeNode		*node, *rootnode;
  char			*text="/", *dummy = "";
  struct DirEntry	*de;

  dir_window = create_mplayer_dirselect_window();
  ctree = lookup_widget( dir_window, "dirselect_ctree" );

  rootnode = gtk_ctree_insert_node( GTK_CTREE( ctree ), NULL, NULL, &text, 2, NULL, NULL, NULL, NULL, FALSE, FALSE );
  de = g_malloc( sizeof( struct DirEntry ));
  de->path = g_strdup( "/" );
  de->read = 0;
  gtk_ctree_node_set_row_data_full( GTK_CTREE( ctree ), rootnode, de, destroy_notify );

  node = gtk_ctree_insert_node( GTK_CTREE(ctree), rootnode, NULL, &dummy, 2, NULL, NULL, NULL, NULL, TRUE, TRUE );
//  gtk_ctree_expand( GTK_CTREE(ctree), rootnode );

  gtk_widget_show( dir_window );
  LoadDialogCallback = _LoadDialogCallback;
  dirsel_mode = mode;
}

//
// Needs to be non-static, needed by interface.c
//

void on_dirselect_ctree_tree_expand( GtkCTree *ctree, GtkCTreeNode *parent, gpointer user_data )
{
  GtkCTreeNode		*node;
  char			*dummy = "", *ptr, *filename;
  DIR			*ds;
  struct dirent		*dir;
  struct stat		buf;
  int			leaf;
  struct DirEntry	*de, *pde;

  pde = (struct DirEntry *) gtk_ctree_node_get_row_data( ctree, parent );
  if( pde->read )	return;

  gtk_clist_freeze( GTK_CLIST( ctree ));

// Remove dummy node
  node = gtk_ctree_find_by_row_data( ctree, parent, NULL );
  gtk_ctree_remove_node( ctree, node );

// Get subdirs
  ds = opendir( pde->path );
  if( ds )
  {
	while(( dir = readdir( ds )) != NULL )
	{
	    if( !strcmp( dir->d_name, "." ) || !strcmp( dir->d_name, ".." ))
		    continue;

	    filename = g_malloc( strlen( pde->path ) + strlen( dir->d_name ) + 2 );
	    sprintf( filename, "%s%s%s", pde->path, ( pde->path[ strlen( pde->path ) - 1 ] != '/' ) ? "/" : "", dir->d_name );

	    if( stat( filename, &buf ) == 0 )
	    {
		    if( S_ISDIR( buf.st_mode ))
		    {
			leaf = has_subdirs( filename );

			ptr = dir->d_name;	// Need pointer
			node = gtk_ctree_insert_node( ctree, parent, NULL, &ptr, 2, NULL, NULL, NULL, NULL, !leaf, FALSE );
			de = g_malloc( sizeof( struct DirEntry ));
			de->path = g_strdup( filename );
			de->read = 0;
			gtk_ctree_node_set_row_data_full( ctree, node, de, destroy_notify );


			if( leaf )	node = gtk_ctree_insert_node( ctree, node, NULL, &dummy, 2, NULL, NULL, NULL, NULL, TRUE, TRUE );
		    }
	    }
	    g_free( filename );
	}
	closedir( ds );
  }

  pde->read = 1;
  gtk_ctree_sort_node( ctree, parent );
  gtk_clist_thaw( GTK_CLIST( ctree ));
}

void on_dirselect_ok_clicked( GtkButton *button, gpointer user_data )
{
  GtkCTree *ctree;
  GList *le;
  struct DirEntry *de;

  ctree = GTK_CTREE( lookup_widget( dir_window, "dirselect_ctree" ));

  for( le = GTK_CLIST( ctree )->selection; le; le = le->next )
  {
	de = gtk_ctree_node_get_row_data( ctree, le->data );
	DoCallback( de->path, MPLAYER_FILESEL_OK );
  }

  gtk_widget_destroy( dir_window );
}

/* clist event: select_row. We need this event, not tree_select_row */
/* I changed first argument, from 'GtkCList *clist' to 'GtkCTree *ctree'*/

void on_dirselect_ctree_select_row( GtkCTree *ctree, gint row, gint column, GdkEvent *event, gpointer user_data )
{
  GtkCTreeNode *node;
  struct DirEntry *de;

  if( event && ( event->type == GDK_2BUTTON_PRESS ))
  {
	node = gtk_ctree_node_nth( ctree, row );
	de = gtk_ctree_node_get_row_data( ctree, node );
	DoCallback( de->path, MPLAYER_FILESEL_SELECT );
  }
}

//
// Private stuff
//

static void destroy_notify( gpointer data )
{
  struct DirEntry *de = (struct DirEntry *)data;

  g_free( de->path );
  g_free( de );
}

static int has_subdirs( char *path )
{
  DIR		*ds;
  struct dirent	*dir;
  struct stat	buf;
  char		*filename;

  ds = opendir( path );
  if( ds == NULL )	return 0;

  while(( dir = readdir( ds )) != NULL )
  {
	if( !strcmp( dir->d_name, "." ) || !strcmp( dir->d_name, ".." ))
		    continue;

	filename = g_malloc( strlen( path ) + strlen( dir->d_name ) + 2 );
	sprintf( filename, "%s/%s", path, dir->d_name );

	if( stat( filename, &buf ) == 0 )
	{
	    if( S_ISDIR( buf.st_mode ))
	    {
		g_free( filename );
		closedir( ds );
		return 1;
	    }
	}
	g_free( filename );
  }

  closedir( ds );
  return 0;
}

static void DoCallback( char *path, int mode )
{
  DIR		*ds;
  struct dirent	*dir;
  struct stat	buf;
  char		*filename;

  if( dirsel_mode )
  {
	ds = opendir( path );
	if( ds == NULL )	return;

	while(( dir = readdir( ds )) != NULL )
	{
	    if( !strcmp( dir->d_name, "." ) || !strcmp( dir->d_name, ".." ))
		    continue;

	    filename = g_malloc( strlen( path ) + strlen( dir->d_name ) + 2 );
	    sprintf( filename, "%s/%s", path, dir->d_name );

	    if( stat( filename, &buf ) == 0 )
	    {
		if( S_ISDIR( buf.st_mode ))
		{
		    g_free( filename );
		    continue;
		}
	    }
	    LoadDialogCallback( filename, mode | MPLAYER_DIRSEL_ISFILE );
	    g_free( filename );
	 }
	 closedir( ds );
  }
  else	LoadDialogCallback( path, mode );
}
