/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <libxmm/libxmm.h>
#include <libxmm/util/utils.h>

#include <libmplayer/mplayerwidgets.h>
#include "gtkwin.h"
#include "parser.h"

//
// Definitions
//

enum TOPLEVEL { TL_INSECTION = 0, TL_INIT, TL_KEYWORD, TL_NAME, TL_SKIP };

//
// Prototypes
//

static int WidgetList_compare_func( gconstpointer a, gconstpointer b );

//
// Global ( nonstatic ) data
//

static GList	*theme_list = NULL;

//
// Main function for reading skin description file
//

int mplayer_LoadTheme( char *theme, char *skin, mptTheme *tdesc )
{
  FILE *stream;
  char buffer[256], option[256]="", *ptr, *ptr2, *filename, temp[256];
  mptWidget *(*parse)( mptTheme *theme, int mode, char *option, char *value );
  int line = 0, started = 0, toplevel = TL_INIT, ret = 0;

  if((( theme == NULL ) && ( skin == NULL )) || !strcmp( theme, "(default)" ))
  {
	tdesc->theme_dir[0] = '\0';
	tdesc->skin_name[0] = '\0';

	filename = xmm_gethome_filename( "tmp/def_skin" );

	if(( stream = fopen( filename, "w" )) == NULL )
	{
	    printf( "Unable to open '%s'\n", filename );
	    free( filename );
	    return 0;
	}

	fprintf( stream, "%s", *(tdesc->skin_def));
	fclose( stream );
  }
  else
  {
	strcpy( tdesc->theme_dir, theme );
	strcpy( tdesc->skin_name, skin );

	ptr = strrchr( theme, '/' );
	sprintf( temp, "%s/%s", theme,  ptr + 1 );

	filename = xmm_gethome_filename( temp );
  }

  if(( stream = fopen( filename, "r" )) == NULL )
  {
	free( filename );
	printf( "ERROR: Cannot open %s\n", filename );
	return 0;
  }

  free( filename );
  parse = NULL;

  while( 1 )
  {
    ptr = fgets( buffer, 255, stream );
    if( feof( stream ))	break;
    line++;

    if(( ptr2 = strchr( buffer, 10 )) != NULL )		*ptr2 = '\0';
    if(( ptr2 = strchr( buffer, '#' )) != NULL )	*ptr2 = '\0';

    do
    {
	while( isspace( *ptr ))	ptr++;

	if( *ptr == '\0' )	break;

	if(( toplevel == TL_INSECTION ) || ( toplevel == TL_SKIP ))
	{
		if( !strncmp( ptr, "PushButton", 10 ) && !started )
		{
		    parse = wPushButtonParse;
		    ptr += 10;
		}
		else if( !strncmp( ptr, "ToggleButton", 10 ) && !started )
		{
		    parse = wToggleButtonParse;
		    ptr += 12;
		}
		else if( !strncmp( ptr, "Slider", 6 ) && !started )
		{
		    parse = wSliderParse;
		    ptr += 6;
		}
		else if( !strncmp( ptr, "TextBox", 7 ) && !started )
		{
		    parse = wTextBoxParse;
		    ptr += 7;
		}
		else if( !strncmp( ptr, "Pixmap", 6 ) && !started )
		{
		    parse = wPixmapParse;
		    ptr += 6;
		}
		else if( !strncmp( ptr, "Number", 6 ) && !started )
		{
		    parse = wNumberParse;
		    ptr += 6;
		}
		else if( !strncmp( ptr, "Status", 6 ) && !started )
		{
		    parse = wStatusParse;
		    ptr += 6;
		}
		else if( !strncmp( ptr, "Menu", 4 ) && !started )
		{
		    parse = wMenuParse;
		    ptr += 4;
		}
		else if( !strncmp( ptr, "List", 4 ) && !started )
		{
		    parse = wListParse;
		    ptr += 4;
		}
		else if( !strncmp( ptr, "Popup", 4 ) && !started )
		{
		    parse = wPopupParse;
		    ptr += 5;
		}
		else if( !strncmp( ptr, "DrawArea", 8 ) && !started )
		{
		    parse = wDrawAreaParse;
		    ptr += 8;
		}
		else if( !strncmp( ptr, "{", 1 ))
		{
		    if( !toplevel )	parse( tdesc, WIDGET_START, option, NULL );
		    option[0] = '\0';
		    started = 1;
		    ptr++;
		}
		else if( !strncmp( ptr, "}", 1 ) && ( started == 0 ))
		{	// Section end
		    toplevel = TL_INIT;
		    ptr++;
		}
		else if( !strncmp( ptr, "}", 1 ))
		{
		    if( !toplevel )
		    {
			if( !parse( tdesc, WIDGET_STOP, 0, NULL ))
			{
			    fclose( stream );
			    return 0;
			}
		    }
		    parse = NULL;
		    started = 0;
		    ptr++;
		}
		else
		{
		    if( parse == NULL )
		    {
			g_print( "ERROR( line %i ): Unexpected: '%s'\n", line, ptr );
			return 0;
		    }

		    if( !started && *option )
		    {
			g_print( "ERROR( line %i ): '{' expected before options. \n", line );
			return 0;
		    }

		    for( ptr2 = ptr; *ptr2 && !isspace( *ptr2 ); ptr2++ );
		    if( *ptr2 == '\0' )	ptr2[1] = '\0';

		    *ptr2 = '\0';

		    if( option[0] == '\0'  )
		    {
			strcpy( option, ptr );
			ptr = ptr2 + 1;
		    }
		    else
		    {
			if( !toplevel )	parse( tdesc, WIDGET_OPTION, option, ptr );
			option[0] = '\0';
			ptr = ptr2 + 1;
		    }
		}
	}
	else
	{
	    if( !strncmp( ptr, "Section", 7 ) && ( toplevel == TL_INIT ))
	    {
		toplevel = TL_KEYWORD;
		ptr += 7;
	    }
	    else if( !strncmp( ptr, "{", 1 ) && ( toplevel == TL_KEYWORD ))
	    {
		ReadString( option, temp );

		if( !strcmp( tdesc->section, temp ))
		{
			toplevel = TL_INSECTION;
			ret = 1;
		}
		else	toplevel = TL_SKIP;

		option[0] = '\0';
		ptr++;
	    }
	    else
	    {
		if( toplevel == TL_INIT )
		{
		    g_print( "ERROR( line %i ): No widgets on top level.\n", line );
		    return 0;
		}

		for( ptr2 = ptr; *ptr2 && !isspace( *ptr2 ); ptr2++ );
		if( *ptr2 == '\0' )	ptr2[1] = '\0';

		*ptr2 = '\0';

		if( option[0] == '\0'  )
		{
		    strcpy( option, ptr );
		    ptr = ptr2 + 1;
		}
	    }
	}
    } while( *ptr != '\0' );
  }

  fclose( stream );

// Sort widget list
  tdesc->wlist = g_list_sort( tdesc->wlist, WidgetList_compare_func );

// Check if section found

  if(( ret == 0 ) && theme && skin )
  {
	ret = mplayer_LoadTheme( NULL, NULL, tdesc );

	filename = xmm_gethome_filename( "tmp/def_skin" );
	remove( filename );
	free( filename );
  }

// Final stuff ( append to theme list )

  theme_list = g_list_append( theme_list, tdesc );

  return ret;
}

//
// Free theme
//

void mplayer_FreeTheme( mptTheme *tdesc )
{
  GList		*le;
  mptWidget	*w; 
  int		i, j;

  theme_list = g_list_remove( theme_list, tdesc );

// Free Widgets

  for( le = tdesc->wlist; le != NULL; le = le->next )
  {
	w = (mptWidget *) le->data;
	if( w->Free )	w->Free( w );
	else	g_free( w );
  }
  g_list_free( tdesc->wlist );
  tdesc->wlist = NULL;

// Free pixmaps and Re-Initialize widget table

  for( i = 0; i < tdesc->wnum; i++ )
  {
    if( tdesc->wtable[i].pixmap )
    {
	gdk_pixmap_unref( tdesc->wtable[i].pixmap );
	for( j = 0; j < tdesc->wnum; j++ )
	    if( tdesc->wtable[j].pixmap == tdesc->wtable[i].pixmap )
		    tdesc->wtable[j].pixmap = NULL;

	tdesc->wtable[i].pixmap = NULL;
	tdesc->wtable[i].pixmapfile[0] = '\0';
    }
    tdesc->wtable[i].pixmapfile[0] = '\0';
 }

 if( tdesc->mask_ds )	gdk_bitmap_unref( tdesc->mask_ds );
 tdesc->mask_ds = NULL;
 if( tdesc->pixmap )	gdk_pixmap_unref( tdesc->pixmap );
 tdesc->pixmap = NULL;
}

//
// Initialize ( resize ) theme windows
//

void mplayer_InitTheme( mptTheme *tdesc, int state )
{
  GList		*le;

  if( tdesc )	mpint_InitTheme( tdesc, state );
  else
  {
	for( le = theme_list; le; le = le->next )
		mpint_InitTheme( (mptTheme *)le->data, state );
  }
}

//
// Draw complete widget list
//

void mplayer_WidgetDrawAll( mptTheme *tdesc )
{
  GList		*le;
  mptWidget	*w; 

  for( le = tdesc->wlist; le != NULL; le = le->next )
  {
	w = (mptWidget *) le->data;
	if( w->visible )	w->cbDraw( w );
  }
  mpint_RefreshTheme( tdesc );
}

//
// Toggle widget visible
//

void mplayer_WidgetVisible( void *w, int state )
{
  ((mptWidget *)w)->visible = state;
}

//
// libmplayer internal code ( reload theme )
//

int mplayer_ReloadDock( char *theme, char *skin )
{
  GList		*le;
  mptEvent	event;
  mptTheme	*tdesc;
  GList		*tdesc_list = NULL;
  
//
// theme_list is modified in mplayer_FreeTheme()
// so we have to remove the first theme in the list until none remains
// the theme descriptions are save in a new list
//

  while( theme_list != NULL )
  {
	tdesc_list = g_list_append( tdesc_list, (mptTheme *)theme_list->data );
	mplayer_FreeTheme((mptTheme *)theme_list->data );
  }

//
// Now we (re)load each theme in the list
//

  for( le = tdesc_list; le; le = le->next )
  {
	tdesc = (mptTheme *)le->data;

	mplayer_LoadTheme( theme, skin, tdesc );
	mplayer_InitTheme( tdesc, tdesc->doubleState );

	if( tdesc->visible )	mplayer_WidgetDrawAll( tdesc );

	if( tdesc->cbMPlayerEvent )
	{
		event.type = MPLAYER_EVENT_THEMECHANGE;
		strcpy( event.data.theme.theme, theme );
		strcpy( event.data.theme.skin, skin );
		tdesc->cbMPlayerEvent( tdesc, NULL, &event );
	}
  }

  return 1;
}

//
// Private code ( Sort widget list )
//

static int WidgetList_compare_func( gconstpointer a, gconstpointer b )
{
  int i = 0, j = 0;
  mptTheme *theme = ((mptWidget *) a)->theme;

  while(( i < theme->wnum ) && (*theme->wtable[i].pWidget != a ))	i++;
  while(( j < theme->wnum ) && (*theme->wtable[j].pWidget != b ))	j++;

  if( i == j )	return 0;
  if( i > j )	return 1;
  else		return -1;
}
