/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

//
// TextBox
//

#include <gtk/gtk.h>
#include <libmplayer/mplayerwidgets.h>
#include "gtkwin.h"

//
// Draw callback
//

static void wTextBoxDraw( mptWidget *widget )
{
  mptTextBox *tb = (mptTextBox *) widget;
  int dx;

  if( !widget->visible )	return;

  dx = tb->PixmapWidth - tb->TextOffset;
  if( dx > tb->sys.width )	dx = tb->sys.width;

  gdk_draw_pixmap(	tb->sys.window,	tb->sys.gc, tb->Pixmap, tb->TextOffset,
			0, tb->sys.x, tb->sys.y, dx, tb->sys.height );

  if( dx < tb->sys.width )
    gdk_draw_pixmap(	tb->sys.window,	tb->sys.gc, tb->Pixmap, 0, 0,
			tb->sys.x + dx, tb->sys.y,
			tb->sys.width - dx, tb->sys.height );
}

static int wTextBoxScroll( gpointer widget )
{
  mptTextBox *tb = (mptTextBox *) widget;

  if( !tb->pressed )
  {
	tb->TextOffset += 2;
	if( tb->TextOffset > tb->PixmapWidth )	tb->TextOffset -= tb->PixmapWidth;
	tb->sys.cbDraw( (mptWidget *)tb );
	mpint_RefreshTheme( tb->sys.theme );
  }
  return 1;
}

//
// Button press callback
//

static int wTextBox_button_pressed( GtkWidget *gtkwidget, GdkEventButton *event, mptWidget *widget )
{
  mptTextBox *tb = (mptTextBox *) widget;

  tb->pressed = 1;
  tb->drag_start = tb->TextOffset;
  tb->drag_x = event->x;
  return 1;
}

//
// Button release callback
//

static int wTextBox_button_released( GtkWidget *gtkwidget, GdkEventButton *event, mptWidget *widget )
{
  mptTextBox *tb = (mptTextBox *) widget;

  if( !tb->pressed )	return 1;
  tb->pressed = 0;

  return 1;
}

//
// Motion callback
//

static int wTextBox_motion( GtkWidget *gtkwidget, GdkEventButton *event, mptWidget *widget )
{
  mptTextBox *tb = (mptTextBox *) widget;

  if( !tb->pressed )	return 1;

  if( tb->PixmapWidth > tb->sys.width )
  {
	tb->TextOffset = tb->drag_start - ( event->x - tb->drag_x );

	while( tb->TextOffset < 0 )	tb->TextOffset += tb->PixmapWidth;
	while( tb->TextOffset > tb->PixmapWidth )	tb->TextOffset -= tb->PixmapWidth;

	tb->sys.cbDraw( (mptWidget *)tb );
	mpint_RefreshTheme( tb->sys.theme );
  }

  return 1;
}

//
// Free textbox
//

static void wTextBoxFree( mptWidget *widget )
{
  mptTextBox *tb = (mptTextBox *)widget;

  if( tb->Text )	g_free( tb->Text );
  if( tb->Pixmap )	gdk_pixmap_unref( tb->Pixmap );
  if( tb->Timer )	gtk_timeout_remove( tb->Timer );
  g_free( tb );
}

//
// Parse and create TextBox
//

#include <string.h>
#include <ctype.h>
#include "parser.h"

static char		name[256];
static char		pixmapfile[256];
static mptTextBox		*tb;

mptWidget *wTextBoxParse( mptTheme *theme, int mode, char *option, char *value )
{
  if( mode == WIDGET_START )
  {
	tb = (mptTextBox *) g_malloc( sizeof( mptTextBox ));

	tb->sys.cbDraw = wTextBoxDraw;
	tb->sys.cbButtonPress = wTextBox_button_pressed;
	tb->sys.cbButtonRelease = wTextBox_button_released;
	tb->sys.cbMotion = wTextBox_motion;
	tb->sys.Free = wTextBoxFree;
	tb->pressed = 0;
	tb->WinAmp = 0;
	tb->Text = NULL;
	tb->Timer = 0;
	tb->Pixmap = NULL;

	if( !ReadString( option, name ))	strcpy( name, option );
  }

  if( mode == WIDGET_STOP )
  {
	if( !LookupWidget( theme, name, pixmapfile, (mptWidget *)tb ))	return NULL;
	mpwTextBoxSetText( tb, "" );
	return (mptWidget *) tb;
  }
  if( mode == WIDGET_OPTION )
  {
	if( !strcasecmp( option, "Position" ))	ReadCoord( value, &tb->sys.x, &tb->sys.y );
	if( !strcasecmp( option, "Size" ))	ReadCoord( value, &tb->sys.width, &tb->sys.height );
	if( !strcasecmp( option, "Filename" ))	ReadString( value, pixmapfile );
	if( !strcasecmp( option, "Offset" ))	ReadCoord( value, &tb->sx, &tb->sy );
	if( !strcasecmp( option, "FontSize" ))	ReadCoord( value, &tb->FontWidth, &tb->FontHeight );
	if( !strcasecmp( option, "Scroll" ))	ReadOrd( value, &tb->scroll );
	if( !strcasecmp( option, "WinAmp" ))	ReadOrd( value, &tb->WinAmp );
  }

  return NULL;
}

//
// Char table, contains positions for winamp skins
//

#define	UX	145
#define	UY	0

short char_table[512] =
{
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 00 .. 07
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 08 .. 0F
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 10 .. 17
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 18 .. 1F
    145, 0, 85, 6, 130, 0, 150, 6, 145, 6, 130, 6, 125, 6, 80, 6,	// 20 .. 27
    65, 6, 70, 6, 20, 12, 95, 6, 135, 6, 75, 6, 135, 6, 105, 6,		// 28 .. 2F
    0, 6, 5, 6, 10, 6, 15, 6, 20, 6, 25, 6, 30, 6, 35, 6,		// 30 .. 37
    40, 6, 45, 6, 60, 6, UX, UY, UX, UY, 140, 6, UX, UY, 15, 12,	// 38 .. 3F
    UX, UY, 0, 0, 5, 0, 10, 0, 15, 0, 20, 0, 25, 0, 30, 0,		// 40 .. 47
    35, 0, 40, 0, 45, 0, 50, 0, 55, 0, 60, 0, 65, 0, 70, 0,		// 48 .. 4F
    75, 0, 80, 0, 85, 0, 90, 0, 95, 0, 100, 0, 105, 0, 110, 0,		// 50 .. 57
    115, 0, 120, 0, 125, 0, 110, 6, 100, 6, 115, 6, 120, 6, 90, 6,	// 58 .. 5F
    80, 6, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 60 .. 67
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 68 .. 6F
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 70 .. 77
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 78 .. 7F
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 80 .. 87
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 88 .. 8F
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 90 .. 97
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// 98 .. 9F
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// A0 .. A7
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// A8 .. AF
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// B0 .. B7
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// B8 .. BF
    UX, UY, UX, UY, UX, UY, UX, UY, 10, 12, 0, 12, UX, UY, UX, UY,	// C0 .. C7
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// C8 .. CF
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, 5, 12, UX, UY,	// D0 .. D7
    UX, UY, UX, UY, UX, UY, UX, UY, 100, 0, UX, UY, UX, UY, UX, UY,	// D8 .. DF
    UX, UY, UX, UY, UX, UY, UX, UY, 10, 12, 0, 12, UX, UY, UX, UY,	// E0 .. E7
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY,	// E8 .. EF
    UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, UX, UY, 5, 12, UX, UY,	// F0 .. F7
    UX, UY, UX, UY, UX, UY, UX, UY, 100, 0, UX, UY, UX, UY, UX, UY	// F8 .. FF
};

int mpwTextBoxSetText( mptTextBox *tb, char *text )
{
  int len, size, i, x, y, c;

  if( tb->Text )	g_free( tb->Text );
  if( tb->Pixmap )	gdk_pixmap_unref( tb->Pixmap );
  if( tb->Timer )	gtk_timeout_remove( tb->Timer );

  tb->Timer = 0;
  tb->TextOffset = 0;

  len = strlen( text );
  size = tb->sys.width / tb->FontWidth;
  if(( size * tb->FontWidth ) != tb->sys.width )	size++;

  if( len <= size )
  {
	tb->Text = g_malloc( size + 1 );
	strcpy( tb->Text, text );
	tb->PixmapWidth = size * tb->FontWidth;
	for( ; len < size; len++ )	strcat( tb->Text, " " );
  }
  else
  {
	tb->Text = g_malloc( len + 8 );
	strcpy( tb->Text, text );
	strcat( tb->Text, "  ***  " );
	len += 7;
	tb->PixmapWidth = len * tb->FontWidth;
  }

  tb->Pixmap = gdk_pixmap_new( tb->sys.window, tb->PixmapWidth,
                            	tb->FontHeight, gdk_rgb_get_visual()->depth );

  for( i = 0; i < len; i++ )
  {
	if( tb->WinAmp )
	{
		c = toupper( tb->Text[i] );
		x = char_table[ c * 2 ];
		y = char_table[ c * 2 + 1 ];
	}
	else
	{
		y = (( tb->Text[i] - 32 ) / 16 ) * tb->FontHeight;
		x = (( tb->Text[i] - 32 ) % 16 ) * tb->FontWidth;
	}

	gdk_draw_pixmap( tb->Pixmap, tb->sys.gc,
			tb->sys.pixmap, x + tb->sx, y + tb->sy,
			i * tb->FontWidth, 0, tb->FontWidth, tb->FontHeight );
  }

  if( !tb->Timer && tb->scroll && ( tb->PixmapWidth > ( tb->sys.width / tb->FontWidth + 1 ) * tb->FontWidth ))
			tb->Timer = gtk_timeout_add( 100, wTextBoxScroll, tb );

  tb->sys.cbDraw( (mptWidget *)tb );
  mpint_RefreshTheme( tb->sys.theme );

  return 1;
}
