/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Clip Information
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <libxmm/libxmm.h>
#include <libxmm/util/utils.h>
#include <libxmmplay/xmmplay.h>

#include <gtk/gtk.h>
#include "interface.h"
#include <libmplayer/support.h>

#include "clipinfo.h"

/*
 * The info window callbacks need the priv struct,
 * however there is no way to pass it directly to the functions.
 */

struct win_lookup_entry
{
    GtkWidget			*widg;
    void			*xmmplay;
    struct win_lookup_entry	*next;
};


static struct win_lookup_entry *win_lookup_list = NULL;

/*
 * Create window
 */
void display_mplayer_clipinfo( char *filename, XMM_ClipInfo *ci, void *xmmplay )
{
  GtkWidget			*info_window;
  char				temp[256];
  struct win_lookup_entry	*le, *last;

  /* Create window */
  info_window = create_window_clipinfo();

  /*
   * Add widget to lookup list
   */

  if(( le = malloc( sizeof( struct win_lookup_entry ))) == NULL )
  {
	xmm_logging( 1, "mplayer/clipinfo! ERROR allocating memory\n" );
	return;
  }

  /* Initialize data */
  le->next = NULL;
  le->widg = info_window;
  le->xmmplay = xmmplay;

  /* Append to list */
  if( win_lookup_list == NULL )	win_lookup_list = le;
  else
  {
	for( last = win_lookup_list; last->next; last = last->next );
	last->next = le;
  }

  /* Date string */
  sprintf( temp, "%c%c.%c%c.%c%c%c%c\n",	ci->date[0], ci->date[1],
						ci->date[2], ci->date[3],
						ci->date[4], ci->date[5],
						ci->date[6], ci->date[7] );

  /* Set entries */
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window, "entry_filename" )), filename );
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window, "entry_name" )), ci->name );
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window, "entry_author" )), ci->author );
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window, "entry_album" )), ci->album );
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window, "entry_content" )), ci->content );
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window, "entry_copyright" )), ci->copyright );
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window, "entry_software" )), ci->software );
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window, "entry_date" )), temp );
  gtk_text_insert( GTK_TEXT( lookup_widget( info_window, "text_comment" )), NULL, NULL, NULL, ci->comment, -1 );

  /* set editable */
  gtk_editable_set_editable( GTK_EDITABLE( lookup_widget( info_window, "entry_name" )), ci->flags & XMM_CIF_NAME ? TRUE : FALSE );
  gtk_editable_set_editable( GTK_EDITABLE( lookup_widget( info_window, "entry_author" )), ci->flags & XMM_CIF_AUTHOR ? TRUE : FALSE );
  gtk_editable_set_editable( GTK_EDITABLE( lookup_widget( info_window, "entry_album" )), ci->flags & XMM_CIF_ALBUM ? TRUE : FALSE );
  gtk_editable_set_editable( GTK_EDITABLE( lookup_widget( info_window, "entry_content" )), ci->flags & XMM_CIF_CONTENT ? TRUE : FALSE );
  gtk_editable_set_editable( GTK_EDITABLE( lookup_widget( info_window, "entry_copyright" )), ci->flags & XMM_CIF_COPYRIGHT ? TRUE : FALSE );
  gtk_editable_set_editable( GTK_EDITABLE( lookup_widget( info_window, "entry_software" )), ci->flags & XMM_CIF_SOFTWARE ? TRUE : FALSE );
  gtk_editable_set_editable( GTK_EDITABLE( lookup_widget( info_window, "entry_date" )), ci->flags & XMM_CIF_DATE ? TRUE : FALSE );
  gtk_text_set_editable( GTK_TEXT( lookup_widget( info_window, "text_comment" )), ci->flags & XMM_CIF_COMMENT ? TRUE : FALSE );

  /* Show window */
  gtk_widget_show( info_window );
}

/*
 * Callbacks
 */

void on_button_save_clicked( GtkWidget *info_window, gpointer user_data )
{
  struct win_lookup_entry	*le, **prev;
  void				*xmmplay = NULL;
  XMM_ClipInfo			ci;
  char				*ptr, *cptr;

  /* Find entry in lookup list */
  if( win_lookup_list )
  {
	for( le = win_lookup_list, prev = &win_lookup_list; le; le = le->next )
	{
		if( le->widg == info_window )
		{
			xmmplay = le->xmmplay;
			*prev = le->next;
			free( le );
			break;
		}

		prev = &le->next;
	}
  }

  /* Set clip info */
  if( xmmplay )
  {
	memset( &ci, 0, sizeof( XMM_ClipInfo ));

	ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window, "entry_name" )));
	strncpy( ci.name, ptr, XMM_CIL_NAME - 1 );
	ci.name[XMM_CIL_NAME - 1] = '\0';
	ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window, "entry_author" )));
	strncpy( ci.author, ptr, XMM_CIL_AUTHOR - 1 );
	ci.author[XMM_CIL_AUTHOR - 1] = '\0';
	ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window, "entry_album" )));
	strncpy( ci.album, ptr, XMM_CIL_ALBUM - 1 );
	ci.album[XMM_CIL_ALBUM - 1] = '\0';
	ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window, "entry_content" )));
	strncpy( ci.content, ptr, XMM_CIL_CONTENT - 1 );
	ci.content[XMM_CIL_CONTENT - 1] = '\0';
	ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window, "entry_copyright" )));
	strncpy( ci.copyright, ptr, XMM_CIL_COPYRIGHT - 1 );
	ci.copyright[XMM_CIL_COPYRIGHT - 1] = '\0';
	ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window, "entry_software" )));
	strncpy( ci.software, ptr, XMM_CIL_SOFTWARE - 1 );
	ci.software[XMM_CIL_SOFTWARE - 1] = '\0';

	ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window, "entry_date" )));
	cptr = strchr( ptr, '.' );
	if( cptr == NULL )
	{
	    memset( ci.date, '0', 4 );
	    strncpy( &ci.date[4], ptr, XMM_CIL_DATE - 5 );
	    ci.date[XMM_CIL_DATE - 1] = '\0';
	}
	else
	{
	    if( isdigit(ptr[0]) && isdigit(ptr[1]) && ( ptr[2] == '.' ) &&
		isdigit(ptr[3]) && isdigit(ptr[4]) && ( ptr[5] == '.' ) &&
		isdigit(ptr[6]) && isdigit(ptr[7]) &&
		isdigit(ptr[8]) && isdigit(ptr[9]) )
	    {
		ci.date[0] = ptr[0];	/* Day */
		ci.date[1] = ptr[1];
		ci.date[2] = ptr[3];	/* Month */
		ci.date[3] = ptr[4];
		ci.date[4] = ptr[6];	/* Year */
		ci.date[5] = ptr[7];
		ci.date[6] = ptr[8];
		ci.date[7] = ptr[9];
	    }
	    else	ci.date[0] = '\0';
	}

	ptr = gtk_editable_get_chars( GTK_EDITABLE( lookup_widget( info_window, "text_comment" )), 0, -1 );
	strncpy( ci.comment, ptr, XMM_CIL_COMMENT - 1 );
	ci.comment[XMM_CIL_COMMENT - 1] = '\0';

	xmmplay_InputSetClipInfo( xmmplay, &ci );
  }

  /* Close window */
  gtk_widget_destroy( info_window );
}

/*
 * Callback for removing clip info
 */
void on_button_remove_clicked( GtkWidget *info_window, gpointer user_data )
{
  struct win_lookup_entry	*le, **prev;
  void				*xmmplay = NULL;

  /* Find entry in lookup list */
  if( win_lookup_list )
  {
	for( le = win_lookup_list, prev = &win_lookup_list; le; le = le->next )
	{
		if( le->widg == info_window )
		{
			xmmplay = le->xmmplay;
			*prev = le->next;
			free( le );
			break;
		}

		prev = &le->next;
	}
  }

  /* Remove clip info */
  if( xmmplay )	xmmplay_InputSetClipInfo( xmmplay, NULL );

  /* Close window */
  gtk_widget_destroy( info_window );
}
