/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Playlist window
 */

#include <stdio.h>
#include <string.h>

#include <gtk/gtk.h>

#include <libmplayer/mplayer.h>
#include <libxmm/util/list.h>
#include <libxmm/util/utils.h>

#include "main.h"
#include "deftheme.h"

#include <gdk/gdkx.h>

//
// Definitions
//

#define	DEFAULT_PLAYLIST	"mplayer.m3u"	// Default playlist

#define	PLWIN_WIDGETS		23		// Widgets

#define FILEMISC_ITEMS		1		// Popup menus
#define MISCSORT_ITEMS		6
#define MISCINFO_ITEMS		2
#define MISCOPTS_ITEMS		1

enum				// Codes for menucallback
{
    PLWIN_REMOVEDEAD, 
    PLWIN_SORTLIST_TITLE, PLWIN_SORTLIST_FILE, PLWIN_SORTLIST_PATH,
    PLWIN_SORTLIST_REVERSE, PLWIN_SORTLIST_RANDOMIZE,
    PLWIN_FILEINFO, PLWIN_PLENTRY,
    PLWIN_HTMLPLAYLIST
};

//
// Prototypes ( needed by global data )
//

static void plwin_menucallback( GtkWidget *widget, gpointer data );
static void mplayer_callback( mptTheme *theme, mptWidget *widget, mptEvent *event );

static void on_pl_add_release( int value );
static void on_pl_remove_release( int value );
static void on_pl_select_release( int value );
static void on_pl_misc_release( int value );
static void on_pl_list_release( int value );
static void on_pl_eject_release( int value );

static void on_menu_release( int value );
static void on_icon_release( int value );
static void on_mini_release( int value );

//
// Global data
//

static mptPixmap	*plwin_back;
static mptStatus	*plwin_titlebar;
static mptList		*plwin_playlist;
static mptPopup		*plwin_pu_add;
static mptPopup		*plwin_pu_remove;
static mptPopup		*plwin_pu_select;
static mptPopup		*plwin_pu_misc;
static mptPopup		*plwin_pu_list;
static mptPushButton	*plwin_rew;
static mptPushButton	*plwin_play;
static mptPushButton	*plwin_pause;
static mptPushButton	*plwin_stop;
static mptPushButton	*plwin_fwd;
static mptPushButton	*plwin_eject;
static mptTextBox	*plwin_info;
static mptTextBox	*plwin_time_h;
static mptTextBox	*plwin_time_m;
static mptTextBox	*plwin_time_s;
static mptPushButton	*plwin_mini;
static mptPushButton	*plwin_icon;
static mptPushButton	*plwin_menu;
static mptPushButton	*plwin_exit;
static mptSlider	*plwin_listpos;

// Popup menus

static GtkItemFactory	*plwin_filemisc;
static GtkItemFactory	*plwin_miscsort;
static GtkItemFactory	*plwin_miscinfo;
static GtkItemFactory	*plwin_miscopts;

//
// Widget set
//

static mptWidgetEntry plwin_widgets[PLWIN_WIDGETS] =
{
 { "pl_back",		(mptWidget **)&plwin_back,	deftheme_plmain_xpm },
 { "pl_titlebar",	(mptWidget **)&plwin_titlebar,	deftheme_pltitle_xpm },

 { "pl_menu",		(mptWidget **)&plwin_menu,	deftheme_pltitle_xpm, on_menu_release },
 { "pl_icon",		(mptWidget **)&plwin_icon,	deftheme_pltitle_xpm, on_icon_release },
 { "pl_mini",		(mptWidget **)&plwin_mini,	deftheme_pltitle_xpm, on_mini_release },
 { "pl_exit",		(mptWidget **)&plwin_exit,	deftheme_pltitle_xpm, mp_on_playlist_release },

 { "pl_rew",		(mptWidget **)&plwin_rew,	deftheme_plmain_xpm, mp_on_rew_release },
 { "pl_play",		(mptWidget **)&plwin_play,	deftheme_plmain_xpm, mp_on_play_release },
 { "pl_pause",		(mptWidget **)&plwin_pause,	deftheme_plmain_xpm, mp_on_pause_release },
 { "pl_stop",		(mptWidget **)&plwin_stop,	deftheme_plmain_xpm, mp_on_stop_release },
 { "pl_fwd",		(mptWidget **)&plwin_fwd,	deftheme_plmain_xpm, mp_on_fwd_release },
 { "pl_eject",		(mptWidget **)&plwin_eject,	deftheme_plmain_xpm, on_pl_eject_release },

 { "pl_list",		(mptWidget **)&plwin_playlist,	deftheme_pltitle_xpm, playlist_SetSelected, playlist_MoveTo },
 { "pl_listpos",	(mptWidget **)&plwin_listpos,	deftheme_plslider_xpm },

 { "pl_pu_add",		(mptWidget **)&plwin_pu_add,	deftheme_plpopup_xpm, on_pl_add_release },
 { "pl_pu_remove",	(mptWidget **)&plwin_pu_remove,	deftheme_plpopup_xpm, on_pl_remove_release },
 { "pl_pu_select",	(mptWidget **)&plwin_pu_select,	deftheme_plpopup_xpm, on_pl_select_release },
 { "pl_pu_misc",	(mptWidget **)&plwin_pu_misc,	deftheme_plpopup_xpm, on_pl_misc_release },
 { "pl_pu_list",	(mptWidget **)&plwin_pu_list,	deftheme_plpopup_xpm, on_pl_list_release },

 { "pl_info",		(mptWidget **)&plwin_info,	deftheme_font_xpm },
 { "pl_time_h",		(mptWidget **)&plwin_time_h,	deftheme_font_xpm },
 { "pl_time_min",	(mptWidget **)&plwin_time_m,	deftheme_font_xpm },
 { "pl_time_sec",	(mptWidget **)&plwin_time_s,	deftheme_font_xpm },
};

//
// Theme
//

static mptTheme plwin_theme =
{
	plwin_widgets, PLWIN_WIDGETS, "PlaylistWin", &deftheme_desc, 0,
	"LinuX Media Player - Playlist", mplayer_callback
};

//
// Popup menus
//

static GtkItemFactoryEntry plwin_filemisc_items[] =
{
    { "/Remove all dead files",		NULL,	plwin_menucallback, PLWIN_REMOVEDEAD, NULL }
};


static GtkItemFactoryEntry plwin_miscsort_items[] =
{
    { "/Sort list by title",			NULL,	plwin_menucallback, PLWIN_SORTLIST_TITLE, NULL },
    { "/Sort list by filename",			NULL,	plwin_menucallback, PLWIN_SORTLIST_FILE, NULL },
    { "/Sort list by path and filename",	NULL,	plwin_menucallback, PLWIN_SORTLIST_PATH, NULL },
    { "/sep1",					NULL,	NULL, 0, "<Separator>" },
    { "/Reverse list",				NULL,	plwin_menucallback, PLWIN_SORTLIST_REVERSE, NULL },
    { "/Randomize list",			NULL,	plwin_menucallback, PLWIN_SORTLIST_RANDOMIZE, NULL },
};


static GtkItemFactoryEntry plwin_miscinfo_items[] =
{
    { "/File info...",			NULL,	plwin_menucallback, PLWIN_FILEINFO, NULL },
    { "/Playlist entry...",		NULL,	plwin_menucallback, PLWIN_PLENTRY, NULL },
};


static GtkItemFactoryEntry plwin_miscopts_items[] =
{
    { "/Generate HTML playlist",	NULL,	plwin_menucallback, PLWIN_HTMLPLAYLIST, NULL },
};

//
// Public code
//

void playlist_show( void )
{
  cfg_ShowPlaylist = 1;
  mplayer_ShowWindow( &plwin_theme );

  mpwTextBoxSetText( plwin_info, "LinuX Media Player" );
  mpwTextBoxSetText( plwin_time_h, "" );
  mpwTextBoxSetText( plwin_time_m, "" );
  mpwTextBoxSetText( plwin_time_s, "" );
}

void playlist_hide( void )
{
  cfg_ShowPlaylist = 0;
  mplayer_HideWindow( &plwin_theme );
}

void playlist_icon( void )
{
  on_icon_release( 0 );
}

//
// Updqtes time display
//

void playlist_UpdateDisplay( int h, int m, int s )
{
  char buffer[3];

  sprintf( buffer, "%1i", h );
  mpwTextBoxSetText( plwin_time_h, buffer );
  sprintf( buffer, "%.2i", m );
  mpwTextBoxSetText( plwin_time_m, buffer );
  sprintf( buffer, "%.2i", s );
  mpwTextBoxSetText( plwin_time_s, buffer );
}

//
// Private data ( event handler )
//

static void mplayer_callback( mptTheme *theme, mptWidget *widget, mptEvent *event )
{
    switch( event->type )
    {
	case	MPLAYER_EVENT_FOCUS_IN:
		    mpwStatusSetStatus( plwin_titlebar, 0 );
		    break;

	case	MPLAYER_EVENT_FOCUS_OUT:
		    mpwStatusSetStatus( plwin_titlebar, 1 );
		    break;

	case	MPLAYER_EVENT_BUTTON_PRESS:
		    if( widget == (mptWidget *)plwin_titlebar )
			    gdk_window_raise( plwin_theme.window->window );
		    break;

	case	MPLAYER_EVENT_BUTTON_RELEASE:
		    break;

	case	MPLAYER_EVENT_MOTION:
		    break;

	case	MPLAYER_EVENT_THEMECHANGE:
		    break;
    }
}

//
// Widget handler
//

static void on_menu_release( int value )
{
}

static void on_icon_release( int value )
{
  XIconifyWindow( GDK_DISPLAY(), GDK_WINDOW_XWINDOW( plwin_theme.window->window ), DefaultScreen( GDK_DISPLAY()));
}


static void on_mini_release( int value )
{
}


//
// Misc stuff needed by playlist popup callback
//

static void fileselect_savelist( char *filename, int mode )
{
  if( filename && *filename )	playlist_Save( filename );
}

static void fileselect_loadlist( char *filename, int mode )
{
  if( filename && *filename )	playlist_Load( filename );
}

static void DirSelectCallback( char *path, int mode )
{
  if( mode & MPLAYER_DIRSEL_ISFILE )
  {
	mp_FileSelectCallback( path, MPLAYER_FILESEL_SELECT );
  }
}

static void UrlSelectCallback( char *url )
{
  XMM_List *le;

  for( le = cfg_URLHistory; le; le = le->next )
	if( !strcmp( url, (char *)le->data ))	return;

  cfg_URLHistory = xmmList_Append( cfg_URLHistory, strdup( url ));

  mp_FileSelectCallback( url, MPLAYER_FILESEL_SELECT );
}

//
// Playlist popup
//

static void on_pl_add_release( int value )
{
  switch( value )
  {
	case 0:	mplayer_UrlSelect( (GList *)cfg_URLHistory, UrlSelectCallback );
		break;

	case 1:	mplayer_DirSelect( 1, DirSelectCallback );
		break;

	case 2:	mplayer_FileSelect( 1, cfg_CurrentDir, mp_FileSelectCallback );
		break;
  }
}

static void on_pl_remove_release( int value )
{
  switch( value )
  {
	case 0:	gtk_menu_popup( GTK_MENU( plwin_filemisc->widget ), NULL, NULL, NULL, NULL, 1, GDK_CURRENT_TIME );
		break;

	case 1:	playlist_Clear();
		break;

	case 2:	playlist_CropSelected();
		break;

	case 3:	playlist_RemoveSelected();
		break;
  }
}

static void on_pl_select_release( int value )
{
  switch( value )
  {
	case 0:	playlist_SelectInverse();
		break;

	case 1:	playlist_SelectNone();
		break;

	case 2:	playlist_SelectAll();
		break;
  }
}

static void on_pl_misc_release( int value )
{
  switch( value )
  {
	case 0:	gtk_menu_popup( GTK_MENU( plwin_miscsort->widget ), NULL, NULL, NULL, NULL, 1, GDK_CURRENT_TIME );
		break;

	case 1:	gtk_menu_popup( GTK_MENU( plwin_miscinfo->widget ), NULL, NULL, NULL, NULL, 1, GDK_CURRENT_TIME );
		break;

	case 2:	gtk_menu_popup( GTK_MENU( plwin_miscopts->widget ), NULL, NULL, NULL, NULL, 1, GDK_CURRENT_TIME );
		break;
  }
}

static void on_pl_list_release( int value )
{
  switch( value )
  {
	case 0:	playlist_Clear();
		break;

	case 1:	mplayer_FileSelect( 0, cfg_CurrentDir, fileselect_savelist );
		break;

	case 2:	mplayer_FileSelect( 0, cfg_CurrentDir, fileselect_loadlist );
		break;
  }
}

static void on_pl_eject_release( int value )
{
  mplayer_FileSelect( 1, cfg_CurrentDir, mp_FileSelectCallback );
}

//
// Popup menus for playlist popups
//

static void plwin_menucallback( GtkWidget *widget, gpointer data )
{
  switch( (int) data )
  {
  }
}

//
// Playlist functions
//
// for manupulating the playlist
//

//
// Playlist entry
//

typedef struct _playlistentry
{
    char	*text;
    char	*file;
} PlaylistEntry;

//
// Global data
//

static GList	*playlist;
static int	playIdx = 0;

extern mptList	*plwin_playlist;


void playlist_SetSelected( int value )
{
  if( value > MPLAYER_LIST_DOUBLE_CLICK )
	playIdx = value - MPLAYER_LIST_DOUBLE_CLICK;

//  printf( "playIdx = %i\n", playIdx );
}

void playlist_MoveTo( int value )
{
  
}

//
// Select playlist entries
//

void playlist_SelectAll( void )
{
  GList		*le;
  int		n;

  for( le = playlist, n = 0; le; le = le->next, n++ )
	mpwListSelectionSet( plwin_playlist, n, 1 );
}

void playlist_SelectNone( void )
{
  GList		*le;
  int		n;

  for( le = playlist, n = 0; le; le = le->next, n++ )
	mpwListSelectionSet( plwin_playlist, n, 0 );
}

void playlist_SelectInverse( void )
{
  GList		*le;
  int		n;

  for( le = playlist, n = 0; le; le = le->next, n++ )
  {
	if( mpwListSelectionGet( plwin_playlist, n ))	mpwListSelectionSet( plwin_playlist, n, 0 );
	else	mpwListSelectionSet( plwin_playlist, n, 1 );
  }
}

//
// Modify playlist
//

void playlist_RemoveSelected( void )
{
  GList		*lec, *le = playlist;
  int		n = 0;

  while( le )	if( mpwListSelectionGet( plwin_playlist, n ))
		{
		    lec = le;
		    le = le->next;

		    g_free((PlaylistEntry *)lec->data );
		    playlist = g_list_remove_link( playlist, lec );

		    mpwListRemove( plwin_playlist, n );
		}
		else
		{
		    le = le->next;
		    n++;
		}
}

void playlist_CropSelected( void )
{
  GList		*lec, *le = playlist;
  int		n = 0;

  while( le )	if( mpwListSelectionGet( plwin_playlist, n ) == 0 )
		{
		    lec = le;
		    le = le->next;

		    g_free((PlaylistEntry *)lec->data );
		    playlist = g_list_remove_link( playlist, lec );

		    mpwListRemove( plwin_playlist, n );
		}
		else
		{
		    le = le->next;
		    n++;
		}
}

void playlist_Clear( void )
{
  GList *le;

  for( le = playlist; le; le = le->next )	g_free( le->data );
  g_list_free( playlist );
  playlist = NULL;

  mpwListClear( plwin_playlist );
}

void playlist_AddFile( char *path, char *filename )
{
  PlaylistEntry *entry;

  entry = g_malloc( sizeof( PlaylistEntry ));
  entry->text = g_strdup( filename );
  entry->file = g_strdup_printf( "%s/%s", path, filename );

  playlist = g_list_append( playlist, entry );

  mpwListAddEntry( plwin_playlist, entry->text );
}

char *playlist_nth( int num )
{
  GList		*le;

  le = g_list_nth( playlist, num );
  if( le == NULL )	return NULL;

  mpwListSetActive( plwin_playlist, num );

  playIdx = num;
  return ((PlaylistEntry *)le->data)->file;
}

char *playlist_Next( void )
{
  GList		*le;
  int		newIdx = playIdx + 1;

  le = g_list_nth( playlist, newIdx );
  if( le == NULL )	return NULL;

  mpwListSetActive( plwin_playlist, newIdx );

  playIdx = newIdx;
  return ((PlaylistEntry *)le->data)->file;
}

char *playlist_Prev( void )
{
  GList		*le;
  int		newIdx = playIdx;

  if( newIdx )	newIdx--;
  else	return NULL;

  le = g_list_nth( playlist, newIdx );
  if( le == NULL )	return NULL;

  mpwListSetActive( plwin_playlist, newIdx );

  playIdx = newIdx;
  return ((PlaylistEntry *)le->data)->file;
}

char *playlist_Current( void )
{
  GList		*le;

  le = g_list_nth( playlist, playIdx );
  if( le == NULL )	return NULL;

  return ((PlaylistEntry *)le->data)->file;
}

int playlist_Load( char *filename )
{
  FILE *stream;
  char *ptr, *ptr2, buffer[256];

  playlist_Clear();

  if(( stream = fopen( filename, "r" )) == NULL )	return 0;

  while( 1 )
  {
	fgets( buffer, 255, stream );
	if( feof( stream ))	break;

	ptr = strchr( buffer, 10 );
	if( ptr )	*ptr = '\0';

	ptr = g_basename( buffer );
	ptr2 = g_dirname( buffer );
	playlist_AddFile( ptr2, ptr );
	g_free( ptr2 );
  }

  fclose( stream );
  return 1;
}

int playlist_Save( char *filename )
{
  FILE *stream;
  GList *le;

  if(( stream = fopen( filename, "w" )) == NULL )	return 0;

  for( le = playlist; le; le = le->next )
	fprintf( stream, "%s\n", ((PlaylistEntry *)le->data)->file );

  fclose( stream );
  return 1;
}

void playlist_Init( void )
{
  char *ptr;
  GtkAccelGroup *plwin_accel;

// Create popup menus
  plwin_accel = gtk_accel_group_new();

  plwin_filemisc = gtk_item_factory_new( GTK_TYPE_MENU, "<Main>", plwin_accel );
  gtk_item_factory_create_items( plwin_filemisc, FILEMISC_ITEMS, plwin_filemisc_items, NULL );

  plwin_miscsort = gtk_item_factory_new( GTK_TYPE_MENU, "<Main>", plwin_accel );
  gtk_item_factory_create_items( plwin_miscsort, MISCSORT_ITEMS, plwin_miscsort_items, NULL );

  plwin_miscinfo = gtk_item_factory_new( GTK_TYPE_MENU, "<Main>", plwin_accel );
  gtk_item_factory_create_items( plwin_miscinfo, MISCINFO_ITEMS, plwin_miscinfo_items, NULL );

  plwin_miscopts = gtk_item_factory_new( GTK_TYPE_MENU, "<Main>", plwin_accel );
  gtk_item_factory_create_items( plwin_miscopts, MISCOPTS_ITEMS, plwin_miscopts_items, NULL );

// Load theme
  mplayer_LoadTheme( cfg_ThemeDir, cfg_SkinName, &plwin_theme );
  if( cfg_ShowPlaylist )	playlist_show();

// Load default playlist
  ptr = xmm_gethome_filename( DEFAULT_PLAYLIST );
  playlist_Load( ptr );

// Initialize widgets
//  mplayer_WidgetVisible( plwin_time_h, 0 );
//  mplayer_WidgetVisible( plwin_time_m, 0 );
//  mplayer_WidgetVisible( plwin_time_s, 0 );
}


void playlist_Exit( void )
{
  char *ptr;

  ptr = xmm_gethome_filename( DEFAULT_PLAYLIST );
  playlist_Save( ptr );
}
